<?php
/**
 * Admin Interface
 * Handles all admin pages and settings
 */

if (!defined('ABSPATH')) {
    exit;
}

class AIAB_Admin {
    
    private static $instance = null;
    
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        add_action('admin_menu', array($this, 'add_menu'));
        add_action('admin_init', array($this, 'register_settings'));
        add_action('admin_init', array($this, 'handle_sphere_actions'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_assets'));
        add_action('admin_notices', array($this, 'show_api_failure_notice'));
        add_action('wp_ajax_aiab_manual_run', array($this, 'ajax_manual_run'));
        add_action('wp_ajax_aiab_clear_lock', array($this, 'ajax_clear_lock'));
        add_action('wp_ajax_aiab_kill_zombies', array($this, 'ajax_kill_zombies'));
        add_action('wp_ajax_aiab_save_persona', array($this, 'ajax_save_persona'));
        add_action('wp_ajax_aiab_delete_persona', array($this, 'ajax_delete_persona'));
        add_action('wp_ajax_aiab_get_logs', array($this, 'ajax_get_logs'));
        add_action('wp_ajax_aiab_export_logs', array($this, 'ajax_export_logs'));
        add_action('wp_ajax_aiab_clear_all_logs', array($this, 'ajax_clear_all_logs'));
        add_action('wp_ajax_aiab_clear_old_logs', array($this, 'ajax_clear_old_logs'));
        add_action('wp_ajax_aiab_get_log_file', array($this, 'ajax_get_log_file'));
        add_action('wp_ajax_aiab_clear_cache', array($this, 'ajax_clear_cache'));
        add_action('wp_ajax_aiab_export_personas', array($this, 'ajax_export_personas'));
        add_action('wp_ajax_aiab_import_personas', array($this, 'ajax_import_personas'));
        add_action('wp_ajax_aiab_get_sphere_articles', array($this, 'ajax_get_sphere_articles'));
        add_action('wp_ajax_aiab_test_serpapi', array($this, 'ajax_test_serpapi'));
        add_action('wp_ajax_aiab_test_google_cse', array($this, 'ajax_test_google_cse'));
        add_action('wp_ajax_aiab_test_walter', array($this, 'ajax_test_walter'));
        add_action('wp_ajax_aiab_delete_sphere', array($this, 'ajax_delete_sphere'));
        add_action('wp_ajax_aiab_force_complete_sphere', array($this, 'ajax_force_complete_sphere'));
        add_action('wp_ajax_aiab_enable_cron', array($this, 'ajax_enable_cron'));
        add_action('wp_ajax_aiab_disable_cron', array($this, 'ajax_disable_cron'));
        add_action('wp_ajax_aiab_reschedule_cron', array($this, 'ajax_reschedule_cron'));
        add_action('wp_ajax_aiab_reset_failed_articles', array($this, 'ajax_reset_failed_articles'));
        add_action('wp_ajax_aiab_save_auto_continue', array($this, 'ajax_save_auto_continue'));
    }
    
    /**
     * Show admin notice when API authentication fails
     */
    public function show_api_failure_notice() {
        // Only show on plugin pages
        $screen = get_current_screen();
        if (!$screen || strpos($screen->id, 'aiab') === false) {
            return;
        }
        
        global $wpdb;
        $articles_table = AIAB_Database::get_table('articles');
        
        // Check for auth_failed articles
        $auth_failed_count = $wpdb->get_var(
            "SELECT COUNT(*) FROM $articles_table WHERE status = 'auth_failed'"
        );
        
        if ($auth_failed_count > 0) {
            $provider = get_option('aiab_ai_provider', 'anthropic');
            $provider_name = ucfirst($provider);
            ?>
            <div class="notice notice-error">
                <p><strong>🔴 API Authentication Failed!</strong></p>
                <p><?php echo $auth_failed_count; ?> article(s) failed due to invalid <?php echo esc_html($provider_name); ?> API key.</p>
                <p>
                    <a href="<?php echo admin_url('admin.php?page=aiab-settings'); ?>" class="button button-primary">Fix API Key</a>
                    <button type="button" class="button" onclick="aiabResetFailedArticles()">Reset Failed Articles</button>
                </p>
            </div>
            <?php
        }
        
        // Check for budget_failed articles
        $budget_failed_count = $wpdb->get_var(
            "SELECT COUNT(*) FROM $articles_table WHERE status = 'budget_failed'"
        );
        
        if ($budget_failed_count > 0) {
            $provider = get_option('aiab_ai_provider', 'anthropic');
            $provider_name = ucfirst($provider);
            ?>
            <div class="notice notice-error" style="border-left-color: #ff9800;">
                <p><strong>💰 API Budget/Quota Exceeded!</strong></p>
                <p><?php echo $budget_failed_count; ?> article(s) failed due to budget/quota limits on <?php echo esc_html($provider_name); ?>.</p>
                <p style="font-size: 13px; color: #666;">
                    You need to add credits or upgrade your plan on your <?php echo esc_html($provider_name); ?> account.
                    Once resolved, click the reset button below.
                </p>
                <p>
                    <a href="<?php echo $this->get_provider_billing_url($provider); ?>" target="_blank" class="button button-primary">
                        Check <?php echo esc_html($provider_name); ?> Billing
                    </a>
                    <button type="button" class="button" onclick="aiabResetFailedArticles()">Reset Failed Articles</button>
                </p>
            </div>
            <?php
        }
        
        // Check for max_retries articles
        $max_retries_count = $wpdb->get_var(
            "SELECT COUNT(*) FROM $articles_table WHERE status = 'max_retries'"
        );
        
        if ($max_retries_count > 0) {
            ?>
            <div class="notice notice-warning">
                <p><strong>⚠️ Articles Failed After Max Retries</strong></p>
                <p><?php echo $max_retries_count; ?> article(s) failed after 3 retry attempts. Check the Activity Log for details.</p>
                <p>
                    <a href="<?php echo admin_url('admin.php?page=aiab-logs'); ?>" class="button">View Logs</a>
                    <button type="button" class="button" onclick="aiabResetFailedArticles()">Reset Failed Articles</button>
                </p>
            </div>
            <?php
        }
        
        // Add the JavaScript once
        if ($auth_failed_count > 0 || $budget_failed_count > 0 || $max_retries_count > 0) {
            ?>
            <script>
            function aiabResetFailedArticles() {
                if (!confirm('Reset all failed articles to retry them? Make sure you have fixed any API issues first!')) {
                    return;
                }
                jQuery.post(ajaxurl, {
                    action: 'aiab_reset_failed_articles',
                    _wpnonce: '<?php echo wp_create_nonce('aiab_reset_failed'); ?>'
                }, function(response) {
                    if (response.success) {
                        alert(response.data.message);
                        location.reload();
                    } else {
                        alert('Error: ' + response.data.message);
                    }
                });
            }
            </script>
            <?php
        }
    }
    
    /**
     * Get billing URL for different providers
     */
    private function get_provider_billing_url($provider) {
        $urls = array(
            'anthropic' => 'https://console.anthropic.com/settings/billing',
            'openai' => 'https://platform.openai.com/account/billing',
            'perplexity' => 'https://www.perplexity.ai/settings/api',
            'walter' => '#' // Walter is self-hosted, no billing!
        );
        return isset($urls[$provider]) ? $urls[$provider] : '#';
    }
    
    /**
     * AJAX handler to reset failed articles
     */
    public function ajax_reset_failed_articles() {
        check_ajax_referer('aiab_reset_failed', '_wpnonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Permission denied'));
        }
        
        global $wpdb;
        $articles_table = AIAB_Database::get_table('articles');
        
        // Reset all failed articles to planned
        $reset_count = $wpdb->query(
            "UPDATE $articles_table 
             SET status = 'planned', retry_count = 0, last_error = NULL 
             WHERE status IN ('auth_failed', 'max_retries', 'budget_failed', 'failed')"
        );
        
        AIAB_Logger::info("Reset $reset_count failed articles to planned status");
        
        wp_send_json_success(array(
            'message' => "Reset $reset_count article(s) to planned status. They will be retried on the next cron run.",
            'count' => $reset_count
        ));
    }
    
    /**
     * AJAX: Save auto-continue preference
     */
    public function ajax_save_auto_continue() {
        check_ajax_referer('aiab_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        
        $enabled = isset($_POST['enabled']) ? intval($_POST['enabled']) : 0;
        update_option('aiab_auto_continue_enabled', $enabled);
        
        wp_send_json_success(array('saved' => true, 'enabled' => $enabled));
    }
    
    /**
     * Handle sphere delete/force-complete via GET requests (runs before headers sent)
     */
    public function handle_sphere_actions() {
        // Only run on our spheres page
        if (!isset($_GET['page']) || $_GET['page'] !== 'aiab-spheres') {
            return;
        }
        
        // Handle delete sphere
        if (isset($_GET['delete_sphere']) && isset($_GET['_wpnonce'])) {
            $sphere_id = intval($_GET['delete_sphere']);
            
            if (!wp_verify_nonce($_GET['_wpnonce'], 'delete_sphere_' . $sphere_id)) {
                wp_die('Security check failed');
            }
            
            if (!current_user_can('manage_options')) {
                wp_die('Permission denied');
            }
            
            global $wpdb;
            $spheres_table = $wpdb->prefix . 'aiab_thought_spheres';
            $articles_table = $wpdb->prefix . 'aiab_articles';
            
            // Debug: Check if sphere exists first
            $exists = $wpdb->get_var($wpdb->prepare(
                "SELECT id FROM $spheres_table WHERE id = %d",
                $sphere_id
            ));
            
            if (!$exists) {
                wp_redirect(admin_url('admin.php?page=aiab-spheres&error=notfound'));
                exit;
            }
            
            // Delete articles first
            $articles_deleted = $wpdb->query($wpdb->prepare(
                "DELETE FROM $articles_table WHERE sphere_id = %d",
                $sphere_id
            ));
            
            // Delete sphere
            $sphere_deleted = $wpdb->query($wpdb->prepare(
                "DELETE FROM $spheres_table WHERE id = %d",
                $sphere_id
            ));
            
            if ($sphere_deleted === false) {
                // Database error
                AIAB_Logger::error("Failed to delete sphere $sphere_id: " . $wpdb->last_error);
                wp_redirect(admin_url('admin.php?page=aiab-spheres&error=db&msg=' . urlencode($wpdb->last_error)));
                exit;
            }
            
            if ($sphere_deleted === 0) {
                // No rows affected
                wp_redirect(admin_url('admin.php?page=aiab-spheres&error=norows'));
                exit;
            }
            
            AIAB_Logger::warning("🗑️ Sphere deleted (ID: $sphere_id, Articles: $articles_deleted)");
            
            wp_redirect(admin_url('admin.php?page=aiab-spheres&deleted=1'));
            exit;
        }
        
        // Handle force complete
        if (isset($_GET['force_complete']) && isset($_GET['_wpnonce'])) {
            $sphere_id = intval($_GET['force_complete']);
            
            if (!wp_verify_nonce($_GET['_wpnonce'], 'force_complete_' . $sphere_id)) {
                wp_die('Security check failed');
            }
            
            if (!current_user_can('manage_options')) {
                wp_die('Permission denied');
            }
            
            global $wpdb;
            $spheres_table = $wpdb->prefix . 'aiab_thought_spheres';
            
            $updated = $wpdb->query($wpdb->prepare(
                "UPDATE $spheres_table SET status = 'completed', phase = 'complete' WHERE id = %d",
                $sphere_id
            ));
            
            if ($updated === false) {
                AIAB_Logger::error("Failed to force complete sphere $sphere_id: " . $wpdb->last_error);
                wp_redirect(admin_url('admin.php?page=aiab-spheres&error=db&msg=' . urlencode($wpdb->last_error)));
                exit;
            }
            
            AIAB_Logger::warning("⚡ Sphere force completed (ID: $sphere_id)");
            
            wp_redirect(admin_url('admin.php?page=aiab-spheres&completed=1'));
            exit;
        }
        
        // Handle force to linking phase
        if (isset($_GET['force_linking']) && isset($_GET['_wpnonce'])) {
            $sphere_id = intval($_GET['force_linking']);
            
            if (!wp_verify_nonce($_GET['_wpnonce'], 'force_linking_' . $sphere_id)) {
                wp_die('Security check failed');
            }
            
            if (!current_user_can('manage_options')) {
                wp_die('Permission denied');
            }
            
            global $wpdb;
            $spheres_table = $wpdb->prefix . 'aiab_thought_spheres';
            
            $updated = $wpdb->query($wpdb->prepare(
                "UPDATE $spheres_table SET phase = 'linking', status = 'writing' WHERE id = %d",
                $sphere_id
            ));
            
            if ($updated === false) {
                AIAB_Logger::error("Failed to force sphere $sphere_id to linking: " . $wpdb->last_error);
                wp_redirect(admin_url('admin.php?page=aiab-spheres&error=db&msg=' . urlencode($wpdb->last_error)));
                exit;
            }
            
            AIAB_Logger::warning("🔗 Sphere forced to linking phase (ID: $sphere_id)");
            
            wp_redirect(admin_url('admin.php?page=aiab-spheres&linking=1'));
            exit;
        }
    }
    
    /**
     * Add admin menu
     */
    public function add_menu() {
        add_menu_page(
            'Eternal Auto Blogger',
            'Eternal Blogger',
            'manage_options',
            'ai-autoblogger',
            array($this, 'page_dashboard'),
            'dashicons-edit-page',
            30
        );
        
        add_submenu_page(
            'ai-autoblogger',
            'Dashboard',
            'Dashboard',
            'manage_options',
            'ai-autoblogger',
            array($this, 'page_dashboard')
        );
        
        add_submenu_page(
            'ai-autoblogger',
            'Personas',
            'Personas',
            'manage_options',
            'aiab-personas',
            array($this, 'page_personas')
        );
        
        add_submenu_page(
            'ai-autoblogger',
            'Thought Spheres',
            'Thought Spheres',
            'manage_options',
            'aiab-spheres',
            array($this, 'page_spheres')
        );
        
        add_submenu_page(
            'ai-autoblogger',
            'Settings',
            'Settings',
            'manage_options',
            'aiab-settings',
            array($this, 'page_settings')
        );
        
        add_submenu_page(
            'ai-autoblogger',
            'Logs',
            'Logs',
            'manage_options',
            'aiab-logs',
            array($this, 'page_logs')
        );
    }
    
    /**
     * Check if license is valid - FREE VERSION: Always licensed
     */
    private function is_licensed() {
        return true; // Free version - no license required
    }
    
    /**
     * Register settings
     */
    public function register_settings() {
        // API Settings - Free version: OpenAI only
        register_setting('aiab_settings', 'aiab_openai_api_key');
        register_setting('aiab_settings', 'aiab_search_provider');
        
        // AI Settings
        register_setting('aiab_settings', 'aiab_ai_provider');
        register_setting('aiab_settings', 'aiab_ai_model');
        register_setting('aiab_settings', 'aiab_image_provider');
        register_setting('aiab_settings', 'aiab_image_style');
        
        // Content Settings
        register_setting('aiab_settings', 'aiab_articles_per_sphere');
        register_setting('aiab_settings', 'aiab_pillar_word_count');
        register_setting('aiab_settings', 'aiab_cluster_word_count');
        register_setting('aiab_settings', 'aiab_max_tokens');
        register_setting('aiab_settings', 'aiab_budget_mode');
        register_setting('aiab_settings', 'aiab_skip_research_ai');
        register_setting('aiab_settings', 'aiab_external_link_sources');
        
        // Publishing Settings
        register_setting('aiab_settings', 'aiab_auto_publish');
        register_setting('aiab_settings', 'aiab_post_status');
        register_setting('aiab_settings', 'aiab_default_category');
        register_setting('aiab_settings', 'aiab_author_id');
        
        // Schedule Settings
        register_setting('aiab_settings', 'aiab_cron_schedule');
        register_setting('aiab_settings', 'aiab_timezone');
        register_setting('aiab_settings', 'aiab_max_execution_time');
        
        // Debug Settings
        register_setting('aiab_debug_settings', 'aiab_debug_mode');
        register_setting('aiab_debug_settings', 'aiab_error_notifications');
        register_setting('aiab_debug_settings', 'aiab_error_notification_email');
        register_setting('aiab_debug_settings', 'aiab_log_retention_days');
        register_setting('aiab_debug_settings', 'aiab_delete_data_on_uninstall');
    }
    
    /**
     * Enqueue admin assets
     */
    public function enqueue_assets($hook) {
        if (strpos($hook, 'ai-autoblogger') === false && strpos($hook, 'aiab') === false) {
            return;
        }
        
        wp_enqueue_style('aiab-admin', AIAB_PLUGIN_URL . 'assets/css/admin.css', array(), AIAB_VERSION);
        wp_enqueue_script('aiab-admin', AIAB_PLUGIN_URL . 'assets/js/admin.js', array('jquery'), AIAB_VERSION, true);
        
        wp_localize_script('aiab-admin', 'aiab', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('aiab_nonce')
        ));
    }
    
    /**
     * Dashboard page
     */
    public function page_dashboard() {
        $personas = AIAB_Persona::get_all(true);
        $stats = $this->get_dashboard_stats();
        $next_run = wp_next_scheduled('aiab_run_autoblogger');
        
        include AIAB_PLUGIN_DIR . 'admin/views/dashboard.php';
    }
    
    /**
     * Personas page
     */
    public function page_personas() {
        $personas = AIAB_Persona::get_all();
        $editing = isset($_GET['edit']) ? AIAB_Persona::get(intval($_GET['edit'])) : null;
        
        include AIAB_PLUGIN_DIR . 'admin/views/personas.php';
    }
    
    /**
     * Thought Spheres page
     */
    public function page_spheres() {
        global $wpdb;
        $table = AIAB_Database::get_table('thought_spheres');
        $articles_table = AIAB_Database::get_table('articles');
        
        $spheres = $wpdb->get_results(
            "SELECT s.*, p.name as persona_name,
                (SELECT COUNT(*) FROM $articles_table a WHERE a.sphere_id = s.id AND a.status IN ('written', 'published', 'linked')) as completed_articles,
                (SELECT COUNT(*) FROM $articles_table a WHERE a.sphere_id = s.id) as actual_total_articles
             FROM $table s 
             LEFT JOIN " . AIAB_Database::get_table('personas') . " p ON s.persona_id = p.id 
             ORDER BY s.created_at DESC 
             LIMIT 50"
        );
        
        include AIAB_PLUGIN_DIR . 'admin/views/spheres.php';
    }
    
    /**
     * Settings page
     */
    public function page_settings() {
        $categories = get_categories(array('hide_empty' => false));
        $users = get_users(array('role__in' => array('administrator', 'editor', 'author')));
        
        include AIAB_PLUGIN_DIR . 'admin/views/settings.php';
    }
    
    /**
     * Logs page
     */
    public function page_logs() {
        $logs = AIAB_Logger::get_logs(200);
        $stats = AIAB_Logger::get_stats();
        
        include AIAB_PLUGIN_DIR . 'admin/views/logs.php';
    }
    
    /**
     * Get dashboard statistics
     */
    private function get_dashboard_stats() {
        global $wpdb;
        
        $personas_table = AIAB_Database::get_table('personas');
        $spheres_table = AIAB_Database::get_table('thought_spheres');
        $articles_table = AIAB_Database::get_table('articles');
        
        // Get in-progress sphere with calculated article counts
        $in_progress = $wpdb->get_row(
            "SELECT s.*, 
                (SELECT COUNT(*) FROM $articles_table a WHERE a.sphere_id = s.id AND a.status IN ('written', 'published', 'linked')) as completed_articles,
                (SELECT COUNT(*) FROM $articles_table a WHERE a.sphere_id = s.id) as actual_total_articles
             FROM $spheres_table s 
             WHERE s.status NOT IN ('completed', 'failed') 
             ORDER BY s.created_at DESC LIMIT 1"
        );
        
        // Check if updated_at column exists
        $has_updated_at_articles = $wpdb->get_var(
            "SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS 
             WHERE TABLE_SCHEMA = DATABASE() 
             AND TABLE_NAME = '$articles_table' 
             AND COLUMN_NAME = 'updated_at'"
        );
        
        $has_updated_at_spheres = $wpdb->get_var(
            "SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS 
             WHERE TABLE_SCHEMA = DATABASE() 
             AND TABLE_NAME = '$spheres_table' 
             AND COLUMN_NAME = 'updated_at'"
        );
        
        // Check for STUCK articles (writing status with 0 words for 10+ minutes)
        if ($has_updated_at_articles) {
            $stuck_articles = $wpdb->get_var(
                "SELECT COUNT(*) FROM $articles_table 
                 WHERE status = 'writing' 
                 AND word_count < 100 
                 AND updated_at < DATE_SUB(NOW(), INTERVAL 10 MINUTE)"
            );
        } else {
            // Fallback: use created_at
            $stuck_articles = $wpdb->get_var(
                "SELECT COUNT(*) FROM $articles_table 
                 WHERE status = 'writing' 
                 AND word_count < 100 
                 AND created_at < DATE_SUB(NOW(), INTERVAL 10 MINUTE)"
            );
        }
        
        // Check for STUCK spheres (in progress but no activity in 30+ minutes)
        if ($has_updated_at_spheres) {
            $stuck_spheres = $wpdb->get_var(
                "SELECT COUNT(*) FROM $spheres_table 
                 WHERE status NOT IN ('completed', 'failed', 'paused') 
                 AND updated_at < DATE_SUB(NOW(), INTERVAL 30 MINUTE)"
            );
        } else {
            // Fallback: use created_at
            $stuck_spheres = $wpdb->get_var(
                "SELECT COUNT(*) FROM $spheres_table 
                 WHERE status NOT IN ('completed', 'failed', 'paused') 
                 AND created_at < DATE_SUB(NOW(), INTERVAL 30 MINUTE)"
            );
        }
        
        // Check cron health
        $cron_enabled = wp_next_scheduled('aiab_run_autoblogger') !== false;
        
        return array(
            'total_personas' => $wpdb->get_var("SELECT COUNT(*) FROM $personas_table"),
            'active_personas' => $wpdb->get_var("SELECT COUNT(*) FROM $personas_table WHERE is_active = 1"),
            'total_spheres' => $wpdb->get_var("SELECT COUNT(*) FROM $spheres_table"),
            'completed_spheres' => $wpdb->get_var("SELECT COUNT(*) FROM $spheres_table WHERE status = 'completed'"),
            'total_articles' => $wpdb->get_var("SELECT COUNT(*) FROM $articles_table"),
            'published_articles' => $wpdb->get_var("SELECT COUNT(*) FROM $articles_table WHERE status IN ('published', 'linked')"),
            'articles_this_week' => $wpdb->get_var("SELECT COUNT(*) FROM $articles_table WHERE published_at >= DATE_SUB(NOW(), INTERVAL 7 DAY)"),
            'in_progress' => $in_progress,
            'stuck_articles' => intval($stuck_articles),
            'stuck_spheres' => intval($stuck_spheres),
            'cron_enabled' => $cron_enabled,
            'needs_attention' => (intval($stuck_articles) > 0 || intval($stuck_spheres) > 0 || !$cron_enabled)
        );
    }
    
    /**
     * AJAX: Manual run
    /**
     * AJAX: Manual run trigger
     */
    public function ajax_manual_run() {
        check_ajax_referer('aiab_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        
        try {
            $orchestrator = new AIAB_Orchestrator();
            
            if (isset($_POST['persona_id']) && !empty($_POST['persona_id'])) {
                $orchestrator->run_for_persona(intval($_POST['persona_id']));
            } else {
                $orchestrator->run();
            }
            
            wp_send_json_success(array(
                'message' => 'Cycle executed successfully',
                'status' => $orchestrator->get_status()
            ));
            
        } catch (Exception $e) {
            wp_send_json_error($e->getMessage());
        }
    }
    
    /**
     * AJAX: Clear process lock
     */
    public function ajax_clear_lock() {
        check_ajax_referer('aiab_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        
        delete_option('aiab_orchestrator_lock');
        AIAB_Logger::info('🔓 Process lock cleared manually by admin');
        
        wp_send_json_success(array('message' => 'Lock cleared! You can now run a new cycle.'));
    }
    
    /**
     * AJAX: Kill all zombie cycles - NUCLEAR OPTION
     * This completely stops all processes and resets stuck articles
     */
    public function ajax_kill_zombies() {
        check_ajax_referer('aiab_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        
        global $wpdb;
        $spheres_table = AIAB_Database::get_table('thought_spheres');
        $articles_table = AIAB_Database::get_table('articles');
        
        // 1. Get count of in-progress spheres
        $zombie_spheres = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$spheres_table} WHERE status NOT IN (%s, %s, %s)",
            'completed', 'failed', 'paused'
        ));
        
        // 2. Mark all in-progress spheres as paused
        $wpdb->query($wpdb->prepare(
            "UPDATE {$spheres_table} SET status = %s WHERE status NOT IN (%s, %s, %s)",
            'paused', 'completed', 'failed', 'paused'
        ));
        
        // 3. Reset stuck articles (in 'writing' status) back to 'planned'
        $stuck_articles = $wpdb->get_var(
            "SELECT COUNT(*) FROM {$articles_table} WHERE status = 'writing'"
        );
        
        if ($stuck_articles > 0) {
            $wpdb->query(
                "UPDATE {$articles_table} 
                 SET status = 'planned', content = '', word_count = 0 
                 WHERE status = 'writing'"
            );
        }
        
        // 4. Reset 'failed' and 'budget_failed' articles that can be retried
        $retry_reset = $wpdb->query(
            "UPDATE {$articles_table} 
             SET status = 'planned', retry_count = 0, last_error = NULL 
             WHERE status IN ('failed', 'budget_failed') AND (retry_count IS NULL OR retry_count < 3)"
        );
        
        // 5. Clear the orchestrator lock
        delete_option('aiab_orchestrator_lock');
        
        // 6. Clear any transients that might be causing issues
        delete_transient('aiab_research_in_progress');
        delete_transient('aiab_api_call_in_progress');
        
        // Log the action
        AIAB_Logger::warning("💀 KILL SWITCH ACTIVATED", array(
            'paused_spheres' => $zombie_spheres,
            'reset_stuck_articles' => $stuck_articles,
            'reset_failed_articles' => $retry_reset
        ));
        
        $message = "💀 Kill Switch Complete!\n";
        $message .= "• Paused {$zombie_spheres} in-progress sphere(s)\n";
        $message .= "• Reset {$stuck_articles} stuck article(s)\n";
        $message .= "• Reset {$retry_reset} failed article(s) for retry\n";
        $message .= "• Cleared all locks and transients";
        
        wp_send_json_success(array(
            'message' => $message,
            'killed_spheres' => intval($zombie_spheres),
            'reset_stuck' => intval($stuck_articles),
            'reset_failed' => intval($retry_reset)
        ));
    }
    
    /**
     * AJAX: Save persona
     */
    public function ajax_save_persona() {
        check_ajax_referer('aiab_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        
        $id = isset($_POST['id']) ? intval($_POST['id']) : 0;
        
        if ($id) {
            $persona = AIAB_Persona::get($id);
            if (!$persona) {
                wp_send_json_error('Persona not found with ID: ' . $id);
            }
        } else {
            $persona = new AIAB_Persona();
        }
        
        // Update persona data
        $persona->set_name(sanitize_text_field($_POST['name']));
        $persona->set_personal_story(isset($_POST['personal_story']) ? sanitize_textarea_field($_POST['personal_story']) : '');
        $persona->set_interests($this->parse_textarea_lines(isset($_POST['interests']) ? $_POST['interests'] : ''));
        $persona->set_passions($this->parse_textarea_lines(isset($_POST['passions']) ? $_POST['passions'] : ''));
        $persona->set_expertise_areas($this->parse_textarea_lines(isset($_POST['expertise_areas']) ? $_POST['expertise_areas'] : ''));
        
        // Localities - handle various input formats
        $localities = array();
        if (isset($_POST['localities']) && is_array($_POST['localities'])) {
            foreach ($_POST['localities'] as $loc) {
                $loc = sanitize_text_field($loc);
                if (!empty($loc)) {
                    $localities[] = $loc;
                }
            }
        }
        $persona->set_localities($localities);
        
        $persona->set_voice_style(isset($_POST['voice_style']) ? sanitize_text_field($_POST['voice_style']) : 'professional');
        $persona->set_tone(isset($_POST['tone']) ? sanitize_text_field($_POST['tone']) : 'informative');
        $persona->set_target_audience(isset($_POST['target_audience']) ? sanitize_text_field($_POST['target_audience']) : '');
        $persona->set_knowledge_sources($this->parse_textarea_lines(isset($_POST['knowledge_sources']) ? $_POST['knowledge_sources'] : ''));
        $persona->set_writing_guidelines(isset($_POST['writing_guidelines']) ? sanitize_textarea_field($_POST['writing_guidelines']) : '');
        $persona->set_avoid_topics($this->parse_textarea_lines(isset($_POST['avoid_topics']) ? $_POST['avoid_topics'] : ''));
        $persona->set_language(isset($_POST['language']) ? sanitize_text_field($_POST['language']) : 'English');
        $persona->set_active(isset($_POST['is_active']) && $_POST['is_active'] === '1');
        
        // WordPress Author
        $wp_author_id = null;
        if (isset($_POST['wp_author_id']) && !empty($_POST['wp_author_id']) && $_POST['wp_author_id'] !== '') {
            $wp_author_id = intval($_POST['wp_author_id']);
            if ($wp_author_id <= 0) {
                $wp_author_id = null;
            }
        }
        $persona->set_wp_author_id($wp_author_id);
        
        // Categories
        $categories = array();
        if (isset($_POST['categories']) && is_array($_POST['categories'])) {
            foreach ($_POST['categories'] as $cat) {
                $cat_id = intval($cat);
                if ($cat_id > 0) {
                    $categories[] = $cat_id;
                }
            }
        }
        $persona->set_categories($categories);
        
        // Attempt save
        global $wpdb;
        $wpdb->suppress_errors(true);
        $result = $persona->save();
        $db_error = $wpdb->last_error;
        $wpdb->suppress_errors(false);
        
        if ($result === false) {
            wp_send_json_error('Database error: ' . ($db_error ? $db_error : 'Save returned false'));
        }
        
        wp_send_json_success(array(
            'message' => 'Persona saved successfully',
            'id' => $persona->get_id()
        ));
    }
    
    /**
     * AJAX: Delete persona
     */
    public function ajax_delete_persona() {
        check_ajax_referer('aiab_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        
        $id = isset($_POST['id']) ? intval($_POST['id']) : 0;
        $persona = AIAB_Persona::get($id);
        
        if (!$persona) {
            wp_send_json_error('Persona not found');
        }
        
        $persona->delete();
        
        wp_send_json_success('Persona deleted');
    }
    
    /**
     * AJAX: Get logs
     */
    public function ajax_get_logs() {
        check_ajax_referer('aiab_nonce', 'nonce');
        
        $level = isset($_POST['level']) ? sanitize_text_field($_POST['level']) : null;
        $logs = AIAB_Logger::get_logs(100, $level);
        
        wp_send_json_success($logs);
    }
    
    /**
     * AJAX: Export logs
     */
    public function ajax_export_logs() {
        check_admin_referer('aiab_nonce', 'nonce');
        
        $format = isset($_GET['format']) ? sanitize_text_field($_GET['format']) : 'json';
        $level = isset($_GET['level']) ? sanitize_text_field($_GET['level']) : null;
        
        $content = AIAB_Logger::export($format, $level);
        
        $filename = 'aiab-logs-' . date('Y-m-d-His') . '.' . ($format === 'json' ? 'json' : 'csv');
        $content_type = $format === 'json' ? 'application/json' : 'text/csv';
        
        header('Content-Type: ' . $content_type);
        header('Content-Disposition: attachment; filename="' . $filename . '"');
        header('Content-Length: ' . strlen($content));
        
        echo $content;
        exit;
    }
    
    /**
     * AJAX: Clear all logs
     */
    public function ajax_clear_all_logs() {
        check_ajax_referer('aiab_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        
        AIAB_Logger::clear_all_logs();
        
        wp_send_json_success('All logs cleared');
    }
    
    /**
     * AJAX: Clear old logs
     */
    public function ajax_clear_old_logs() {
        check_ajax_referer('aiab_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        
        $days = get_option('aiab_log_retention_days', 30);
        $deleted = AIAB_Logger::clear_old_logs($days);
        
        wp_send_json_success(array('deleted' => $deleted));
    }
    
    /**
     * AJAX: Get log file contents
     */
    public function ajax_get_log_file() {
        check_ajax_referer('aiab_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        
        $filename = isset($_POST['filename']) ? sanitize_file_name($_POST['filename']) : '';
        
        if (empty($filename)) {
            wp_send_json_error('No filename provided');
        }
        
        $contents = AIAB_Logger::get_log_file_contents($filename);
        
        wp_send_json_success($contents);
    }
    
    /**
     * AJAX: Clear research cache
     */
    public function ajax_clear_cache() {
        check_ajax_referer('aiab_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        
        AIAB_Database::clean_expired_cache();
        
        // Also clear all cache
        global $wpdb;
        $table = AIAB_Database::get_table('research_cache');
        $wpdb->query("TRUNCATE TABLE $table");
        
        wp_send_json_success('Cache cleared');
    }
    
    /**
     * AJAX: Export all personas
     */
    public function ajax_export_personas() {
        check_ajax_referer('aiab_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        
        global $wpdb;
        $table = AIAB_Database::get_table('personas');
        
        // Check if exporting single persona
        $persona_id = isset($_POST['persona_id']) ? intval($_POST['persona_id']) : 0;
        
        if ($persona_id) {
            // Export single persona
            $persona = $wpdb->get_row($wpdb->prepare("SELECT * FROM $table WHERE id = %d", $persona_id), ARRAY_A);
            
            if (!$persona) {
                wp_send_json_error('Persona not found');
            }
            
            // Remove site-specific data
            unset($persona['id']);
            unset($persona['wp_author_id']); // Will need to be reassigned on import
            unset($persona['total_articles']);
            unset($persona['total_spheres']);
            unset($persona['created_at']);
            unset($persona['updated_at']);
            
            $export_data = array(
                'eab_version' => AIAB_VERSION,
                'export_type' => 'single_persona',
                'exported_at' => current_time('mysql'),
                'source_site' => get_site_url(),
                'persona' => $persona
            );
            
            wp_send_json_success($export_data);
        } else {
            // Export all personas
            $personas = $wpdb->get_results("SELECT * FROM $table", ARRAY_A);
            
            $export_data = array(
                'version' => AIAB_VERSION,
                'exported_at' => current_time('mysql'),
                'site_url' => get_site_url(),
                'personas' => $personas
            );
            
            wp_send_json_success($export_data);
        }
    }
    
    /**
     * AJAX: Import personas from file upload
     */
    public function ajax_import_personas() {
        check_ajax_referer('aiab_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        
        // Handle file upload
        if (isset($_FILES['persona_file']) && $_FILES['persona_file']['error'] === UPLOAD_ERR_OK) {
            $file = $_FILES['persona_file'];
            $filename = $file['name'];
            $ext = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
            
            // Validate extension
            if ($ext !== 'eab' && $ext !== 'zip') {
                wp_send_json_error('Invalid file type. Please upload a .eab file.');
            }
            
            // Read file content
            $content = file_get_contents($file['tmp_name']);
            
            // If it's a zip, extract the JSON
            if ($ext === 'zip' || $ext === 'eab') {
                // Check if content starts with PK (zip signature)
                if (substr($content, 0, 2) === 'PK') {
                    $zip = new ZipArchive();
                    if ($zip->open($file['tmp_name']) === TRUE) {
                        $json_content = $zip->getFromName('persona.json');
                        $zip->close();
                        
                        if ($json_content === false) {
                            wp_send_json_error('Invalid .eab file - no persona.json found inside');
                        }
                        $content = $json_content;
                    } else {
                        wp_send_json_error('Failed to open .eab file');
                    }
                }
            }
            
            // Parse JSON
            $import_data = json_decode($content, true);
            
            if (!$import_data) {
                wp_send_json_error('Invalid file content - could not parse JSON');
            }
            
            global $wpdb;
            $table = AIAB_Database::get_table('personas');
            $now = current_time('mysql');
            
            // Handle single persona export format
            if (isset($import_data['export_type']) && $import_data['export_type'] === 'single_persona') {
                $persona_data = $import_data['persona'];
                
                // Check if persona with same name exists
                $exists = $wpdb->get_var($wpdb->prepare(
                    "SELECT id FROM $table WHERE name = %s",
                    $persona_data['name']
                ));
                
                if ($exists) {
                    // Append number to name
                    $base_name = $persona_data['name'];
                    $counter = 2;
                    while ($wpdb->get_var($wpdb->prepare("SELECT id FROM $table WHERE name = %s", $persona_data['name']))) {
                        $persona_data['name'] = $base_name . ' (' . $counter . ')';
                        $counter++;
                    }
                }
                
                // Generate new slug
                $persona_data['slug'] = sanitize_title($persona_data['name']);
                $base_slug = $persona_data['slug'];
                $counter = 1;
                while ($wpdb->get_var($wpdb->prepare("SELECT id FROM $table WHERE slug = %s", $persona_data['slug']))) {
                    $persona_data['slug'] = $base_slug . '-' . $counter;
                    $counter++;
                }
                
                // Reset stats and timestamps
                $persona_data['total_articles'] = 0;
                $persona_data['total_spheres'] = 0;
                $persona_data['wp_author_id'] = null;
                $persona_data['created_at'] = $now;
                $persona_data['updated_at'] = $now;
                
                $result = $wpdb->insert($table, $persona_data);
                
                if ($result === false) {
                    wp_send_json_error('Failed to import persona: ' . $wpdb->last_error);
                }
                
                wp_send_json_success(array(
                    'imported' => 1,
                    'skipped' => 0,
                    'message' => 'Successfully imported persona: ' . $persona_data['name'],
                    'persona_id' => $wpdb->insert_id
                ));
            }
            
            // Handle legacy multi-persona format
            if (isset($import_data['personas'])) {
                $imported = 0;
                $skipped = 0;
                
                foreach ($import_data['personas'] as $persona_data) {
                    // Check if persona with same name exists
                    $exists = $wpdb->get_var($wpdb->prepare(
                        "SELECT id FROM $table WHERE name = %s",
                        $persona_data['name']
                    ));
                    
                    if ($exists) {
                        $skipped++;
                        continue;
                    }
                    
                    // Remove id to create new
                    unset($persona_data['id']);
                    $persona_data['created_at'] = $now;
                    $persona_data['updated_at'] = $now;
                    
                    // Reset stats
                    $persona_data['total_articles'] = 0;
                    $persona_data['total_spheres'] = 0;
                    $persona_data['wp_author_id'] = null;
                    
                    $wpdb->insert($table, $persona_data);
                    $imported++;
                }
                
                wp_send_json_success(array(
                    'imported' => $imported,
                    'skipped' => $skipped,
                    'message' => "Imported {$imported} personas" . ($skipped > 0 ? ", skipped {$skipped} (already exist)" : "")
                ));
            }
            
            wp_send_json_error('Invalid import data format');
        }
        
        // Handle JSON data (legacy method)
        $import_data = isset($_POST['data']) ? $_POST['data'] : null;
        
        if (!$import_data || !isset($import_data['personas'])) {
            wp_send_json_error('No file uploaded or invalid import data');
        }
        
        global $wpdb;
        $table = AIAB_Database::get_table('personas');
        $now = current_time('mysql');
        
        $imported = 0;
        $skipped = 0;
        
        foreach ($import_data['personas'] as $persona_data) {
            // Check if persona with same name exists
            $exists = $wpdb->get_var($wpdb->prepare(
                "SELECT id FROM $table WHERE name = %s",
                $persona_data['name']
            ));
            
            if ($exists) {
                $skipped++;
                continue;
            }
            
            // Remove id to create new
            unset($persona_data['id']);
            $persona_data['created_at'] = $now;
            $persona_data['updated_at'] = $now;
            
            // Reset stats
            $persona_data['total_articles'] = 0;
            $persona_data['total_spheres'] = 0;
            
            $wpdb->insert($table, $persona_data);
            $imported++;
        }
        
        wp_send_json_success(array(
            'imported' => $imported,
            'skipped' => $skipped,
            'message' => "Imported {$imported} personas" . ($skipped > 0 ? ", skipped {$skipped} (already exist)" : "")
        ));
    }
    
    /**
     * AJAX: Get sphere articles for diagram
     */
    public function ajax_get_sphere_articles() {
        check_ajax_referer('aiab_nonce', 'nonce');
        
        $sphere_id = isset($_POST['sphere_id']) ? intval($_POST['sphere_id']) : 0;
        
        if (!$sphere_id) {
            wp_send_json_error('Invalid sphere ID');
        }
        
        global $wpdb;
        $table = AIAB_Database::get_table('articles');
        
        $articles = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM $table WHERE sphere_id = %d ORDER BY is_pillar DESC, id ASC",
            $sphere_id
        ));
        
        wp_send_json_success($articles);
    }
    
    /**
     * Parse textarea into array of lines
     */
    private function parse_textarea_lines($text) {
        if (empty($text) || !is_string($text)) {
            return array();
        }
        $lines = explode("\n", $text);
        $lines = array_map('trim', $lines);
        $lines = array_filter($lines);
        return array_values($lines);
    }
    
    /**
     * AJAX: Test SerpAPI connection
     */
    public function ajax_test_serpapi() {
        check_ajax_referer('aiab_test_serpapi', 'nonce');
        
        $api_key = isset($_POST['api_key']) ? sanitize_text_field($_POST['api_key']) : '';
        
        if (empty($api_key)) {
            wp_send_json_error('Please enter an API key');
        }
        
        // Test query - use a common search term
        $test_query = 'how to improve indoor air quality';
        
        $url = 'https://serpapi.com/search.json?' . http_build_query(array(
            'q' => $test_query,
            'api_key' => $api_key,
            'engine' => 'google',
            'gl' => 'us',
            'hl' => 'en',
            'num' => 10
        ));
        
        $response = wp_remote_get($url, array('timeout' => 20));
        
        if (is_wp_error($response)) {
            wp_send_json_error('Connection failed: ' . $response->get_error_message());
        }
        
        $status = wp_remote_retrieve_response_code($response);
        $body = json_decode(wp_remote_retrieve_body($response), true);
        
        if ($status !== 200) {
            $error_msg = isset($body['error']) ? $body['error'] : 'Unknown error (HTTP ' . $status . ')';
            wp_send_json_error('SerpAPI error: ' . $error_msg);
        }
        
        // Extract People Also Ask questions
        $questions = array();
        if (!empty($body['related_questions'])) {
            foreach (array_slice($body['related_questions'], 0, 5) as $q) {
                $questions[] = $q['question'];
            }
        }
        
        // Get account info if available
        $credits = 'Check your account';
        if (isset($body['search_metadata']['total_time_taken'])) {
            $credits = 'API responded in ' . round($body['search_metadata']['total_time_taken'], 2) . 's';
        }
        
        if (empty($questions)) {
            // Check if we at least got organic results
            if (!empty($body['organic_results'])) {
                wp_send_json_success(array(
                    'questions' => array('No PAA questions found for test query, but organic results work!'),
                    'credits' => $credits
                ));
            } else {
                wp_send_json_error('API connected but no results returned. Check your API key permissions.');
            }
        }
        
        wp_send_json_success(array(
            'questions' => $questions,
            'credits' => $credits
        ));
    }
    
    /**
     * AJAX: Test Google Custom Search API
     */
    public function ajax_test_google_cse() {
        check_ajax_referer('aiab_test_google_cse', 'nonce');
        
        $api_key = isset($_POST['api_key']) ? sanitize_text_field($_POST['api_key']) : '';
        $cx = isset($_POST['cx']) ? sanitize_text_field($_POST['cx']) : '';
        
        if (empty($api_key) || empty($cx)) {
            wp_send_json_error('Please enter both API Key and Search Engine ID');
        }
        
        // Test query
        $test_query = 'pest control tips';
        
        $url = 'https://www.googleapis.com/customsearch/v1?' . http_build_query(array(
            'key' => $api_key,
            'cx' => $cx,
            'q' => $test_query,
            'num' => 5
        ));
        
        $response = wp_remote_get($url, array('timeout' => 15));
        
        if (is_wp_error($response)) {
            wp_send_json_error('Connection failed: ' . $response->get_error_message());
        }
        
        $status = wp_remote_retrieve_response_code($response);
        $body = json_decode(wp_remote_retrieve_body($response), true);
        
        if ($status !== 200) {
            $error_msg = isset($body['error']['message']) ? $body['error']['message'] : 'Unknown error (HTTP ' . $status . ')';
            
            // Provide helpful error messages
            if (strpos($error_msg, 'API key not valid') !== false) {
                $error_msg = 'Invalid API Key. Make sure you copied it correctly from Google Cloud Console.';
            } elseif (strpos($error_msg, 'cx') !== false || strpos($error_msg, 'search engine') !== false) {
                $error_msg = 'Invalid Search Engine ID (CX). Make sure you created a Programmable Search Engine and copied the ID.';
            } elseif (strpos($error_msg, 'Custom Search API has not been used') !== false) {
                $error_msg = 'Custom Search API is not enabled. Go to Google Cloud Console → APIs → Enable "Custom Search API"';
            } elseif (strpos($error_msg, 'quota') !== false) {
                $error_msg = 'Daily quota exceeded. Free tier allows 100 queries/day. Quota resets at midnight Pacific Time.';
            }
            
            wp_send_json_error('Google CSE error: ' . $error_msg);
        }
        
        // Extract results
        $results = array();
        $related = array();
        
        if (!empty($body['items'])) {
            foreach (array_slice($body['items'], 0, 5) as $item) {
                $title = preg_replace('/\s*[\|\-–—]\s*[^|–—]+$/', '', $item['title']); // Clean title
                $results[] = $title;
            }
        }
        
        // Get related searches if available
        if (!empty($body['queries']['request'])) {
            foreach ($body['queries']['request'] as $req) {
                if (!empty($req['searchTerms']) && $req['searchTerms'] !== $test_query) {
                    $related[] = $req['searchTerms'];
                }
            }
        }
        
        if (empty($results)) {
            wp_send_json_error('API connected but no results returned. Your Search Engine might not be set to "Search the entire web".');
        }
        
        wp_send_json_success(array(
            'results' => $results,
            'related' => $related
        ));
    }
    
    /**
     * AJAX: Test Walter (Ollama) connection
     */
    public function ajax_test_walter() {
        check_ajax_referer('aiab_test_walter', 'nonce');
        
        $server_url = isset($_POST['server_url']) ? esc_url_raw($_POST['server_url']) : '';
        $model = isset($_POST['model']) ? sanitize_text_field($_POST['model']) : '';
        
        if (empty($server_url)) {
            wp_send_json_error('Please enter your Custom Server URL');
        }
        
        $server_url = rtrim($server_url, '/');
        
        // First, try to list available models
        $list_url = $server_url . '/api/tags';
        $response = wp_remote_get($list_url, array('timeout' => 15));
        
        if (is_wp_error($response)) {
            $error = $response->get_error_message();
            if (strpos($error, 'cURL error 7') !== false || strpos($error, 'Connection refused') !== false) {
                wp_send_json_error('Cannot connect to server. Is Ollama running? Check the URL and ensure port is accessible.');
            }
            wp_send_json_error('Connection failed: ' . $error);
        }
        
        $status = wp_remote_retrieve_response_code($response);
        $body = json_decode(wp_remote_retrieve_body($response), true);
        
        if ($status !== 200) {
            wp_send_json_error('Server returned HTTP ' . $status . '. Is this an Ollama server?');
        }
        
        // Extract model names
        $models = array();
        if (!empty($body['models'])) {
            foreach ($body['models'] as $m) {
                if (!empty($m['name'])) {
                    $models[] = $m['name'];
                }
            }
        }
        
        // Optionally test a simple generation
        $test_response = null;
        if (!empty($model) && in_array($model, $models)) {
            $generate_url = $server_url . '/v1/chat/completions';
            $gen_response = wp_remote_post($generate_url, array(
                'timeout' => 60,
                'headers' => array('Content-Type' => 'application/json'),
                'body' => json_encode(array(
                    'model' => $model,
                    'messages' => array(
                        array('role' => 'user', 'content' => 'Reply with exactly: Server is ready to write articles!')
                    ),
                    'max_tokens' => 50,
                    'stream' => false
                ))
            ));
            
            if (!is_wp_error($gen_response)) {
                $gen_body = json_decode(wp_remote_retrieve_body($gen_response), true);
                if (isset($gen_body['choices'][0]['message']['content'])) {
                    $test_response = substr($gen_body['choices'][0]['message']['content'], 0, 100);
                } elseif (isset($gen_body['message']['content'])) {
                    $test_response = substr($gen_body['message']['content'], 0, 100);
                }
            }
        }
        
        if (empty($models)) {
            wp_send_json_error('Connected to server but no models found. Run: ollama pull qwen2.5:32b-instruct-q4_K_M');
        }
        
        wp_send_json_success(array(
            'models' => $models,
            'test_response' => $test_response
        ));
    }
    
    /**
     * AJAX: Delete sphere and its articles
     */
    public function ajax_delete_sphere() {
        check_ajax_referer('aiab_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        
        $sphere_id = intval($_POST['sphere_id']);
        
        if (!$sphere_id) {
            wp_send_json_error('Invalid sphere ID');
        }
        
        global $wpdb;
        
        // Get sphere info for logging
        $spheres_table = AIAB_Database::get_table('thought_spheres');
        $sphere = $wpdb->get_row($wpdb->prepare(
            "SELECT pillar_keyword FROM $spheres_table WHERE id = %d",
            $sphere_id
        ));
        
        if (!$sphere) {
            wp_send_json_error('Sphere not found');
        }
        
        // Delete all articles in this sphere (from plugin DB only, not WordPress posts)
        $articles_table = AIAB_Database::get_table('articles');
        $deleted_articles = $wpdb->delete($articles_table, array('sphere_id' => $sphere_id), array('%d'));
        
        // Delete the sphere itself
        $deleted_sphere = $wpdb->delete($spheres_table, array('id' => $sphere_id), array('%d'));
        
        AIAB_Logger::warning("🗑️ Sphere deleted: {$sphere->pillar_keyword} (ID: $sphere_id, Articles deleted: $deleted_articles)");
        
        wp_send_json_success(array(
            'message' => 'Sphere deleted',
            'articles_deleted' => $deleted_articles
        ));
    }
    
    /**
     * AJAX: Force complete a stuck sphere
     */
    public function ajax_force_complete_sphere() {
        check_ajax_referer('aiab_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        
        $sphere_id = intval($_POST['sphere_id']);
        
        if (!$sphere_id) {
            wp_send_json_error('Invalid sphere ID');
        }
        
        global $wpdb;
        $spheres_table = AIAB_Database::get_table('thought_spheres');
        
        // Get sphere info
        $sphere = $wpdb->get_row($wpdb->prepare(
            "SELECT pillar_keyword, status FROM $spheres_table WHERE id = %d",
            $sphere_id
        ));
        
        if (!$sphere) {
            wp_send_json_error('Sphere not found');
        }
        
        // Update to completed status
        $updated = $wpdb->update(
            $spheres_table,
            array(
                'status' => 'completed',
                'phase' => 'complete'
            ),
            array('id' => $sphere_id),
            array('%s', '%s'),
            array('%d')
        );
        
        AIAB_Logger::warning("⚡ Sphere force completed: {$sphere->pillar_keyword} (ID: $sphere_id, Previous status: {$sphere->status})");
        
        wp_send_json_success(array(
            'message' => 'Sphere marked as complete'
        ));
    }
    
    /**
     * AJAX: Enable cron scheduler
     */
    public function ajax_enable_cron() {
        check_ajax_referer('aiab_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        
        // Get schedule from option (or use daily as default)
        $schedule = get_option('aiab_cron_schedule', 'daily');
        
        // Clear any existing schedule first
        $timestamp = wp_next_scheduled('aiab_run_autoblogger');
        if ($timestamp) {
            wp_unschedule_event($timestamp, 'aiab_run_autoblogger');
        }
        
        // Schedule new cron
        $scheduled = wp_schedule_event(time() + 60, $schedule, 'aiab_run_autoblogger');
        
        if ($scheduled === false) {
            wp_send_json_error('Failed to schedule cron event');
        }
        
        $next_run = wp_next_scheduled('aiab_run_autoblogger');
        
        AIAB_Logger::info("✅ Cron enabled with schedule: $schedule");
        
        wp_send_json_success(array(
            'message' => 'Cron scheduler enabled',
            'schedule' => $schedule,
            'next_run' => $next_run,
            'next_run_human' => human_time_diff($next_run) . ' from now'
        ));
    }
    
    /**
     * AJAX: Disable cron scheduler
     */
    public function ajax_disable_cron() {
        check_ajax_referer('aiab_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        
        // Clear all scheduled events for our hook
        $timestamp = wp_next_scheduled('aiab_run_autoblogger');
        if ($timestamp) {
            wp_unschedule_event($timestamp, 'aiab_run_autoblogger');
        }
        
        // Also clear any other scheduled instances
        wp_clear_scheduled_hook('aiab_run_autoblogger');
        
        AIAB_Logger::info("⏸️ Cron disabled");
        
        wp_send_json_success(array(
            'message' => 'Cron scheduler disabled'
        ));
    }
    
    /**
     * AJAX: Reschedule cron with new interval
     */
    public function ajax_reschedule_cron() {
        check_ajax_referer('aiab_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        
        $schedule = sanitize_text_field($_POST['schedule'] ?? 'daily');
        
        // Validate schedule
        $valid_schedules = array('hourly', 'every_six_hours', 'twice_daily', 'daily', 'weekly');
        if (!in_array($schedule, $valid_schedules)) {
            wp_send_json_error('Invalid schedule');
        }
        
        // Save the new schedule option
        update_option('aiab_cron_schedule', $schedule);
        
        // Clear existing schedule
        $timestamp = wp_next_scheduled('aiab_run_autoblogger');
        if ($timestamp) {
            wp_unschedule_event($timestamp, 'aiab_run_autoblogger');
        }
        wp_clear_scheduled_hook('aiab_run_autoblogger');
        
        // Schedule with new interval
        $scheduled = wp_schedule_event(time() + 60, $schedule, 'aiab_run_autoblogger');
        
        if ($scheduled === false) {
            wp_send_json_error('Failed to reschedule cron event');
        }
        
        $next_run = wp_next_scheduled('aiab_run_autoblogger');
        
        $schedule_labels = array(
            'hourly' => 'Every Hour',
            'every_six_hours' => 'Every 6 Hours', 
            'twice_daily' => 'Twice Daily',
            'daily' => 'Once Daily',
            'weekly' => 'Once Weekly'
        );
        
        AIAB_Logger::info("🔄 Cron rescheduled to: " . $schedule_labels[$schedule]);
        
        wp_send_json_success(array(
            'message' => 'Cron schedule updated to: ' . $schedule_labels[$schedule],
            'schedule' => $schedule,
            'schedule_label' => $schedule_labels[$schedule],
            'next_run' => $next_run,
            'next_run_human' => human_time_diff($next_run) . ' from now'
        ));
    }
}
