<?php
/**
 * Logs & Debug View
 */

if (!defined('ABSPATH')) {
    exit;
}

$health = AIAB_Logger::health_check();
$log_files = AIAB_Logger::get_log_files();
$debug_mode = get_option('aiab_debug_mode', false);
$aiab_timezone = new DateTimeZone(get_option('aiab_timezone', 'UTC'));
?>

<div class="wrap aiab-wrap">
    <h1>🔍 Debug & Logs</h1>
    
    <!-- Tab Navigation -->
    <nav class="aiab-tabs">
        <a href="#logs" class="aiab-tab active" data-tab="logs">📋 Activity Logs</a>
        <a href="#files" class="aiab-tab" data-tab="files">📁 Log Files</a>
        <a href="#health" class="aiab-tab" data-tab="health">💚 System Health</a>
        <a href="#settings" class="aiab-tab" data-tab="settings">⚙️ Debug Settings</a>
    </nav>
    
    <!-- LOGS TAB -->
    <div id="tab-logs" class="aiab-tab-content active">
        <!-- Log Stats Dashboard -->
        <div class="aiab-log-stats-grid">
            <div class="aiab-log-stat-card total">
                <span class="stat-number"><?php echo esc_html($stats['total']); ?></span>
                <span class="stat-label">Total Logs</span>
            </div>
            <div class="aiab-log-stat-card today">
                <span class="stat-number"><?php echo esc_html($stats['today']); ?></span>
                <span class="stat-label">Today</span>
            </div>
            <div class="aiab-log-stat-card debug">
                <span class="stat-number"><?php echo esc_html($stats['debug'] ?? 0); ?></span>
                <span class="stat-label">Debug</span>
            </div>
            <div class="aiab-log-stat-card info">
                <span class="stat-number"><?php echo esc_html($stats['info'] ?? 0); ?></span>
                <span class="stat-label">Info</span>
            </div>
            <div class="aiab-log-stat-card warning">
                <span class="stat-number"><?php echo esc_html($stats['warnings']); ?></span>
                <span class="stat-label">Warnings</span>
            </div>
            <div class="aiab-log-stat-card error">
                <span class="stat-number"><?php echo esc_html($stats['errors']); ?></span>
                <span class="stat-label">Errors</span>
            </div>
            <div class="aiab-log-stat-card critical">
                <span class="stat-number"><?php echo esc_html($stats['critical'] ?? 0); ?></span>
                <span class="stat-label">Critical</span>
            </div>
        </div>
        
        <?php if (!empty($stats['top_errors'])): ?>
        <div class="aiab-card aiab-top-errors">
            <h3>🔥 Top Errors (Last 7 Days)</h3>
            <ul>
                <?php foreach (array_slice($stats['top_errors'], 0, 5) as $error): ?>
                    <li>
                        <span class="error-count"><?php echo esc_html($error->count); ?>x</span>
                        <span class="error-message"><?php echo esc_html(substr($error->message, 0, 100)); ?></span>
                    </li>
                <?php endforeach; ?>
            </ul>
        </div>
        <?php endif; ?>
        
        <!-- Log Filters -->
        <div class="aiab-card aiab-log-filters">
            <div class="filter-row">
                <div class="filter-group">
                    <label for="aiab-log-level">Level</label>
                    <select id="aiab-log-level">
                        <option value="all">All Levels</option>
                        <option value="debug">🔧 Debug</option>
                        <option value="info">ℹ️ Info</option>
                        <option value="warning">⚠️ Warning</option>
                        <option value="error">❌ Error</option>
                        <option value="critical">🚨 Critical</option>
                    </select>
                </div>
                
                <div class="filter-group">
                    <label for="aiab-log-search">Search</label>
                    <input type="text" id="aiab-log-search" placeholder="Search logs...">
                </div>
                
                <div class="filter-group">
                    <label for="aiab-log-limit">Show</label>
                    <select id="aiab-log-limit">
                        <option value="50">50 entries</option>
                        <option value="100" selected>100 entries</option>
                        <option value="200">200 entries</option>
                        <option value="500">500 entries</option>
                    </select>
                </div>
                
                <div class="filter-actions">
                    <button type="button" id="aiab-refresh-logs" class="button">🔄 Refresh</button>
                    <button type="button" id="aiab-export-json" class="button">📥 Export JSON</button>
                    <button type="button" id="aiab-export-csv" class="button">📥 Export CSV</button>
                    <button type="button" id="aiab-clear-logs" class="button button-link-delete">🗑️ Clear All</button>
                </div>
            </div>
        </div>
        
        <!-- Logs Table -->
        <div class="aiab-card">
            <div class="aiab-logs-wrapper">
                <table class="aiab-table aiab-logs-table" id="aiab-logs-table">
                    <thead>
                        <tr>
                            <th width="160">Timestamp</th>
                            <th width="90">Level</th>
                            <th>Message</th>
                            <th width="80">Persona</th>
                            <th width="80">Sphere</th>
                            <th width="90">Details</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (empty($logs)): ?>
                            <tr>
                                <td colspan="6" class="aiab-empty">No logs yet. Run the autoblogger to generate activity.</td>
                            </tr>
                        <?php else: ?>
                            <?php foreach ($logs as $log): ?>
                                <tr class="log-level-<?php echo esc_attr($log->level); ?>">
                                    <td class="aiab-log-time">
                                        <?php 
                                        $log_dt = new DateTime($log->created_at, new DateTimeZone('UTC'));
                                        $log_dt->setTimezone($aiab_timezone);
                                        ?>
                                        <span class="log-date"><?php echo esc_html($log_dt->format('M j')); ?></span>
                                        <span class="log-time"><?php echo esc_html($log_dt->format('g:i:s A')); ?></span>
                                    </td>
                                    <td>
                                        <span class="aiab-log-level level-<?php echo esc_attr($log->level); ?>">
                                            <?php 
                                            $icons = array(
                                                'debug' => '🔧',
                                                'info' => 'ℹ️', 
                                                'warning' => '⚠️', 
                                                'error' => '❌',
                                                'critical' => '🚨'
                                            );
                                            echo isset($icons[$log->level]) ? $icons[$log->level] : '❓';
                                            echo ' ' . esc_html(ucfirst($log->level));
                                            ?>
                                        </span>
                                    </td>
                                    <td class="aiab-log-message">
                                        <?php echo esc_html($log->message); ?>
                                    </td>
                                    <td>
                                        <?php echo $log->persona_id ? '#' . esc_html($log->persona_id) : '-'; ?>
                                    </td>
                                    <td>
                                        <?php echo $log->sphere_id ? '#' . esc_html($log->sphere_id) : '-'; ?>
                                    </td>
                                    <td>
                                        <?php if ($log->context): ?>
                                            <button type="button" class="button button-small aiab-view-context" 
                                                    data-context="<?php echo esc_attr($log->context); ?>">
                                                View
                                            </button>
                                        <?php else: ?>
                                            -
                                        <?php endif; ?>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>
    
    <!-- LOG FILES TAB -->
    <div id="tab-files" class="aiab-tab-content">
        <div class="aiab-card">
            <h2>📁 Debug Log Files</h2>
            <p>Log files are stored in <code>wp-content/aiab-logs/</code> when debug mode is enabled.</p>
            
            <?php if (empty($log_files)): ?>
                <div class="aiab-empty-state">
                    <span class="aiab-empty-icon">📁</span>
                    <p>No log files found. Enable debug mode to generate file-based logs.</p>
                </div>
            <?php else: ?>
                <table class="aiab-table">
                    <thead>
                        <tr>
                            <th>File</th>
                            <th>Size</th>
                            <th>Modified</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($log_files as $file): ?>
                            <tr>
                                <td><code><?php echo esc_html($file['name']); ?></code></td>
                                <td><?php echo esc_html($file['size']); ?></td>
                                <td><?php echo esc_html($file['modified']); ?></td>
                                <td>
                                    <button type="button" class="button button-small aiab-view-file" 
                                            data-file="<?php echo esc_attr($file['name']); ?>">
                                        View
                                    </button>
                                    <button type="button" class="button button-small aiab-download-file"
                                            data-file="<?php echo esc_attr($file['name']); ?>">
                                        Download
                                    </button>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            <?php endif; ?>
        </div>
        
        <div class="aiab-card" id="aiab-file-viewer" style="display: none;">
            <div class="aiab-file-header">
                <h3>📄 File Contents: <span id="aiab-file-name"></span></h3>
                <div class="aiab-file-actions">
                    <button type="button" id="aiab-copy-log" class="button">
                        📋 Copy All
                    </button>
                    <button type="button" id="aiab-close-viewer" class="button">
                        ✕ Close
                    </button>
                </div>
            </div>
            <div class="aiab-terminal">
                <div class="aiab-terminal-header">
                    <span class="aiab-terminal-dot red"></span>
                    <span class="aiab-terminal-dot yellow"></span>
                    <span class="aiab-terminal-dot green"></span>
                    <span class="aiab-terminal-title">debug.log</span>
                </div>
                <pre id="aiab-file-contents" class="aiab-terminal-content"></pre>
            </div>
            <div id="aiab-copy-toast" class="aiab-copy-toast">✓ Copied to clipboard!</div>
        </div>
    </div>
    
    <!-- HEALTH CHECK TAB -->
    <div id="tab-health" class="aiab-tab-content">
        <div class="aiab-card">
            <h2>💚 System Health Check</h2>
            <p>Check if your server meets the requirements for Eternal Auto Blogger.</p>
            
            <table class="aiab-table aiab-health-table">
                <thead>
                    <tr>
                        <th>Check</th>
                        <th>Value</th>
                        <th>Status</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($health as $key => $check): ?>
                        <tr class="health-<?php echo esc_attr($check['status']); ?>">
                            <td><?php echo esc_html($check['label']); ?></td>
                            <td><code><?php echo esc_html($check['value']); ?></code></td>
                            <td>
                                <?php 
                                $status_icons = array(
                                    'good' => '✅',
                                    'warning' => '⚠️',
                                    'error' => '❌'
                                );
                                echo isset($status_icons[$check['status']]) 
                                    ? $status_icons[$check['status']] . ' ' . ucfirst($check['status'])
                                    : '❓';
                                ?>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        
        <div class="aiab-card">
            <h3>📊 Memory & Performance</h3>
            <div class="aiab-perf-grid">
                <div class="perf-item">
                    <span class="perf-label">Current Memory</span>
                    <span class="perf-value"><?php echo esc_html(size_format(memory_get_usage())); ?></span>
                </div>
                <div class="perf-item">
                    <span class="perf-label">Peak Memory</span>
                    <span class="perf-value"><?php echo esc_html(size_format(memory_get_peak_usage())); ?></span>
                </div>
                <div class="perf-item">
                    <span class="perf-label">Memory Limit</span>
                    <span class="perf-value"><?php echo esc_html(ini_get('memory_limit')); ?></span>
                </div>
                <div class="perf-item">
                    <span class="perf-label">PHP Version</span>
                    <span class="perf-value"><?php echo esc_html(PHP_VERSION); ?></span>
                </div>
            </div>
        </div>
    </div>
    
    <!-- DEBUG SETTINGS TAB -->
    <div id="tab-settings" class="aiab-tab-content">
        <form method="post" action="options.php">
            <?php settings_fields('aiab_debug_settings'); ?>
            
            <div class="aiab-card">
                <h2>⚙️ Debug Settings</h2>
                
                <table class="form-table">
                    <tr>
                        <th scope="row">Debug Mode</th>
                        <td>
                            <label class="aiab-toggle">
                                <input type="checkbox" name="aiab_debug_mode" value="1" 
                                       <?php checked(get_option('aiab_debug_mode', false)); ?>>
                                <span class="aiab-toggle-slider"></span>
                                <span class="aiab-toggle-label">Enable Debug Mode</span>
                            </label>
                            <p class="description">When enabled, detailed debug logs will be written to files and database.</p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Error Notifications</th>
                        <td>
                            <label class="aiab-toggle">
                                <input type="checkbox" name="aiab_error_notifications" value="1"
                                       <?php checked(get_option('aiab_error_notifications', false)); ?>>
                                <span class="aiab-toggle-slider"></span>
                                <span class="aiab-toggle-label">Email on Critical Errors</span>
                            </label>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Notification Email</th>
                        <td>
                            <input type="email" name="aiab_error_notification_email" class="regular-text"
                                   value="<?php echo esc_attr(get_option('aiab_error_notification_email', get_option('admin_email'))); ?>"
                                   placeholder="<?php echo esc_attr(get_option('admin_email')); ?>">
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Log Retention</th>
                        <td>
                            <select name="aiab_log_retention_days">
                                <?php $retention = get_option('aiab_log_retention_days', 30); ?>
                                <option value="7" <?php selected($retention, 7); ?>>7 days</option>
                                <option value="14" <?php selected($retention, 14); ?>>14 days</option>
                                <option value="30" <?php selected($retention, 30); ?>>30 days</option>
                                <option value="60" <?php selected($retention, 60); ?>>60 days</option>
                                <option value="90" <?php selected($retention, 90); ?>>90 days</option>
                            </select>
                            <p class="description">Automatically delete logs older than this.</p>
                        </td>
                    </tr>
                </table>
                
                <?php submit_button('Save Debug Settings'); ?>
            </div>
        </form>
        
        <div class="aiab-card">
            <h3>🧹 Maintenance</h3>
            <p>Clean up old logs and cached data.</p>
            
            <div class="aiab-maintenance-actions">
                <button type="button" id="aiab-clear-old-logs" class="button">
                    🗑️ Clear Logs Older Than <?php echo esc_html(get_option('aiab_log_retention_days', 30)); ?> Days
                </button>
                <button type="button" id="aiab-clear-cache" class="button">
                    🧹 Clear Research Cache
                </button>
                <button type="button" id="aiab-clear-all-logs" class="button button-link-delete">
                    ⚠️ Delete All Logs
                </button>
            </div>
        </div>
        
        <div class="aiab-card">
            <h3>💾 Backup & Restore Personas</h3>
            <p>Export your personas to a file for backup, or import from a previous export.</p>
            
            <div class="aiab-backup-actions">
                <button type="button" id="aiab-export-personas" class="button button-primary">
                    📥 Export All Personas
                </button>
                
                <div class="aiab-import-section">
                    <input type="file" id="aiab-import-file" accept=".json" style="display:none;">
                    <button type="button" id="aiab-import-personas" class="button">
                        📤 Import Personas
                    </button>
                </div>
            </div>
            
            <div id="aiab-backup-result" class="aiab-backup-result" style="display:none; margin-top:15px;"></div>
        </div>
        
        <div class="aiab-card aiab-warning-card">
            <h3>⚠️ Data Preservation</h3>
            <p><strong>Important:</strong> When updating the plugin, <strong>do NOT delete/uninstall</strong> it. Instead:</p>
            <ol>
                <li>Deactivate the plugin</li>
                <li>Upload the new version (overwrite files)</li>
                <li>Reactivate the plugin</li>
            </ol>
            <p>Your personas, thought spheres, and all data will be preserved.</p>
            
            <div class="aiab-data-option">
                <label class="aiab-toggle">
                    <input type="checkbox" name="aiab_delete_data_on_uninstall" value="1"
                           <?php checked(get_option('aiab_delete_data_on_uninstall', false)); ?>>
                    <span class="aiab-toggle-slider"></span>
                    <span class="aiab-toggle-label">Delete all data when plugin is uninstalled</span>
                </label>
                <p class="description" style="color:#c62828;">⚠️ Only enable this if you want to completely remove all Eternal Auto Blogger data when deleting the plugin.</p>
            </div>
        </div>
    </div>
</div>

<!-- Context Modal -->
<div id="aiab-context-modal" class="aiab-modal hidden">
    <div class="aiab-modal-content aiab-modal-large">
        <span class="aiab-modal-close">&times;</span>
        <h2>📄 Log Context Details</h2>
        <pre id="aiab-context-content" class="aiab-context-viewer"></pre>
    </div>
</div>

<script>
jQuery(document).ready(function($) {
    // Tab switching
    $('.aiab-tab').on('click', function(e) {
        e.preventDefault();
        var tab = $(this).data('tab');
        
        $('.aiab-tab').removeClass('active');
        $(this).addClass('active');
        
        $('.aiab-tab-content').removeClass('active');
        $('#tab-' + tab).addClass('active');
    });
    
    // View context
    $('.aiab-view-context').on('click', function() {
        var context = $(this).data('context');
        try {
            var parsed = JSON.parse(context);
            var formatted = JSON.stringify(parsed, null, 2);
            $('#aiab-context-content').text(formatted);
        } catch (e) {
            $('#aiab-context-content').text(context);
        }
        $('#aiab-context-modal').removeClass('hidden');
    });
    
    // Close modal
    $('.aiab-modal-close').on('click', function() {
        $(this).closest('.aiab-modal').addClass('hidden');
    });
    
    // Filter logs
    function filterLogs() {
        var level = $('#aiab-log-level').val();
        var search = $('#aiab-log-search').val().toLowerCase();
        
        $('.aiab-logs-table tbody tr').each(function() {
            var $row = $(this);
            var showLevel = (level === 'all' || $row.hasClass('log-level-' + level));
            var showSearch = (search === '' || $row.text().toLowerCase().indexOf(search) > -1);
            
            $row.toggle(showLevel && showSearch);
        });
    }
    
    $('#aiab-log-level, #aiab-log-limit').on('change', filterLogs);
    $('#aiab-log-search').on('keyup', filterLogs);
    
    // Refresh logs
    $('#aiab-refresh-logs').on('click', function() {
        location.reload();
    });
    
    // Export JSON
    $('#aiab-export-json').on('click', function() {
        window.location.href = ajaxurl + '?action=aiab_export_logs&format=json&nonce=' + aiab.nonce;
    });
    
    // Export CSV
    $('#aiab-export-csv').on('click', function() {
        window.location.href = ajaxurl + '?action=aiab_export_logs&format=csv&nonce=' + aiab.nonce;
    });
    
    // Clear all logs
    $('#aiab-clear-logs, #aiab-clear-all-logs').on('click', function() {
        if (!confirm('Are you sure you want to delete ALL logs? This cannot be undone.')) {
            return;
        }
        
        $.post(aiab.ajax_url, {
            action: 'aiab_clear_all_logs',
            nonce: aiab.nonce
        }, function(response) {
            if (response.success) {
                location.reload();
            } else {
                alert('Error: ' + response.data);
            }
        });
    });
    
    // Clear old logs
    $('#aiab-clear-old-logs').on('click', function() {
        $.post(aiab.ajax_url, {
            action: 'aiab_clear_old_logs',
            nonce: aiab.nonce
        }, function(response) {
            if (response.success) {
                alert('Cleared ' + response.data.deleted + ' old log entries.');
                location.reload();
            }
        });
    });
    
    // View log file
    $('.aiab-view-file').on('click', function() {
        var filename = $(this).data('file');
        
        $.post(aiab.ajax_url, {
            action: 'aiab_get_log_file',
            nonce: aiab.nonce,
            filename: filename
        }, function(response) {
            if (response.success) {
                $('#aiab-file-name').text(filename);
                $('.aiab-terminal-title').text(filename);
                $('#aiab-file-contents').text(response.data);
                $('#aiab-file-viewer').show();
                
                // Scroll to viewer
                $('html, body').animate({
                    scrollTop: $('#aiab-file-viewer').offset().top - 50
                }, 500);
            }
        });
    });
    
    // Copy log contents
    $('#aiab-copy-log').on('click', function() {
        var content = $('#aiab-file-contents').text();
        
        if (navigator.clipboard && navigator.clipboard.writeText) {
            navigator.clipboard.writeText(content).then(function() {
                showCopyToast();
            });
        } else {
            // Fallback for older browsers
            var $temp = $('<textarea>');
            $('body').append($temp);
            $temp.val(content).select();
            document.execCommand('copy');
            $temp.remove();
            showCopyToast();
        }
    });
    
    function showCopyToast() {
        var $toast = $('#aiab-copy-toast');
        $toast.addClass('show');
        setTimeout(function() {
            $toast.removeClass('show');
        }, 2000);
    }
    
    // Close file viewer
    $('#aiab-close-viewer').on('click', function() {
        $('#aiab-file-viewer').hide();
    });
    
    // Export personas
    $('#aiab-export-personas').on('click', function() {
        var $btn = $(this);
        $btn.prop('disabled', true).text('Exporting...');
        
        $.post(aiab.ajax_url, {
            action: 'aiab_export_personas',
            nonce: aiab.nonce
        }, function(response) {
            $btn.prop('disabled', false).text('📥 Export All Personas');
            
            if (response.success) {
                // Download as JSON file
                var blob = new Blob([JSON.stringify(response.data, null, 2)], {type: 'application/json'});
                var url = URL.createObjectURL(blob);
                var a = document.createElement('a');
                a.href = url;
                a.download = 'aiab-personas-backup-' + new Date().toISOString().split('T')[0] + '.json';
                a.click();
                URL.revokeObjectURL(url);
                
                $('#aiab-backup-result')
                    .removeClass('error').addClass('success')
                    .html('✅ Exported ' + response.data.personas.length + ' personas')
                    .show();
            } else {
                $('#aiab-backup-result')
                    .removeClass('success').addClass('error')
                    .html('❌ Export failed: ' + response.data)
                    .show();
            }
        });
    });
    
    // Import personas - trigger file input
    $('#aiab-import-personas').on('click', function() {
        $('#aiab-import-file').click();
    });
    
    // Handle file selection
    $('#aiab-import-file').on('change', function(e) {
        var file = e.target.files[0];
        if (!file) return;
        
        var reader = new FileReader();
        reader.onload = function(e) {
            try {
                var data = JSON.parse(e.target.result);
                
                if (!data.personas || !Array.isArray(data.personas)) {
                    alert('Invalid backup file format');
                    return;
                }
                
                if (!confirm('Import ' + data.personas.length + ' personas from backup?\n\nExisting personas with the same name will be skipped.')) {
                    return;
                }
                
                $.post(aiab.ajax_url, {
                    action: 'aiab_import_personas',
                    nonce: aiab.nonce,
                    data: data
                }, function(response) {
                    if (response.success) {
                        $('#aiab-backup-result')
                            .removeClass('error').addClass('success')
                            .html('✅ ' + response.data.message)
                            .show();
                        
                        // Reload page after 2 seconds
                        setTimeout(function() {
                            location.reload();
                        }, 2000);
                    } else {
                        $('#aiab-backup-result')
                            .removeClass('success').addClass('error')
                            .html('❌ Import failed: ' + response.data)
                            .show();
                    }
                });
            } catch (err) {
                alert('Error reading file: ' + err.message);
            }
        };
        reader.readAsText(file);
        
        // Reset input
        $(this).val('');
    });
});
</script>
