<?php
/**
 * Thought Spheres View
 */

if (!defined('ABSPATH')) {
    exit;
}
?>

<div class="wrap aiab-wrap">
    <h1>🌐 Thought Spheres</h1>
    
    <?php if (isset($_GET['deleted'])): ?>
        <div class="notice notice-success is-dismissible"><p>✅ Sphere deleted successfully!</p></div>
    <?php endif; ?>
    <?php if (isset($_GET['completed'])): ?>
        <div class="notice notice-success is-dismissible"><p>✅ Sphere marked as complete!</p></div>
    <?php endif; ?>
    <?php if (isset($_GET['linking'])): ?>
        <div class="notice notice-success is-dismissible"><p>🔗 Sphere advanced to linking phase! Run a cycle to publish and link articles.</p></div>
    <?php endif; ?>
    <?php if (isset($_GET['error'])): ?>
        <div class="notice notice-error is-dismissible">
            <p>❌ Error: 
            <?php 
            switch($_GET['error']) {
                case 'notfound': echo 'Sphere not found in database'; break;
                case 'norows': echo 'No rows were deleted (sphere may already be deleted)'; break;
                case 'db': echo 'Database error: ' . esc_html($_GET['msg'] ?? 'Unknown'); break;
                default: echo 'Unknown error';
            }
            ?>
            </p>
        </div>
    <?php endif; ?>
    
    <p>Each thought sphere is a cluster of 7 interconnected articles around a central pillar topic.</p>
    
    <?php if (empty($spheres)): ?>
        <div class="aiab-card">
            <div class="aiab-empty-state">
                <span class="aiab-empty-icon">🌐</span>
                <p>No thought spheres created yet. Run the autoblogger to start generating content!</p>
                <a href="<?php echo admin_url('admin.php?page=ai-autoblogger'); ?>" class="button button-primary">Go to Dashboard</a>
            </div>
        </div>
    <?php else: ?>
        <div class="aiab-spheres-grid">
            <?php foreach ($spheres as $sphere): ?>
                <div class="aiab-card aiab-sphere-card status-<?php echo esc_attr($sphere->status); ?>">
                    <div class="aiab-sphere-header">
                        <h3><?php echo esc_html($sphere->pillar_keyword); ?></h3>
                        <span class="aiab-sphere-status status-<?php echo esc_attr($sphere->status); ?>">
                            <?php 
                            $status_icons = array(
                                'planned' => '📋',
                                'researching' => '🔍',
                                'writing' => '✍️',
                                'publishing' => '📤',
                                'completed' => '✅',
                                'failed' => '❌'
                            );
                            echo isset($status_icons[$sphere->status]) ? $status_icons[$sphere->status] : '❓';
                            echo ' ' . ucfirst($sphere->status);
                            ?>
                        </span>
                    </div>
                    
                    <div class="aiab-sphere-meta">
                        <span class="aiab-sphere-persona">👤 <?php echo esc_html($sphere->persona_name ?: 'Unknown'); ?></span>
                        <span class="aiab-sphere-date">📅 <?php 
                            $display_date = !empty($sphere->created_at) && $sphere->created_at !== '0000-00-00 00:00:00' 
                                ? strtotime($sphere->created_at) 
                                : (!empty($sphere->started_at) && $sphere->started_at !== '0000-00-00 00:00:00' 
                                    ? strtotime($sphere->started_at) 
                                    : time());
                            echo esc_html(date('M j, Y', $display_date)); 
                        ?></span>
                    </div>
                    
                    <div class="aiab-sphere-progress">
                        <div class="aiab-progress-bar">
                            <div class="aiab-progress-fill" style="width: <?php echo ($sphere->completed_articles / $sphere->total_articles) * 100; ?>%"></div>
                        </div>
                        <span class="aiab-progress-text">
                            <?php echo esc_html($sphere->completed_articles); ?>/<?php echo esc_html($sphere->total_articles); ?> articles
                        </span>
                    </div>
                    
                    <?php if ($sphere->status !== 'completed' && $sphere->status !== 'failed'): ?>
                        <div class="aiab-sphere-phase">
                            <strong>Current Phase:</strong> <?php echo esc_html(ucwords(str_replace('_', ' ', $sphere->phase))); ?>
                        </div>
                    <?php endif; ?>
                    
                    <div class="aiab-sphere-actions">
                        <button type="button" class="button button-small aiab-view-sphere" 
                                data-id="<?php echo esc_attr($sphere->id); ?>">
                            View Articles
                        </button>
                        <button type="button" class="button button-small aiab-view-diagram" 
                                data-id="<?php echo esc_attr($sphere->id); ?>"
                                data-pillar="<?php echo esc_attr($sphere->pillar_keyword); ?>">
                            🔗 View Diagram
                        </button>
                        <?php if ($sphere->status === 'completed'): ?>
                            <a href="<?php echo admin_url('edit.php?s=&post_status=all&post_type=post&aiab_sphere=' . $sphere->id); ?>" 
                               class="button button-small">View in Posts</a>
                        <?php endif; ?>
                        <?php if ($sphere->status !== 'completed' && in_array($sphere->phase, array('writing_pillar', 'writing_clusters'))): ?>
                            <a href="<?php echo wp_nonce_url(admin_url('admin.php?page=aiab-spheres&force_linking=' . $sphere->id), 'force_linking_' . $sphere->id); ?>" 
                               class="button button-small"
                               style="color: #0073aa;"
                               onclick="return confirm('🔗 FORCE TO LINKING PHASE?\n\nThis will skip remaining writing and publish whatever articles are ready.');">
                                🔗 Force to Linking
                            </a>
                        <?php endif; ?>
                        <?php if ($sphere->status !== 'completed'): ?>
                            <a href="<?php echo wp_nonce_url(admin_url('admin.php?page=aiab-spheres&force_complete=' . $sphere->id), 'force_complete_' . $sphere->id); ?>" 
                               class="button button-small"
                               style="color: #b26900;"
                               onclick="return confirm('⚡ FORCE COMPLETE this sphere?\n\nThis will mark it as completed even if articles are unfinished.');">
                                ⚡ Force Complete
                            </a>
                        <?php endif; ?>
                        <a href="<?php echo wp_nonce_url(admin_url('admin.php?page=aiab-spheres&delete_sphere=' . $sphere->id), 'delete_sphere_' . $sphere->id); ?>" 
                           class="button button-small"
                           style="color: #dc3545;"
                           onclick="return confirm('🗑️ DELETE this sphere?\n\nThis will remove it from the database.\nPublished WordPress posts will NOT be deleted.');">
                            🗑️ Delete
                        </a>
                    </div>
                </div>
            <?php endforeach; ?>
        </div>
    <?php endif; ?>
</div>

<!-- Sphere Details Modal -->
<div id="aiab-sphere-modal" class="aiab-modal hidden">
    <div class="aiab-modal-content">
        <span class="aiab-modal-close">&times;</span>
        <h2>📚 Sphere Articles</h2>
        <div id="aiab-sphere-articles"></div>
    </div>
</div>

<!-- Cluster Diagram Modal -->
<div id="aiab-diagram-modal" class="aiab-modal hidden">
    <div class="aiab-modal-content aiab-modal-diagram">
        <span class="aiab-modal-close">&times;</span>
        <h2>🔗 Topic Cluster Diagram</h2>
        <p class="aiab-diagram-subtitle">Visual representation of the thought sphere structure and internal linking</p>
        
        <div class="aiab-cluster-container">
            <div id="aiab-cluster-diagram" class="aiab-cluster-diagram">
                <!-- Diagram will be populated by JavaScript -->
            </div>
        </div>
        
        <div class="aiab-diagram-legend">
            <div class="legend-item">
                <span class="legend-dot pillar"></span>
                <span>Pillar Article (Hub)</span>
            </div>
            <div class="legend-item">
                <span class="legend-dot cluster"></span>
                <span>Supporting Article</span>
            </div>
            <div class="legend-item">
                <span class="legend-line linked" style="height: 3px;"></span>
                <span>Hub Link (Active)</span>
            </div>
            <div class="legend-item">
                <span class="legend-arc linked"></span>
                <span>Circular Link (Active)</span>
            </div>
            <div class="legend-item">
                <span class="legend-line pending" style="height: 3px;"></span>
                <span>Pending</span>
            </div>
        </div>
        
        <div id="aiab-diagram-details" class="aiab-diagram-details"></div>
    </div>
</div>

<style>
/* Cluster Diagram Styles */
.aiab-modal-diagram {
    max-width: 900px;
}

.aiab-diagram-subtitle {
    color: #666;
    margin-top: -10px;
    margin-bottom: 20px;
}

.aiab-cluster-container {
    background: linear-gradient(135deg, #f8f9ff 0%, #f0f4ff 100%);
    border-radius: 16px;
    padding: 40px;
    margin-bottom: 20px;
}

.aiab-cluster-diagram {
    position: relative;
    width: 500px;
    height: 500px;
    margin: 0 auto;
}

/* Legend arc indicator */
.legend-arc {
    width: 30px;
    height: 15px;
    border: 2px solid #4caf50;
    border-bottom: none;
    border-radius: 15px 15px 0 0;
}

.legend-arc.pending {
    border-color: #ff9800;
    border-style: dashed;
}

/* Pillar (center) */
.cluster-node {
    position: absolute;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    text-align: center;
    font-size: 11px;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.3s ease;
    box-shadow: 0 4px 15px rgba(0,0,0,0.1);
    padding: 10px;
    box-sizing: border-box;
    line-height: 1.3;
}

.cluster-node:hover {
    transform: scale(1.1);
    z-index: 100;
}

.cluster-node.pillar {
    width: 140px;
    height: 140px;
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    font-size: 13px;
    left: 50%;
    top: 50%;
    transform: translate(-50%, -50%);
    z-index: 10;
}

.cluster-node.pillar:hover {
    transform: translate(-50%, -50%) scale(1.1);
}

.cluster-node.cluster {
    width: 110px;
    height: 110px;
    background: white;
    border: 3px solid #667eea;
    color: #333;
}

.cluster-node.linked {
    border-color: #4caf50;
    background: #f1f8e9;
}

.cluster-node.pending {
    border-color: #ff9800;
    border-style: dashed;
}

.cluster-node .node-label {
    max-width: 90%;
    overflow: hidden;
    text-overflow: ellipsis;
    display: -webkit-box;
    -webkit-line-clamp: 3;
    -webkit-box-orient: vertical;
}

/* Connection lines */
.cluster-line {
    position: absolute;
    background: #667eea;
    height: 3px;
    transform-origin: left center;
    z-index: 1;
}

.cluster-line.linked {
    background: #4caf50;
}

.cluster-line.pending {
    background: #ff9800;
    background: repeating-linear-gradient(
        90deg,
        #ff9800,
        #ff9800 5px,
        transparent 5px,
        transparent 10px
    );
}

/* Legend */
.aiab-diagram-legend {
    display: flex;
    justify-content: center;
    gap: 30px;
    flex-wrap: wrap;
    padding: 15px;
    background: #f8f9fa;
    border-radius: 8px;
    margin-bottom: 20px;
}

.legend-item {
    display: flex;
    align-items: center;
    gap: 8px;
    font-size: 13px;
}

.legend-dot {
    width: 16px;
    height: 16px;
    border-radius: 50%;
}

.legend-dot.pillar {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
}

.legend-dot.cluster {
    background: white;
    border: 2px solid #667eea;
}

.legend-line {
    width: 30px;
    height: 3px;
}

.legend-line.linked {
    background: #4caf50;
}

.legend-line.pending {
    background: repeating-linear-gradient(
        90deg,
        #ff9800,
        #ff9800 3px,
        transparent 3px,
        transparent 6px
    );
}

/* Details panel */
.aiab-diagram-details {
    background: white;
    border: 1px solid #e0e0e0;
    border-radius: 8px;
    padding: 15px;
    max-height: 200px;
    overflow-y: auto;
}

.aiab-diagram-details h4 {
    margin: 0 0 10px 0;
    color: #667eea;
}

.aiab-diagram-details ul {
    margin: 0;
    padding-left: 20px;
}

.aiab-diagram-details li {
    margin-bottom: 5px;
}

.aiab-diagram-details .link-status {
    font-size: 12px;
    padding: 2px 6px;
    border-radius: 4px;
    margin-left: 5px;
}

.aiab-diagram-details .link-status.active {
    background: #e8f5e9;
    color: #2e7d32;
}

.aiab-diagram-details .link-status.pending {
    background: #fff3e0;
    color: #e65100;
}

.view-post-link {
    font-size: 11px;
    margin-left: 5px;
}
</style>

<script>
jQuery(document).ready(function($) {
    // View Articles Modal
    $('.aiab-view-sphere').on('click', function() {
        var sphereId = $(this).data('id');
        
        $.post(aiab.ajax_url, {
            action: 'aiab_get_sphere_articles',
            nonce: aiab.nonce,
            sphere_id: sphereId
        }, function(response) {
            if (response.success) {
                var html = '<table class="aiab-table"><thead><tr><th>Type</th><th>Title</th><th>Status</th><th>Words</th><th>Links</th></tr></thead><tbody>';
                response.data.forEach(function(article) {
                    var linkCount = 0;
                    try {
                        var links = JSON.parse(article.internal_links || '[]');
                        linkCount = links.length;
                    } catch(e) {}
                    
                    html += '<tr>';
                    html += '<td>' + (article.is_pillar == 1 ? '⭐ Pillar' : '📄 Cluster') + '</td>';
                    html += '<td>' + (article.title || article.keyword);
                    if (article.wp_post_id) {
                        html += ' <a href="<?php echo admin_url('post.php?action=edit&post='); ?>' + article.wp_post_id + '" target="_blank" class="view-post-link">✏️ Edit</a>';
                    }
                    html += '</td>';
                    html += '<td><span class="aiab-badge badge-' + article.status + '">' + article.status + '</span></td>';
                    html += '<td>' + (article.word_count || '-') + '</td>';
                    html += '<td>' + linkCount + ' links</td>';
                    html += '</tr>';
                });
                html += '</tbody></table>';
                $('#aiab-sphere-articles').html(html);
                $('#aiab-sphere-modal').removeClass('hidden');
            }
        });
    });
    
    // View Diagram Modal
    $('.aiab-view-diagram').on('click', function() {
        var sphereId = $(this).data('id');
        var pillarKeyword = $(this).data('pillar');
        
        $.post(aiab.ajax_url, {
            action: 'aiab_get_sphere_articles',
            nonce: aiab.nonce,
            sphere_id: sphereId
        }, function(response) {
            if (response.success) {
                renderClusterDiagram(response.data, pillarKeyword);
                $('#aiab-diagram-modal').removeClass('hidden');
            }
        });
    });
    
    // Render the cluster diagram
    function renderClusterDiagram(articles, pillarKeyword) {
        var $diagram = $('#aiab-cluster-diagram');
        $diagram.empty();
        
        var pillar = null;
        var clusters = [];
        
        // Separate pillar from clusters
        articles.forEach(function(article) {
            if (article.is_pillar == 1) {
                pillar = article;
            } else {
                clusters.push(article);
            }
        });
        
        if (!pillar) {
            pillar = { keyword: pillarKeyword, title: pillarKeyword, status: 'pending' };
        }
        
        var centerX = 250;
        var centerY = 250;
        var radius = 170;
        
        // Calculate positions for all clusters first
        var clusterPositions = [];
        var angleStep = (2 * Math.PI) / clusters.length;
        var startAngle = -Math.PI / 2; // Start from top
        
        clusters.forEach(function(cluster, index) {
            var angle = startAngle + (index * angleStep);
            clusterPositions.push({
                x: centerX + radius * Math.cos(angle),
                y: centerY + radius * Math.sin(angle),
                angle: angle,
                cluster: cluster,
                index: index
            });
        });
        
        // Draw INTER-CLUSTER links first (so they appear behind nodes)
        // Each cluster links to previous and next cluster (circular)
        if (clusters.length > 1) {
            clusterPositions.forEach(function(pos, index) {
                var nextIndex = (index + 1) % clusters.length;
                var nextPos = clusterPositions[nextIndex];
                
                // Check if this cluster has links
                var hasLinks = false;
                try {
                    var links = JSON.parse(pos.cluster.internal_links || '[]');
                    hasLinks = links.length > 0;
                } catch(e) {}
                
                var nextHasLinks = false;
                try {
                    var links = JSON.parse(nextPos.cluster.internal_links || '[]');
                    nextHasLinks = links.length > 0;
                } catch(e) {}
                
                var bothLinked = pos.cluster.status === 'published' && nextPos.cluster.status === 'published' && hasLinks && nextHasLinks;
                var statusClass = bothLinked ? 'linked' : 'pending';
                
                // Draw curved line between clusters using SVG
                var $arc = drawArcBetweenPoints(pos.x, pos.y, nextPos.x, nextPos.y, statusClass, 'cluster-to-cluster');
                $diagram.append($arc);
            });
        }
        
        // Draw pillar node (center)
        var $pillarNode = $('<div class="cluster-node pillar" data-article-id="' + (pillar.id || 0) + '">' +
            '<span class="node-label">' + truncate(pillar.title || pillar.keyword, 40) + '</span>' +
            '</div>');
        $diagram.append($pillarNode);
        
        // Draw cluster nodes and lines to pillar
        clusterPositions.forEach(function(pos) {
            var cluster = pos.cluster;
            var x = pos.x - 55; // 55 = half of node width
            var y = pos.y - 55;
            
            var hasLinks = false;
            try {
                var links = JSON.parse(cluster.internal_links || '[]');
                hasLinks = links.length > 0;
            } catch(e) {}
            
            var statusClass = cluster.status === 'published' && hasLinks ? 'linked' : 'pending';
            
            var $clusterNode = $('<div class="cluster-node cluster ' + statusClass + '" ' +
                'data-article-id="' + cluster.id + '" ' +
                'data-index="' + pos.index + '" ' +
                'style="left: ' + x + 'px; top: ' + y + 'px;">' +
                '<span class="node-label">' + truncate(cluster.title || cluster.keyword, 35) + '</span>' +
                '</div>');
            $diagram.append($clusterNode);
            
            // Draw line to pillar
            var lineLength = radius - 70;
            var lineAngle = pos.angle * (180 / Math.PI);
            
            var $line = $('<div class="cluster-line pillar-link ' + statusClass + '" style="' +
                'left: ' + centerX + 'px; ' +
                'top: ' + centerY + 'px; ' +
                'width: ' + lineLength + 'px; ' +
                'transform: rotate(' + lineAngle + 'deg);' +
                '"></div>');
            $diagram.append($line);
        });
        
        // Build details panel
        var detailsHtml = '<h4>📊 Sphere Structure</h4>';
        detailsHtml += '<p><strong>Pillar:</strong> ' + (pillar.title || pillar.keyword);
        if (pillar.wp_post_id) {
            detailsHtml += ' <a href="<?php echo home_url('/?p='); ?>' + pillar.wp_post_id + '" target="_blank" class="view-post-link">🔗 View Post</a>';
        }
        detailsHtml += '</p>';
        
        detailsHtml += '<p><strong>Supporting Articles:</strong></p><ul>';
        clusters.forEach(function(cluster, index) {
            var hasLinks = false;
            var linkCount = 0;
            try {
                var links = JSON.parse(cluster.internal_links || '[]');
                hasLinks = links.length > 0;
                linkCount = links.length;
            } catch(e) {}
            
            var statusLabel = cluster.status === 'published' && hasLinks 
                ? '<span class="link-status active">✓ Linked (' + linkCount + ')</span>' 
                : '<span class="link-status pending">⏳ Pending</span>';
            
            detailsHtml += '<li>' + (index + 1) + '. ' + truncate(cluster.title || cluster.keyword, 50) + ' ' + statusLabel;
            if (cluster.wp_post_id) {
                detailsHtml += ' <a href="<?php echo home_url('/?p='); ?>' + cluster.wp_post_id + '" target="_blank" class="view-post-link">🔗</a>';
            }
            detailsHtml += '</li>';
        });
        detailsHtml += '</ul>';
        
        // Add linking explanation
        detailsHtml += '<div style="margin-top: 15px; padding: 10px; background: #e8f5e9; border-radius: 6px; font-size: 12px;">';
        detailsHtml += '<strong>🔗 Link Structure:</strong><br>';
        detailsHtml += '• Pillar → links to all ' + clusters.length + ' supporting articles<br>';
        detailsHtml += '• Each supporting article → links to pillar + 2 adjacent articles (circular)<br>';
        detailsHtml += '• Total internal links per sphere: ~' + (clusters.length + (clusters.length * 3)) + ' links';
        detailsHtml += '</div>';
        
        $('#aiab-diagram-details').html(detailsHtml);
        
        // Click on node to highlight
        $('.cluster-node').on('click', function() {
            var articleId = $(this).data('article-id');
            var article = articles.find(function(a) { return a.id == articleId; });
            if (article && article.wp_post_id) {
                window.open('<?php echo home_url('/?p='); ?>' + article.wp_post_id, '_blank');
            }
        });
    }
    
    // Draw curved arc between two points
    function drawArcBetweenPoints(x1, y1, x2, y2, statusClass, lineType) {
        // Calculate the midpoint and curve control point
        var midX = (x1 + x2) / 2;
        var midY = (y1 + y2) / 2;
        
        // Calculate distance and angle between points
        var dx = x2 - x1;
        var dy = y2 - y1;
        var distance = Math.sqrt(dx * dx + dy * dy);
        var angle = Math.atan2(dy, dx);
        
        // Offset the control point outward for curve
        var curveOffset = distance * 0.15; // Curve intensity
        var controlX = midX + curveOffset * Math.cos(angle + Math.PI/2);
        var controlY = midY + curveOffset * Math.sin(angle + Math.PI/2);
        
        // Create SVG element for the curved line
        var svg = document.createElementNS('http://www.w3.org/2000/svg', 'svg');
        svg.setAttribute('class', 'cluster-arc ' + statusClass);
        svg.setAttribute('style', 'position: absolute; left: 0; top: 0; width: 500px; height: 500px; pointer-events: none; z-index: 0;');
        
        var path = document.createElementNS('http://www.w3.org/2000/svg', 'path');
        var d = 'M ' + x1 + ' ' + y1 + ' Q ' + controlX + ' ' + controlY + ' ' + x2 + ' ' + y2;
        path.setAttribute('d', d);
        path.setAttribute('fill', 'none');
        path.setAttribute('stroke-width', '2');
        
        if (statusClass === 'linked') {
            path.setAttribute('stroke', '#4caf50');
        } else {
            path.setAttribute('stroke', '#ff9800');
            path.setAttribute('stroke-dasharray', '5,5');
        }
        
        svg.appendChild(path);
        return svg;
    }
    
    function truncate(str, length) {
        if (!str) return '';
        return str.length > length ? str.substring(0, length) + '...' : str;
    }
    
    // Close modals
    $('.aiab-modal-close').on('click', function() {
        $(this).closest('.aiab-modal').addClass('hidden');
    });
    
    $(document).on('click', '.aiab-modal', function(e) {
        if (e.target === this) {
            $(this).addClass('hidden');
        }
    });
    
    // Note: Delete and Force Complete now use direct PHP links (no AJAX) to bypass WAF blocks
});
</script>
