<?php
/**
 * Plugin Name: Eternal Auto Blogger - Free
 * Plugin URI: https://eternalautoblogger.com
 * Description: An autonomous AI-powered content engine that researches, plans, writes, and publishes SEO-optimized article clusters based on persona-driven thought spheres. Free version with OpenAI support.
 * Version: 2.6.0-free
 * Author: Ventus Media
 * Author URI: https://www.ventusserver.com
 * License: GPL v2 or later
 * Text Domain: ai-autoblogger
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Plugin constants
define('AIAB_VERSION', '2.6.0-free');
define('AIAB_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('AIAB_PLUGIN_URL', plugin_dir_url(__FILE__));
define('AIAB_PLUGIN_BASENAME', plugin_basename(__FILE__));

/**
 * Main Plugin Class
 */
class AI_AutoBlogger {
    
    private static $instance = null;
    
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        $this->load_dependencies();
        $this->init_hooks();
    }
    
    private function load_dependencies() {
        // Core classes
        require_once AIAB_PLUGIN_DIR . 'includes/class-database.php';
        require_once AIAB_PLUGIN_DIR . 'includes/class-persona.php';
        require_once AIAB_PLUGIN_DIR . 'includes/class-research-engine.php';
        require_once AIAB_PLUGIN_DIR . 'includes/class-thought-sphere.php';
        require_once AIAB_PLUGIN_DIR . 'includes/class-ai-writer.php';
        require_once AIAB_PLUGIN_DIR . 'includes/class-article-publisher.php';
        require_once AIAB_PLUGIN_DIR . 'includes/class-link-manager.php';
        require_once AIAB_PLUGIN_DIR . 'includes/class-image-generator.php';
        require_once AIAB_PLUGIN_DIR . 'includes/class-orchestrator.php';
        require_once AIAB_PLUGIN_DIR . 'includes/class-logger.php';
        // Admin classes
        if (is_admin()) {
            require_once AIAB_PLUGIN_DIR . 'admin/class-admin.php';
        }
    }
    
    private function init_hooks() {
        // Activation/Deactivation
        register_activation_hook(__FILE__, array($this, 'activate'));
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));
        
        // Initialize components
        add_action('plugins_loaded', array($this, 'init'));
        
        // Register cron schedules
        add_filter('cron_schedules', array($this, 'add_cron_schedules'));
        
        // The main cron hook - THE ETERNAL LOOP
        add_action('aiab_run_autoblogger', array($this, 'run_cycle'));
        
        // Continuation cron - checks for incomplete work and triggers continuation
        add_action('aiab_continuation_check', array($this, 'continuation_check'));
        
        // Output schema on single posts
        add_action('wp_head', array($this, 'output_schema_markup'));
    }
    
    public function init() {
        // Initialize admin
        if (is_admin()) {
            AIAB_Admin::get_instance();
            
            // Always ensure tables exist when on plugin admin pages
            if (isset($_GET['page']) && strpos($_GET['page'], 'aiab') === 0) {
                $this->ensure_tables_exist();
            }
        }
        
        // Initialize logger
        AIAB_Logger::get_instance();
        
        // Check for version upgrade and run migrations
        $current_version = get_option('aiab_db_version', '0');
        if (version_compare($current_version, AIAB_VERSION, '<')) {
            AIAB_Database::create_tables();
            AIAB_Database::run_migrations();
            update_option('aiab_db_version', AIAB_VERSION);
        }
        
        // Ensure continuation check is scheduled with correct interval based on provider
        $provider = get_option('aiab_ai_provider', 'anthropic');
        $walter_continuous = get_option('aiab_walter_continuous_mode', 1);
        $is_walter_mode = ($provider === 'walter' && $walter_continuous);
        $desired_interval = $is_walter_mode ? 'every_two_minutes' : 'every_five_minutes';
        
        // Check if continuation is scheduled with correct interval
        $current_continuation = wp_next_scheduled('aiab_continuation_check');
        if ($current_continuation) {
            // Get current schedule
            $crons = _get_cron_array();
            $current_interval = null;
            foreach ($crons as $time => $cron) {
                if (isset($cron['aiab_continuation_check'])) {
                    foreach ($cron['aiab_continuation_check'] as $hook) {
                        $current_interval = $hook['schedule'];
                        break 2;
                    }
                }
            }
            
            // Reschedule if interval doesn't match
            if ($current_interval !== $desired_interval) {
                wp_clear_scheduled_hook('aiab_continuation_check');
                wp_schedule_event(time() + 60, $desired_interval, 'aiab_continuation_check');
                AIAB_Logger::info("Continuation check rescheduled to {$desired_interval} (Custom Server mode: " . ($is_walter_mode ? 'ON' : 'OFF') . ")");
            }
        } else {
            // Not scheduled, schedule it
            wp_schedule_event(time() + 60, $desired_interval, 'aiab_continuation_check');
        }
    }
    
    public function activate() {
        // Create database tables
        AIAB_Database::create_tables();
        
        // Run migrations for existing installations
        AIAB_Database::run_migrations();
        
        // Set default options
        $this->set_default_options();
        
        // Schedule main cron if not already scheduled
        // NOTE: Schedule for 1 hour from now, NOT immediately (time())
        // This prevents auto-running cycles on every plugin activation during testing
        if (!wp_next_scheduled('aiab_run_autoblogger')) {
            $schedule = get_option('aiab_cron_schedule', 'daily');
            $first_run = time() + HOUR_IN_SECONDS; // 1 hour from now, not immediately
            wp_schedule_event($first_run, $schedule, 'aiab_run_autoblogger');
            AIAB_Logger::info("Cron scheduled for first run in 1 hour");
        }
        
        // Schedule continuation check based on provider
        // Walter mode uses more aggressive 2-minute checks
        $provider = get_option('aiab_ai_provider', 'anthropic');
        $continuation_interval = ($provider === 'walter') ? 'every_two_minutes' : 'every_five_minutes';
        
        if (!wp_next_scheduled('aiab_continuation_check')) {
            wp_schedule_event(time() + 120, $continuation_interval, 'aiab_continuation_check');
            AIAB_Logger::info("Continuation check scheduled ({$continuation_interval})");
        }
        
        // Flush rewrite rules
        flush_rewrite_rules();
    }
    
    public function deactivate() {
        // Clear scheduled cron events
        wp_clear_scheduled_hook('aiab_run_autoblogger');
        wp_clear_scheduled_hook('aiab_continuation_check');
    }
    
    /**
     * Ensure all required database tables exist
     */
    private function ensure_tables_exist() {
        global $wpdb;
        
        // Check if log table exists (it's the one most likely to be missing)
        $log_table = $wpdb->prefix . 'aiab_log';
        $exists = $wpdb->get_var("SHOW TABLES LIKE '$log_table'");
        
        if (!$exists) {
            // Tables are missing, create them
            AIAB_Database::create_tables();
            AIAB_Database::run_migrations();
        }
    }
    
    private function set_default_options() {
        $defaults = array(
            'aiab_cron_schedule' => 'daily',
            'aiab_articles_per_sphere' => 4,  // Free version limit: 1 pillar + 3 clusters
            'aiab_ai_provider' => 'openai',  // Free version: OpenAI only
            'aiab_ai_model' => 'gpt-4-turbo-preview',  // Free version: GPT-4 Turbo
            'aiab_image_provider' => 'dalle',
            'aiab_search_provider' => 'free',  // Free version: Free search only
            'aiab_auto_publish' => 1,
            'aiab_post_status' => 'publish',
            'aiab_external_link_sources' => '',
            'aiab_pillar_word_count' => 600,   // Free version: OpenAI max ~600 words
            'aiab_cluster_word_count' => 600,  // Free version: OpenAI max ~600 words
            'aiab_max_tokens' => 4096,
        );
        
        foreach ($defaults as $key => $value) {
            if (get_option($key) === false) {
                add_option($key, $value);
            }
        }
    }
    
    public function add_cron_schedules($schedules) {
        // Continuation check - runs frequently to ensure work continues
        $schedules['every_five_minutes'] = array(
            'interval' => 300,
            'display' => __('Every 5 Minutes (Continuation)', 'ai-autoblogger')
        );
        // Walter mode - more aggressive continuation
        $schedules['every_two_minutes'] = array(
            'interval' => 120,
            'display' => __('Every 2 Minutes (Custom Server Mode)', 'ai-autoblogger')
        );
        $schedules['weekly'] = array(
            'interval' => 604800,
            'display' => __('Once Weekly', 'ai-autoblogger')
        );
        $schedules['twice_daily'] = array(
            'interval' => 43200,
            'display' => __('Twice Daily', 'ai-autoblogger')
        );
        $schedules['every_six_hours'] = array(
            'interval' => 21600,
            'display' => __('Every 6 Hours', 'ai-autoblogger')
        );
        $schedules['every_three_hours'] = array(
            'interval' => 10800,
            'display' => __('Every 3 Hours', 'ai-autoblogger')
        );
        return $schedules;
    }
    
    /**
     * Output schema markup on single posts
     */
    public function output_schema_markup() {
        if (!is_singular('post')) {
            return;
        }
        
        $post_id = get_the_ID();
        
        // Check if this is an AI-generated post
        $is_aiab = get_post_meta($post_id, '_aiab_generated', true);
        if (!$is_aiab) {
            return;
        }
        
        // Get stored schema
        $schema_json = get_post_meta($post_id, '_aiab_schema_json', true);
        if (empty($schema_json)) {
            return;
        }
        
        // Don't output if RankMath or Yoast is handling schema
        if (class_exists('RankMath') || defined('WPSEO_VERSION')) {
            // These plugins handle their own schema - only output if they don't have schema
            return;
        }
        
        // Output the schema
        echo "\n<!-- Eternal Auto Blogger Schema -->\n";
        echo '<script type="application/ld+json">' . "\n";
        echo $schema_json;
        echo "\n</script>\n";
    }
    
    /**
     * THE ETERNAL CYCLE - Main entry point triggered by cron
     */
    public function run_cycle() {
        // License check - don't run if not licensed
        if (function_exists('aiab_license_client') && !aiab_license_client()->is_valid()) {
            AIAB_Logger::log('Cycle skipped - license not active');
            return;
        }
        
        AIAB_Logger::log('=== CRON CYCLE STARTED ===');
        
        try {
            $orchestrator = new AIAB_Orchestrator();
            $orchestrator->run();
        } catch (Exception $e) {
            AIAB_Logger::log('CRITICAL ERROR: ' . $e->getMessage(), 'error');
        }
        
        AIAB_Logger::log('=== CRON CYCLE COMPLETED ===');
    }
    
    /**
     * AUTONOMOUS CONTINUATION CHECK
     * Runs every 5 minutes (or 2 minutes in Walter mode) to ensure incomplete work continues
     * This is the key to true autonomy - the system monitors itself
     */
    public function continuation_check() {
        // License check - don't continue if not licensed
        if (function_exists('aiab_license_client') && !aiab_license_client()->is_valid()) {
            return;
        }
        
        global $wpdb;
        
        // Check if Walter Continuous Mode is enabled
        $provider = get_option('aiab_ai_provider', 'anthropic');
        $walter_continuous = get_option('aiab_walter_continuous_mode', 1); // Default ON for Walter
        $is_walter_mode = ($provider === 'walter' && $walter_continuous);
        
        // Only continue if main cron is enabled (user wants autonomous operation)
        if (!wp_next_scheduled('aiab_run_autoblogger')) {
            return; // Main cron disabled, don't auto-continue
        }
        
        // STEP 0: Auto-advance any spheres stuck in writing phase
        // This runs BEFORE checking for locks, ensuring stuck spheres get fixed
        $this->auto_advance_stuck_writing_spheres();
        
        // Check if there's an orchestrator lock (cycle currently running)
        $lock = get_option('aiab_orchestrator_lock');
        if ($lock) {
            $lock_data = json_decode($lock, true);
            $lock_time = isset($lock_data['time']) ? intval($lock_data['time']) : 0;
            
            // Walter mode: Allow longer locks (20 minutes) since inference is slow
            $max_lock_age = $is_walter_mode ? 1200 : 600; // 20 min for Walter, 10 min otherwise
            
            // If lock is fresh, a cycle is running
            if ($lock_time > 0 && (time() - $lock_time) < $max_lock_age) {
                return; // Cycle in progress, don't interrupt
            }
            
            // If lock is stale, clear it
            if ($lock_time > 0 && (time() - $lock_time) >= $max_lock_age) {
                delete_option('aiab_orchestrator_lock');
                AIAB_Logger::warning("🔧 Continuation check cleared stale lock (" . round((time() - $lock_time) / 60) . " minutes old)");
            }
        }
        
        // Check for incomplete spheres that need continuation
        $spheres_table = AIAB_Database::get_table('thought_spheres');
        
        $incomplete_sphere = $wpdb->get_row(
            "SELECT id, status, phase, updated_at, created_at 
             FROM $spheres_table 
             WHERE status NOT IN ('completed', 'failed') 
             ORDER BY created_at ASC 
             LIMIT 1"
        );
        
        if (!$incomplete_sphere) {
            AIAB_Logger::debug("Continuation check: No incomplete spheres found");
            return; // No pending work
        }
        
        AIAB_Logger::debug("Continuation check: Found incomplete sphere", array(
            'sphere_id' => $incomplete_sphere->id,
            'status' => $incomplete_sphere->status,
            'phase' => $incomplete_sphere->phase
        ));
        
        // Check if enough time has passed since last activity (avoid hammering)
        $last_activity_str = $incomplete_sphere->updated_at ?: ($incomplete_sphere->created_at ?: null);
        
        if (empty($last_activity_str) || $last_activity_str === '0000-00-00 00:00:00') {
            $minutes_since_activity = 10; 
        } else {
            $last_activity = strtotime($last_activity_str);
            if ($last_activity === false || $last_activity === 0) {
                $minutes_since_activity = 10;
            } else {
                $minutes_since_activity = (time() - $last_activity) / 60;
            }
        }
        
        // Walter mode: Be more aggressive - only wait 2 minutes
        // Standard mode: Wait 5 minutes between continuation attempts
        $min_wait = $is_walter_mode ? 2 : 5;
        
        if ($minutes_since_activity < $min_wait) {
            AIAB_Logger::debug("Continuation check: Waiting for activity timeout", array(
                'sphere_id' => $incomplete_sphere->id,
                'minutes_since_activity' => round($minutes_since_activity, 1),
                'min_wait' => $min_wait
            ));
            return; // Recent activity, wait a bit
        }
        
        // Check for pending articles in this sphere
        $articles_table = AIAB_Database::get_table('articles');
        $pending_articles = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM $articles_table 
             WHERE sphere_id = %d 
             AND status IN ('planned', 'writing')",
            $incomplete_sphere->id
        ));
        
        // Check for retryable failed articles (failed but under max retries)
        $retryable_failed = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM $articles_table 
             WHERE sphere_id = %d 
             AND status = 'failed' 
             AND (retry_count IS NULL OR retry_count < 3)",
            $incomplete_sphere->id
        ));
        
        // Also check for written but not published
        $unpublished = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM $articles_table 
             WHERE sphere_id = %d 
             AND status = 'written'",
            $incomplete_sphere->id
        ));
        
        $needs_work = ($pending_articles > 0) || ($retryable_failed > 0) || ($unpublished > 0) || 
                      !in_array($incomplete_sphere->phase, array('complete'));
        
        if ($needs_work) {
            AIAB_Logger::info("🔄 AUTONOMOUS CONTINUATION: Detected pending work", array(
                'sphere_id' => $incomplete_sphere->id,
                'status' => $incomplete_sphere->status,
                'phase' => $incomplete_sphere->phase,
                'pending_articles' => $pending_articles,
                'retryable_failed' => $retryable_failed,
                'unpublished' => $unpublished,
                'minutes_since_activity' => round($minutes_since_activity, 1),
                'walter_mode' => $is_walter_mode ? 'ON' : 'OFF'
            ));
            
            // Trigger the main cycle
            $this->run_cycle();
        }
    }
    
    /**
     * Auto-advance spheres stuck in writing phase to linking
     * Called by continuation_check to ensure autonomous operation
     */
    private function auto_advance_stuck_writing_spheres() {
        global $wpdb;
        $spheres_table = AIAB_Database::get_table('thought_spheres');
        $articles_table = AIAB_Database::get_table('articles');
        
        // Find spheres stuck in writing phases for more than 10 minutes
        $stuck_spheres = $wpdb->get_results(
            "SELECT id, pillar_keyword, phase, status, updated_at FROM $spheres_table 
             WHERE phase IN ('writing_pillar', 'writing_clusters') 
             AND status NOT IN ('completed', 'failed', 'paused')
             AND (updated_at IS NULL OR updated_at < DATE_SUB(NOW(), INTERVAL 10 MINUTE))"
        );
        
        foreach ($stuck_spheres as $sphere) {
            // Count articles that CAN still be worked on
            $workable = $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM $articles_table 
                 WHERE sphere_id = %d 
                 AND (
                     status IN ('planned', 'writing')
                     OR (status = 'failed' AND (retry_count IS NULL OR retry_count < 3))
                 )",
                $sphere->id
            ));
            
            // Count written articles
            $written = $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM $articles_table 
                 WHERE sphere_id = %d 
                 AND status = 'written'",
                $sphere->id
            ));
            
            // Count permanently failed
            $perm_failed = $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM $articles_table 
                 WHERE sphere_id = %d 
                 AND status IN ('auth_failed', 'budget_failed', 'max_retries', 'failed')
                 AND retry_count >= 3",
                $sphere->id
            ));
            
            // If no workable articles AND we have at least some written ones, advance to linking
            if ($workable == 0 && $written > 0) {
                AIAB_Logger::warning("🔧 CONTINUATION AUTO-ADVANCE: Sphere stuck in writing - advancing to linking", array(
                    'sphere_id' => $sphere->id,
                    'pillar' => $sphere->pillar_keyword,
                    'current_phase' => $sphere->phase,
                    'written_articles' => $written,
                    'permanently_failed' => $perm_failed
                ));
                
                $wpdb->update(
                    $spheres_table,
                    array(
                        'phase' => 'linking',
                        'updated_at' => current_time('mysql')
                    ),
                    array('id' => $sphere->id)
                );
            }
        }
    }
    
    /**
     * Manual trigger for testing
     */
    public function manual_run() {
        $this->run_cycle();
    }
}

// Initialize the plugin
function aiab_init() {
    return AI_AutoBlogger::get_instance();
}

// Start the engine
aiab_init();

// Helper function for manual triggering via admin
function aiab_manual_trigger() {
    $plugin = AI_AutoBlogger::get_instance();
    $plugin->manual_run();
}
