/**
 * AI AutoBlogger Admin JavaScript
 */

(function($) {
    'use strict';

    // Initialize on document ready
    $(document).ready(function() {
        console.log('AIAB Admin JS v2.0.3 loaded');
        initManualRun();
        initClearLock();
        initKillZombies();
        initPersonaForm();
        initPersonaDelete();
        initPersonaExport();
        initPersonaImport();
        initSearchProviderToggle();
        initLocalitySearch();
        console.log('AIAB Admin JS initialized');
    });

    /**
     * Clear Lock Button
     */
    function initClearLock() {
        $('#aiab-clear-lock').on('click', function() {
            var $button = $(this);
            var $result = $('#aiab-run-result');
            
            $button.prop('disabled', true).text('🔄 Clearing...');
            
            $.ajax({
                url: aiab.ajax_url,
                type: 'POST',
                data: {
                    action: 'aiab_clear_lock',
                    nonce: aiab.nonce
                },
                success: function(response) {
                    $button.prop('disabled', false).text('🔓 Clear Lock');
                    $result.removeClass('hidden error').addClass('success');
                    
                    if (response.success) {
                        $result.html('<strong>✅ ' + response.data.message + '</strong>');
                    } else {
                        $result.removeClass('success').addClass('error');
                        $result.html('<strong>❌ Error:</strong> ' + response.data);
                    }
                },
                error: function() {
                    $button.prop('disabled', false).text('🔓 Clear Lock');
                    $result.removeClass('hidden success').addClass('error');
                    $result.html('<strong>❌ Connection error</strong>');
                }
            });
        });
    }

    /**
     * Kill Zombies Button
     */
    function initKillZombies() {
        $('#aiab-kill-zombies').on('click', function() {
            if (!confirm('💀 This will STOP all running cycles and pause all in-progress spheres.\n\nAre you sure?')) {
                return;
            }
            
            var $button = $(this);
            var $result = $('#aiab-run-result');
            
            $button.prop('disabled', true).text('💀 Killing...');
            
            $.ajax({
                url: aiab.ajax_url,
                type: 'POST',
                data: {
                    action: 'aiab_kill_zombies',
                    nonce: aiab.nonce
                },
                success: function(response) {
                    $button.prop('disabled', false).text('💀 Kill All Zombies');
                    $result.removeClass('hidden error').addClass('success');
                    
                    if (response.success) {
                        $result.html('<strong>💀 ' + response.data.message + '</strong>');
                        // Reload page after 2 seconds to refresh status
                        setTimeout(function() {
                            location.reload();
                        }, 2000);
                    } else {
                        $result.removeClass('success').addClass('error');
                        $result.html('<strong>❌ Error:</strong> ' + response.data);
                    }
                },
                error: function() {
                    $button.prop('disabled', false).text('💀 Kill All Zombies');
                    $result.removeClass('hidden success').addClass('error');
                    $result.html('<strong>❌ Connection error</strong>');
                }
            });
        });
    }

    /**
     * Manual Run Button with Auto-Continue Support
     */
    function initManualRun() {
        var autoContinueActive = false;
        var cycleCount = 0;
        var maxCycles = 50; // Safety limit
        
        function runCycle() {
            var $button = $('#aiab-manual-run');
            var $status = $('#aiab-run-status');
            var $result = $('#aiab-run-result');
            var $autoContinue = $('#aiab-auto-continue');
            var personaId = $('#aiab-persona-select').val();

            // Disable button and show loading
            $button.prop('disabled', true).text('⏳ Running...');
            $status.removeClass('hidden');
            
            if (autoContinueActive) {
                cycleCount++;
                $status.html('<div class="aiab-loading-spinner"></div> Auto-running cycle ' + cycleCount + '... (uncheck "Auto-continue" to stop)');
            }

            // Make AJAX call with longer timeout for Walter
            $.ajax({
                url: aiab.ajax_url,
                type: 'POST',
                timeout: 900000, // 15 minutes for slow inference
                data: {
                    action: 'aiab_manual_run',
                    nonce: aiab.nonce,
                    persona_id: personaId
                },
                success: function(response) {
                    $status.addClass('hidden');
                    $result.removeClass('hidden');

                    if (response.success) {
                        $result.removeClass('error').addClass('success');
                        var html = '<strong>✅ ' + response.data.message + '</strong>';
                        
                        var sphereComplete = false;
                        var hasMoreWork = true;
                        
                        if (response.data.status && response.data.status.sphere) {
                            var sphere = response.data.status.sphere;
                            html += '<br><br>';
                            html += '<strong>Sphere:</strong> ' + sphere.pillar + '<br>';
                            html += '<strong>Status:</strong> ' + sphere.status + '<br>';
                            html += '<strong>Phase:</strong> ' + sphere.phase + '<br>';
                            html += '<strong>Progress:</strong> ' + sphere.progress + ' articles';
                            
                            if (autoContinueActive) {
                                html += '<br><br><em>🔄 Cycle ' + cycleCount + ' complete</em>';
                            }
                            
                            // Check if sphere is complete
                            if (sphere.status === 'completed' || sphere.status === 'complete') {
                                sphereComplete = true;
                                html += '<br><br><strong>🎉 Sphere completed!</strong>';
                            }
                            
                            // Check progress to determine if more work needed
                            var progressMatch = sphere.progress.match(/(\d+)\/(\d+)/);
                            if (progressMatch) {
                                var current = parseInt(progressMatch[1]);
                                var total = parseInt(progressMatch[2]);
                                if (current >= total) {
                                    hasMoreWork = false;
                                }
                            }
                        }
                        
                        $result.html(html);
                        
                        // Auto-continue logic
                        if (autoContinueActive && $autoContinue.is(':checked') && hasMoreWork && cycleCount < maxCycles) {
                            // Wait 3 seconds then run next cycle
                            $status.removeClass('hidden').html('<div class="aiab-loading-spinner"></div> Starting next cycle in 3 seconds... (uncheck to stop)');
                            setTimeout(function() {
                                if ($autoContinue.is(':checked')) {
                                    runCycle();
                                } else {
                                    autoContinueActive = false;
                                    $status.addClass('hidden');
                                    $button.prop('disabled', false).text('▶️ Run Cycle Now');
                                }
                            }, 3000);
                        } else {
                            autoContinueActive = false;
                            $button.prop('disabled', false).text('▶️ Run Cycle Now');
                            
                            if (cycleCount >= maxCycles) {
                                $result.append('<br><br><strong>⚠️ Stopped after ' + maxCycles + ' cycles (safety limit)</strong>');
                            }
                            if (!hasMoreWork || sphereComplete) {
                                $result.append('<br><br><strong>✅ All work complete!</strong>');
                            }
                            cycleCount = 0;
                        }
                    } else {
                        $result.removeClass('success').addClass('error');
                        $result.html('<strong>❌ Error:</strong> ' + response.data);
                        autoContinueActive = false;
                        $button.prop('disabled', false).text('▶️ Run Cycle Now');
                        cycleCount = 0;
                    }
                },
                error: function(xhr, status, error) {
                    $status.addClass('hidden');
                    $result.removeClass('hidden success').addClass('error');
                    
                    if (status === 'timeout') {
                        $result.html('<strong>⏱️ Request timed out</strong> - The AI is still generating. Check logs and try again in a few minutes.');
                    } else {
                        $result.html('<strong>❌ Request failed:</strong> ' + error);
                    }
                    
                    // On error during auto-continue, retry after delay
                    if (autoContinueActive && $('#aiab-auto-continue').is(':checked') && cycleCount < maxCycles) {
                        $status.removeClass('hidden').html('<div class="aiab-loading-spinner"></div> Error occurred. Retrying in 30 seconds...');
                        setTimeout(function() {
                            if ($('#aiab-auto-continue').is(':checked')) {
                                runCycle();
                            } else {
                                autoContinueActive = false;
                                $status.addClass('hidden');
                                $('#aiab-manual-run').prop('disabled', false).text('▶️ Run Cycle Now');
                            }
                        }, 30000);
                    } else {
                        autoContinueActive = false;
                        $button.prop('disabled', false).text('▶️ Run Cycle Now');
                        cycleCount = 0;
                    }
                }
            });
        }
        
        $('#aiab-manual-run').on('click', function() {
            var $autoContinue = $('#aiab-auto-continue');
            
            if ($autoContinue.length && $autoContinue.is(':checked')) {
                autoContinueActive = true;
                cycleCount = 0;
            } else {
                autoContinueActive = false;
            }
            
            runCycle();
        });
        
        // Stop auto-continue when checkbox unchecked
        $(document).on('change', '#aiab-auto-continue', function() {
            var isChecked = $(this).is(':checked');
            if (!isChecked) {
                autoContinueActive = false;
            }
            
            // Save preference to database
            $.post(aiab.ajax_url, {
                action: 'aiab_save_auto_continue',
                nonce: aiab.nonce,
                enabled: isChecked ? 1 : 0
            });
            
            // Update visual style
            var $container = $(this).closest('div');
            if (isChecked) {
                $container.css({
                    'background': '#e8f5e9',
                    'border-color': '#4caf50'
                });
            } else {
                $container.css({
                    'background': '#f5f5f5',
                    'border-color': '#ddd'
                });
            }
        });
    }

    /**
     * Persona Form Handler
     */
    function initPersonaForm() {
        $('#aiab-persona-form').on('submit', function(e) {
            e.preventDefault();

            var $form = $(this);
            var $button = $form.find('button[type="submit"]');
            var originalText = $button.text();

            // Disable button
            $button.prop('disabled', true).text('💾 Saving...');

            // Collect form data
            var formData = {
                action: 'aiab_save_persona',
                nonce: aiab.nonce,
                id: $form.find('input[name="id"]').val(),
                name: $form.find('#persona-name').val(),
                personal_story: $form.find('#persona-story').val(),
                interests: $form.find('#persona-interests').val(),
                passions: $form.find('#persona-passions').val(),
                expertise_areas: $form.find('#persona-expertise').val(),
                localities: [],
                voice_style: $form.find('#persona-voice').val(),
                tone: $form.find('#persona-tone').val(),
                target_audience: $form.find('#persona-audience').val(),
                knowledge_sources: $form.find('#persona-sources').val(),
                writing_guidelines: $form.find('#persona-guidelines').val(),
                avoid_topics: $form.find('#persona-avoid').val(),
                language: $form.find('#persona-language').val(),
                is_active: $form.find('#persona-active').is(':checked') ? '1' : '0',
                wp_author_id: $form.find('#persona-author').val() || '',
                categories: []
            };
            
            // Collect localities from hidden inputs
            $form.find('input[name="localities[]"]').each(function() {
                formData.localities.push($(this).val());
            });
            
            // Collect checked categories
            $form.find('input[name="categories[]"]:checked').each(function() {
                formData.categories.push($(this).val());
            });

            $.ajax({
                url: aiab.ajax_url,
                type: 'POST',
                data: formData,
                success: function(response) {
                    if (response.success) {
                        // Show success message
                        alert('✅ ' + response.data.message);
                        
                        // Redirect to list if creating new
                        if (!formData.id) {
                            window.location.href = window.location.href.split('?')[0] + '?page=aiab-personas';
                        } else {
                            window.location.reload();
                        }
                    } else {
                        alert('❌ Error: ' + response.data);
                    }
                },
                error: function(xhr, status, error) {
                    alert('❌ Request failed: ' + error);
                },
                complete: function() {
                    $button.prop('disabled', false).text(originalText);
                }
            });
        });
    }

    /**
     * Persona Delete Handler
     */
    function initPersonaDelete() {
        $(document).on('click', '.aiab-delete-persona', function(e) {
            e.preventDefault();
            
            var $button = $(this);
            var personaId = $button.data('id');
            var personaName = $button.data('name');

            if (!confirm('Are you sure you want to delete "' + personaName + '"?\n\nThis cannot be undone.')) {
                return;
            }

            $button.prop('disabled', true).text('Deleting...');

            $.ajax({
                url: aiab.ajax_url,
                type: 'POST',
                data: {
                    action: 'aiab_delete_persona',
                    nonce: aiab.nonce,
                    id: personaId
                },
                success: function(response) {
                    if (response.success) {
                        $button.closest('tr').fadeOut(function() {
                            $(this).remove();
                        });
                    } else {
                        alert('❌ Error: ' + response.data);
                        $button.prop('disabled', false).text('Delete');
                    }
                },
                error: function(xhr, status, error) {
                    alert('❌ Request failed: ' + error);
                    $button.prop('disabled', false).text('Delete');
                }
            });
        });
    }

    /**
     * Persona Export - Download as .eab file
     */
    function initPersonaExport() {
        // Use event delegation for dynamically loaded content
        $(document).on('click', '.aiab-export-persona', function(e) {
            e.preventDefault();
            
            var $button = $(this);
            var personaId = $button.data('id');
            var personaName = $button.data('name');
            
            console.log('Export clicked for persona:', personaId, personaName);
            
            $button.prop('disabled', true).text('Exporting...');
            
            $.ajax({
                url: aiab.ajax_url,
                type: 'POST',
                data: {
                    action: 'aiab_export_personas',
                    nonce: aiab.nonce,
                    persona_id: personaId
                },
                success: function(response) {
                    console.log('Export response:', response);
                    $button.prop('disabled', false).html('📤 Export');
                    
                    if (response.success) {
                        // Create the JSON content
                        var jsonContent = JSON.stringify(response.data, null, 2);
                        
                        // Create blob for download
                        var blob = new Blob([jsonContent], { type: 'application/json' });
                        
                        // Create download link
                        var filename = personaName.toLowerCase()
                            .replace(/[^a-z0-9]+/g, '-')
                            .replace(/(^-|-$)/g, '') + '.eab';
                        
                        var url = URL.createObjectURL(blob);
                        var a = document.createElement('a');
                        a.href = url;
                        a.download = filename;
                        document.body.appendChild(a);
                        a.click();
                        document.body.removeChild(a);
                        URL.revokeObjectURL(url);
                        
                        // Show success message
                        alert('✅ Persona exported successfully!\n\nFile: ' + filename);
                    } else {
                        alert('❌ Export failed: ' + response.data);
                    }
                },
                error: function(xhr, status, error) {
                    console.log('Export error:', xhr, status, error);
                    $button.prop('disabled', false).html('📤 Export');
                    alert('❌ Export request failed: ' + error);
                }
            });
        });
    }

    /**
     * Persona Import - Upload .eab file
     */
    function initPersonaImport() {
        // Toggle import form visibility
        $(document).on('click', '#aiab-import-persona-btn', function(e) {
            e.preventDefault();
            console.log('Import button clicked');
            $('#aiab-import-form').slideToggle();
        });
        
        $(document).on('click', '#aiab-cancel-import', function(e) {
            e.preventDefault();
            $('#aiab-import-form').slideUp();
            $('#persona-import-file').val('');
        });
        
        // Handle import form submission
        $(document).on('submit', '#aiab-import-persona-form', function(e) {
            e.preventDefault();
            console.log('Import form submitted');
            
            var fileInput = document.getElementById('persona-import-file');
            if (!fileInput.files || !fileInput.files[0]) {
                alert('Please select a .eab file to import');
                return;
            }
            
            var file = fileInput.files[0];
            var filename = file.name.toLowerCase();
            
            console.log('Importing file:', filename);
            
            if (!filename.endsWith('.eab') && !filename.endsWith('.zip')) {
                alert('Invalid file type. Please select a .eab file.');
                return;
            }
            
            var formData = new FormData();
            formData.append('action', 'aiab_import_personas');
            formData.append('nonce', aiab.nonce);
            formData.append('persona_file', file);
            
            var $submitBtn = $(this).find('button[type="submit"]');
            $submitBtn.prop('disabled', true).text('Importing...');
            
            $.ajax({
                url: aiab.ajax_url,
                type: 'POST',
                data: formData,
                processData: false,
                contentType: false,
                success: function(response) {
                    console.log('Import response:', response);
                    $submitBtn.prop('disabled', false).text('📥 Import');
                    
                    if (response.success) {
                        alert('✅ ' + response.data.message);
                        // Reload page to show imported persona
                        location.reload();
                    } else {
                        alert('❌ Import failed: ' + response.data);
                    }
                },
                error: function(xhr, status, error) {
                    console.log('Import error:', xhr, status, error);
                    $submitBtn.prop('disabled', false).text('📥 Import');
                    alert('❌ Import request failed: ' + error);
                }
            });
        });
    }

    /**
     * Search Provider Toggle (show/hide API key fields)
     */
    function initSearchProviderToggle() {
        var $select = $('#aiab_search_provider');
        
        if ($select.length === 0) return;
        
        function updateVisibility() {
            var provider = $select.val();
            
            // Hide all API key rows first
            $('.search-api-key-row').hide();
            
            // Show the row for selected provider
            $('.search-api-key-row[data-provider="' + provider + '"]').show();
        }
        
        // Initial update
        updateVisibility();
        
        // Update on change
        $select.on('change', updateVisibility);
    }

    /**
     * Initialize Cron Controls (on document ready)
     */
    $(document).ready(function() {
        initCronControls();
    });
    
    function initCronControls() {
        var $result = $('#aiab-cron-result');
        
        // Enable Cron
        $('#aiab-enable-cron').on('click', function() {
            var $button = $(this);
            $button.prop('disabled', true).text('⏳ Enabling...');
            $result.html('');
            
            $.ajax({
                url: aiab.ajax_url,
                type: 'POST',
                data: {
                    action: 'aiab_enable_cron',
                    nonce: aiab.nonce
                },
                success: function(response) {
                    if (response.success) {
                        $result.html('<span style="color: green;">✅ ' + response.data.message + ' - Next run: ' + response.data.next_run_human + '</span>');
                        // Reload page to update UI
                        setTimeout(function() { location.reload(); }, 1500);
                    } else {
                        $button.prop('disabled', false).text('▶️ Enable Auto-Run');
                        $result.html('<span style="color: red;">❌ ' + response.data + '</span>');
                    }
                },
                error: function() {
                    $button.prop('disabled', false).text('▶️ Enable Auto-Run');
                    $result.html('<span style="color: red;">❌ Connection error</span>');
                }
            });
        });
        
        // Disable Cron
        $('#aiab-disable-cron').on('click', function() {
            if (!confirm('⏸️ This will stop the automatic schedule. You can re-enable it anytime.\n\nContinue?')) {
                return;
            }
            
            var $button = $(this);
            $button.prop('disabled', true).text('⏳ Disabling...');
            $result.html('');
            
            $.ajax({
                url: aiab.ajax_url,
                type: 'POST',
                data: {
                    action: 'aiab_disable_cron',
                    nonce: aiab.nonce
                },
                success: function(response) {
                    if (response.success) {
                        $result.html('<span style="color: orange;">⏸️ ' + response.data.message + '</span>');
                        // Reload page to update UI
                        setTimeout(function() { location.reload(); }, 1500);
                    } else {
                        $button.prop('disabled', false).text('⏸️ Disable Auto-Run');
                        $result.html('<span style="color: red;">❌ ' + response.data + '</span>');
                    }
                },
                error: function() {
                    $button.prop('disabled', false).text('⏸️ Disable Auto-Run');
                    $result.html('<span style="color: red;">❌ Connection error</span>');
                }
            });
        });
        
        // Reschedule Cron
        $('#aiab-reschedule-cron').on('click', function() {
            var $button = $(this);
            var schedule = $('#aiab-cron-schedule-select').val();
            
            $button.prop('disabled', true).text('⏳ Updating...');
            $result.html('');
            
            $.ajax({
                url: aiab.ajax_url,
                type: 'POST',
                data: {
                    action: 'aiab_reschedule_cron',
                    nonce: aiab.nonce,
                    schedule: schedule
                },
                success: function(response) {
                    $button.prop('disabled', false).text('🔄 Update Schedule');
                    if (response.success) {
                        $result.html('<span style="color: green;">✅ ' + response.data.message + ' - Next run: ' + response.data.next_run_human + '</span>');
                        // Reload page to update UI
                        setTimeout(function() { location.reload(); }, 1500);
                    } else {
                        $result.html('<span style="color: red;">❌ ' + response.data + '</span>');
                    }
                },
                error: function() {
                    $button.prop('disabled', false).text('🔄 Update Schedule');
                    $result.html('<span style="color: red;">❌ Connection error</span>');
                }
            });
        });
    }

    /**
     * Locality Search and Management
     */
    function initLocalitySearch() {
        var searchInput = document.getElementById('locality-search');
        var addBtn = document.getElementById('add-locality-btn');
        var suggestionsDiv = document.getElementById('locality-suggestions');
        var selectedContainer = document.getElementById('selected-localities');
        var noLocalitiesMsg = document.getElementById('no-localities-msg');
        
        // Exit if not on persona page
        if (!searchInput || !addBtn) {
            return;
        }
        
        console.log('AIAB: Locality search initialized');
        
        // Comprehensive list of world cities
        var worldCities = [
            // UK Cities (Primary)
            'London, United Kingdom',
            'Manchester, United Kingdom',
            'Birmingham, United Kingdom',
            'Leeds, United Kingdom',
            'Glasgow, United Kingdom',
            'Liverpool, United Kingdom',
            'Bristol, United Kingdom',
            'Sheffield, United Kingdom',
            'Edinburgh, United Kingdom',
            'Cardiff, United Kingdom',
            'Belfast, United Kingdom',
            'Newcastle, United Kingdom',
            'Nottingham, United Kingdom',
            'Southampton, United Kingdom',
            'Brighton, United Kingdom',
            'Oxford, United Kingdom',
            'Cambridge, United Kingdom',
            // UAE - All Emirates (Secondary)
            'Dubai, United Arab Emirates',
            'Abu Dhabi, United Arab Emirates',
            'Sharjah, United Arab Emirates',
            'Ajman, United Arab Emirates',
            'Fujairah, United Arab Emirates',
            'Ras Al Khaimah, United Arab Emirates',
            'Umm Al Quwain, United Arab Emirates',
            'Al Ain, United Arab Emirates',
            // Saudi Arabia
            'Riyadh, Saudi Arabia',
            'Jeddah, Saudi Arabia',
            'Mecca, Saudi Arabia',
            'Medina, Saudi Arabia',
            'Dammam, Saudi Arabia',
            // Gulf States
            'Doha, Qatar',
            'Kuwait City, Kuwait',
            'Manama, Bahrain',
            'Muscat, Oman',
            // USA Cities
            'New York, United States',
            'Los Angeles, United States',
            'Chicago, United States',
            'Houston, United States',
            'San Francisco, United States',
            'Miami, United States',
            'Boston, United States',
            'Seattle, United States',
            // Canada
            'Toronto, Canada',
            'Vancouver, Canada',
            'Montreal, Canada',
            // Australia
            'Sydney, Australia',
            'Melbourne, Australia',
            'Brisbane, Australia',
            // Europe
            'Paris, France',
            'Berlin, Germany',
            'Amsterdam, Netherlands',
            'Madrid, Spain',
            'Rome, Italy',
            'Zurich, Switzerland',
            // Asia
            'Singapore',
            'Hong Kong',
            'Tokyo, Japan',
            'Mumbai, India',
            'Delhi, India',
            'Bangkok, Thailand',
            'Kuala Lumpur, Malaysia'
        ];
        
        // Function to add a location
        function addLocation(location) {
            if (!location || location.trim() === '') return;
            
            location = location.trim();
            console.log('AIAB: Adding location:', location);
            
            // Check if already added
            var existing = selectedContainer.querySelector('[data-code="' + location + '"]');
            if (existing) {
                console.log('AIAB: Location already exists');
                return;
            }
            
            // Hide "no localities" message
            if (noLocalitiesMsg) {
                noLocalitiesMsg.style.display = 'none';
            }
            
            // Create tag element
            var tag = document.createElement('span');
            tag.className = 'locality-tag';
            tag.setAttribute('data-code', location);
            tag.style.cssText = 'display: inline-flex; align-items: center; gap: 5px; background: #e3f2fd; border: 1px solid #90caf9; padding: 5px 10px; border-radius: 20px; margin: 3px; font-size: 13px;';
            tag.innerHTML = '📍 ' + location + 
                '<button type="button" class="remove-locality" style="background: none; border: none; color: #f44336; cursor: pointer; font-size: 16px; padding: 0 2px; line-height: 1;">&times;</button>' +
                '<input type="hidden" name="localities[]" value="' + location + '">';
            
            selectedContainer.appendChild(tag);
            
            // Update quick select button
            var quickBtn = document.querySelector('.quick-locality[data-location="' + location + '"]');
            if (quickBtn) {
                quickBtn.classList.add('selected');
                quickBtn.style.background = '#4CAF50';
                quickBtn.style.color = '#fff';
            }
            
            console.log('AIAB: Location added successfully');
        }
        
        // Function to remove a location
        function removeLocation(location) {
            var tag = selectedContainer.querySelector('[data-code="' + location + '"]');
            if (tag) {
                tag.remove();
            }
            
            // Show "no localities" message if empty
            if (selectedContainer.querySelectorAll('.locality-tag').length === 0 && noLocalitiesMsg) {
                noLocalitiesMsg.style.display = 'block';
            }
            
            // Update quick select button
            var quickBtn = document.querySelector('.quick-locality[data-location="' + location + '"]');
            if (quickBtn) {
                quickBtn.classList.remove('selected');
                quickBtn.style.background = '#fff';
                quickBtn.style.color = '#333';
            }
        }
        
        // Add button click handler
        addBtn.addEventListener('click', function(e) {
            e.preventDefault();
            console.log('AIAB: Add button clicked');
            var location = searchInput.value;
            addLocation(location);
            searchInput.value = '';
            if (suggestionsDiv) suggestionsDiv.style.display = 'none';
        });
        
        // Enter key handler
        searchInput.addEventListener('keypress', function(e) {
            if (e.key === 'Enter' || e.keyCode === 13) {
                e.preventDefault();
                console.log('AIAB: Enter pressed');
                var location = searchInput.value;
                addLocation(location);
                searchInput.value = '';
                if (suggestionsDiv) suggestionsDiv.style.display = 'none';
            }
        });
        
        // Search input handler (show suggestions)
        var searchTimeout = null;
        searchInput.addEventListener('input', function() {
            var query = this.value.toLowerCase().trim();
            
            clearTimeout(searchTimeout);
            
            if (query.length < 2) {
                if (suggestionsDiv) suggestionsDiv.style.display = 'none';
                return;
            }
            
            searchTimeout = setTimeout(function() {
                var matches = worldCities.filter(function(city) {
                    return city.toLowerCase().indexOf(query) !== -1;
                }).slice(0, 8);
                
                if (matches.length > 0 && suggestionsDiv) {
                    var html = '';
                    matches.forEach(function(city) {
                        html += '<div class="locality-suggestion" data-location="' + city + '" style="padding: 10px 15px; cursor: pointer; border-bottom: 1px solid #eee;">📍 ' + city + '</div>';
                    });
                    html += '<div class="locality-suggestion custom" data-location="' + searchInput.value + '" style="padding: 10px 15px; cursor: pointer; background: #f0f0f0;">➕ Add "' + searchInput.value + '" as custom</div>';
                    suggestionsDiv.innerHTML = html;
                    suggestionsDiv.style.display = 'block';
                }
            }, 150);
        });
        
        // Suggestion click handler (using event delegation)
        document.addEventListener('click', function(e) {
            if (e.target.classList.contains('locality-suggestion') || e.target.closest('.locality-suggestion')) {
                var suggestionEl = e.target.classList.contains('locality-suggestion') ? e.target : e.target.closest('.locality-suggestion');
                var location = suggestionEl.getAttribute('data-location');
                console.log('AIAB: Suggestion clicked:', location);
                addLocation(location);
                searchInput.value = '';
                if (suggestionsDiv) suggestionsDiv.style.display = 'none';
            }
            
            // Remove button click
            if (e.target.classList.contains('remove-locality')) {
                var tag = e.target.closest('.locality-tag');
                if (tag) {
                    var location = tag.getAttribute('data-code');
                    console.log('AIAB: Remove clicked:', location);
                    removeLocation(location);
                }
            }
            
            // Quick select button click
            if (e.target.classList.contains('quick-locality')) {
                var location = e.target.getAttribute('data-location');
                console.log('AIAB: Quick select clicked:', location);
                
                if (e.target.classList.contains('selected')) {
                    removeLocation(location);
                } else {
                    addLocation(location);
                }
            }
            
            // Hide suggestions when clicking outside
            if (!e.target.closest('#locality-search') && !e.target.closest('#locality-suggestions')) {
                if (suggestionsDiv) suggestionsDiv.style.display = 'none';
            }
        });
        
        // Hover effects for suggestions
        document.addEventListener('mouseover', function(e) {
            if (e.target.classList.contains('locality-suggestion')) {
                e.target.style.background = '#e3f2fd';
            }
        });
        document.addEventListener('mouseout', function(e) {
            if (e.target.classList.contains('locality-suggestion')) {
                e.target.style.background = e.target.classList.contains('custom') ? '#f0f0f0' : '#fff';
            }
        });
    }

})(jQuery);
