<?php
/**
 * AI Writer
 * Generates article content using AI APIs (Anthropic Claude or OpenAI)
 */

if (!defined('ABSPATH')) {
    exit;
}

class AIAB_AI_Writer {
    
    private $persona;
    private $provider;
    private $model;
    private $api_key;
    
    // Map deprecated model names to new ones
    private static $model_migrations = array(
        'llama-3.1-sonar-large-128k-online' => 'sonar',
        'llama-3.1-sonar-huge-128k-online' => 'sonar-pro',
        'llama-3.1-sonar-small-128k-online' => 'sonar',
    );
    
    public function __construct(AIAB_Persona $persona) {
        $this->persona = $persona;
        $this->provider = get_option('aiab_ai_provider', 'anthropic');
        
        // Get the appropriate model based on provider
        if ($this->provider === 'walter') {
            $this->model = get_option('aiab_walter_model', 'qwen2.5:14b-instruct-q4_K_M');
        } else {
            $this->model = get_option('aiab_ai_model', 'claude-sonnet-4-20250514');
        }
        
        // Auto-migrate deprecated model names (only for non-Walter providers)
        if ($this->provider !== 'walter' && isset(self::$model_migrations[$this->model])) {
            $new_model = self::$model_migrations[$this->model];
            update_option('aiab_ai_model', $new_model);
            AIAB_Logger::info("Auto-migrated deprecated model '{$this->model}' to '{$new_model}'");
            $this->model = $new_model;
        }
        
        // Get the appropriate API key based on provider
        if ($this->provider === 'anthropic') {
            $this->api_key = get_option('aiab_anthropic_api_key', '');
        } elseif ($this->provider === 'perplexity') {
            $this->api_key = get_option('aiab_perplexity_api_key', '');
        } elseif ($this->provider === 'walter') {
            // Walter (Ollama) doesn't need an API key, just the server URL
            $this->api_key = 'walter-local';
        } else {
            $this->api_key = get_option('aiab_openai_api_key', '');
        }
        
        // DIAGNOSTIC: Log what provider/model/key is being used
        AIAB_Logger::debug("AI Writer initialized", array(
            'provider_setting' => $this->provider,
            'model_setting' => $this->model,
            'api_key_preview' => !empty($this->api_key) ? substr($this->api_key, 0, 12) . '...' : 'EMPTY',
            'key_type_detected' => $this->detect_key_type($this->api_key)
        ));
        
        // AUTO-FIX: Detect if API key doesn't match the provider setting (skip for Walter)
        if ($this->provider !== 'walter') {
            $detected_type = $this->detect_key_type($this->api_key);
            if ($detected_type && $detected_type !== $this->provider) {
                AIAB_Logger::warning("⚠️ API KEY MISMATCH DETECTED!", array(
                    'configured_provider' => $this->provider,
                    'key_appears_to_be' => $detected_type,
                    'action' => 'Auto-correcting provider to match API key'
                ));
                
                // Auto-correct the provider to match the key type
                $this->provider = $detected_type;
                
                // Also fix the model if it doesn't match the provider
                $this->auto_fix_model_for_provider();
            }
        }
        
        // If we still have no API key after all that, try to find ANY valid key (skip for Walter)
        if (empty($this->api_key) && $this->provider !== 'walter') {
            $this->try_fallback_api_keys();
        }
    }
    
    /**
     * Detect what type of API key this is based on its prefix
     */
    private function detect_key_type($key) {
        if (empty($key)) {
            return null;
        }
        
        $key = trim($key);
        
        if (strpos($key, 'sk-ant-') === 0 || strpos($key, 'sk-') === 0 && strpos($key, 'sk-proj-') !== 0) {
            // Anthropic keys start with sk-ant- (newer) or just sk- (older, but not sk-proj-)
            // Actually, let me be more careful - OpenAI also uses sk-
            // Anthropic uses: sk-ant-api...
            if (strpos($key, 'sk-ant-') === 0) {
                return 'anthropic';
            }
        }
        
        if (strpos($key, 'pplx-') === 0) {
            return 'perplexity';
        }
        
        if (strpos($key, 'sk-') === 0 || strpos($key, 'sk-proj-') === 0) {
            return 'openai';
        }
        
        return null; // Unknown key format
    }
    
    /**
     * Auto-fix model to match the provider
     */
    private function auto_fix_model_for_provider() {
        $provider_models = array(
            'anthropic' => 'claude-sonnet-4-20250514',
            'openai' => 'gpt-4-turbo-preview',
            'perplexity' => 'sonar',
            'walter' => get_option('aiab_walter_model', 'qwen2.5:32b-instruct-q4_K_M')
        );
        
        // Check if current model belongs to current provider
        $anthropic_models = array('claude-sonnet-4-20250514', 'claude-opus-4-20250514', 'claude-3-5-haiku-20241022');
        $openai_models = array('gpt-4-turbo-preview', 'gpt-4', 'gpt-3.5-turbo');
        $perplexity_models = array('sonar', 'sonar-pro', 'sonar-reasoning');
        // Walter models are dynamic, so we'll just use the stored model
        $walter_model = get_option('aiab_walter_model', 'qwen2.5:32b-instruct-q4_K_M');
        
        $model_matches_provider = false;
        if ($this->provider === 'anthropic' && in_array($this->model, $anthropic_models)) {
            $model_matches_provider = true;
        } elseif ($this->provider === 'openai' && in_array($this->model, $openai_models)) {
            $model_matches_provider = true;
        } elseif ($this->provider === 'perplexity' && in_array($this->model, $perplexity_models)) {
            $model_matches_provider = true;
        } elseif ($this->provider === 'walter') {
            // For Walter, the model is whatever is configured
            $model_matches_provider = true;
            $this->model = $walter_model;
        }
        
        if (!$model_matches_provider && isset($provider_models[$this->provider])) {
            $old_model = $this->model;
            $this->model = $provider_models[$this->provider];
            AIAB_Logger::warning("Auto-fixed model mismatch", array(
                'old_model' => $old_model,
                'new_model' => $this->model,
                'provider' => $this->provider
            ));
        }
    }
    
    /**
     * Try to find any valid API key if the selected one is empty
     */
    private function try_fallback_api_keys() {
        $keys_to_try = array(
            'openai' => get_option('aiab_openai_api_key', ''),
            'anthropic' => get_option('aiab_anthropic_api_key', ''),
            'perplexity' => get_option('aiab_perplexity_api_key', '')
        );
        
        foreach ($keys_to_try as $provider => $key) {
            if (!empty($key) && strlen(trim($key)) > 10) {
                AIAB_Logger::warning("Using fallback API key", array(
                    'original_provider' => $this->provider,
                    'fallback_provider' => $provider
                ));
                $this->api_key = $key;
                $this->provider = $provider;
                $this->auto_fix_model_for_provider();
                return;
            }
        }
    }
    
    /**
     * Generate a completion from the AI
     */
    public function generate_completion($prompt, $max_tokens = 4096) {
        if (empty($this->api_key)) {
            throw new Exception('AI API key not configured');
        }
        
        if ($this->provider === 'anthropic') {
            return $this->call_anthropic($prompt, $max_tokens);
        } elseif ($this->provider === 'perplexity') {
            return $this->call_perplexity($prompt, $max_tokens);
        } elseif ($this->provider === 'walter') {
            return $this->call_walter($prompt, $max_tokens);
        } else {
            return $this->call_openai($prompt, $max_tokens);
        }
    }
    
    /**
     * Call Anthropic Claude API
     */
    private function call_anthropic($prompt, $max_tokens) {
        $url = 'https://api.anthropic.com/v1/messages';
        
        // Validate API key first
        if (empty($this->api_key) || strlen(trim($this->api_key)) < 10) {
            AIAB_Logger::error("Anthropic API key is missing or invalid", array(
                'key_length' => strlen($this->api_key ?? ''),
                'key_preview' => !empty($this->api_key) ? substr($this->api_key, 0, 10) . '...' : 'EMPTY'
            ));
            throw new Exception('Anthropic API key is not configured. Please check Settings → AI Provider.');
        }
        
        $system_prompt = $this->persona->generate_system_prompt();
        $system_prompt .= "\n\n=== SEO CONTENT WRITING RULES (RankMath 100/100 Optimization) ===\n";
        $system_prompt .= "You are writing SEO-optimized content optimized for RankMath SEO plugin.\n\n";
        $system_prompt .= "TITLE RULES:\n";
        $system_prompt .= "- Create COMPLETE titles under 60 characters (no truncated phrases!)\n";
        $system_prompt .= "- Keyword at beginning of title (first 50%)\n";
        $system_prompt .= "- Include a number (7, 10, 5, 2024, etc.)\n";
        $system_prompt .= "- Include power word (Ultimate, Essential, Proven, Best, Top)\n";
        $system_prompt .= "- Include sentiment word (Amazing, Perfect, Avoid, Never, Mistakes)\n\n";
        $system_prompt .= "CONTENT RULES:\n";
        $system_prompt .= "- Keyword in first paragraph (first 10% of content)\n";
        $system_prompt .= "- Keyword in 2-3 H2 subheadings\n";
        $system_prompt .= "- Keyword density 1-1.5%\n";
        $system_prompt .= "- ALL paragraphs under 120 words\n";
        $system_prompt .= "- Use transition words throughout\n";
        $system_prompt .= "- Proper H2 > H3 hierarchy (no H1 in content, no H4+)\n";
        $system_prompt .= "- Include Table of Contents with anchor links\n";
        $system_prompt .= "- Target 2500+ words for pillar articles\n";
        $system_prompt .= "- ⚠️ NO citation markers [1][2][3], NO footnotes, NO bibliography sections\n\n";
        $system_prompt .= "META DESCRIPTION RULES:\n";
        $system_prompt .= "- Keyword in first 120 characters\n";
        $system_prompt .= "- Total length 150-160 characters\n";
        $system_prompt .= "- Include call-to-action\n";
        
        $request_body = array(
            'model' => $this->model,
            'max_tokens' => $max_tokens,
            'system' => $system_prompt,
            'messages' => array(
                array('role' => 'user', 'content' => $prompt)
            )
        );
        
        AIAB_Logger::debug("Anthropic API Request", array(
            'model' => $this->model,
            'max_tokens' => $max_tokens,
            'prompt_length' => strlen($prompt),
            'system_prompt_length' => strlen($system_prompt)
        ));
        
        $start_time = microtime(true);
        
        $response = wp_remote_post($url, array(
            'timeout' => 120,
            'headers' => array(
                'Content-Type' => 'application/json',
                'x-api-key' => $this->api_key,
                'anthropic-version' => '2023-06-01'
            ),
            'body' => json_encode($request_body)
        ));
        
        $duration = microtime(true) - $start_time;
        
        if (is_wp_error($response)) {
            AIAB_Logger::api_call('Anthropic', 'messages', $request_body, array(
                'error' => $response->get_error_message()
            ), $duration, false);
            throw new Exception('API request failed: ' . $response->get_error_message());
        }
        
        $status_code = wp_remote_retrieve_response_code($response);
        $raw_body = wp_remote_retrieve_body($response);
        $body = json_decode($raw_body, true);
        
        // Handle specific HTTP error codes with clear messages
        if ($status_code === 401) {
            AIAB_Logger::error("🔴 AUTHENTICATION FAILED - Invalid API Key", array(
                'status_code' => $status_code,
                'api_key_preview' => substr($this->api_key, 0, 15) . '...',
                'raw_response' => substr($raw_body, 0, 500)
            ));
            throw new Exception('Authentication failed: Your Anthropic API key is invalid. Please check Settings → AI Provider.');
        }
        
        if ($status_code === 403) {
            AIAB_Logger::error("🔴 ACCESS FORBIDDEN - API Key lacks permissions", array(
                'status_code' => $status_code,
                'raw_response' => substr($raw_body, 0, 500)
            ));
            throw new Exception('Access forbidden: Your API key may be disabled or lack permissions.');
        }
        
        if ($status_code === 429) {
            AIAB_Logger::error("🔴 RATE LIMITED - Too many requests", array(
                'status_code' => $status_code,
                'raw_response' => substr($raw_body, 0, 500)
            ));
            throw new Exception('Rate limited: Too many API requests. Please wait a few minutes.');
        }
        
        if ($status_code === 402) {
            $error_msg = isset($body['error']['message']) ? $body['error']['message'] : $raw_body;
            AIAB_Logger::error("🔴 ANTHROPIC PAYMENT REQUIRED - Budget/Credit Issue", array(
                'status_code' => $status_code,
                'error_message' => $error_msg
            ));
            
            // Send admin email notification
            $this->send_budget_alert_email('Anthropic', $error_msg);
            
            throw new Exception('Budget/Payment required: ' . $error_msg . ' - Admin has been notified.');
        }
        
        if ($status_code === 400) {
            $error_msg = isset($body['error']['message']) ? $body['error']['message'] : $raw_body;
            AIAB_Logger::error("🔴 API ERROR ({$status_code})", array(
                'status_code' => $status_code,
                'error_message' => $error_msg,
                'raw_response' => substr($raw_body, 0, 500)
            ));
            throw new Exception("API error ({$status_code}): {$error_msg}");
        }
        
        if (isset($body['error'])) {
            $error_msg = $body['error']['message'] ?? json_encode($body['error']);
            
            // Check for budget-related errors in message
            if ($this->is_budget_error($error_msg, $body['error']['type'] ?? '', '', $status_code)) {
                AIAB_Logger::error("🔴 ANTHROPIC BUDGET ERROR", array(
                    'status_code' => $status_code,
                    'error' => $body['error']
                ));
                $this->send_budget_alert_email('Anthropic', $error_msg);
                throw new Exception('Budget/Quota exceeded: ' . $error_msg . ' - Admin has been notified.');
            }
            
            AIAB_Logger::api_call('Anthropic', 'messages', $request_body, array(
                'status_code' => $status_code,
                'error' => $body['error']
            ), $duration, false);
            throw new Exception('API error: ' . $error_msg);
        }
        
        if (isset($body['content'][0]['text'])) {
            $result_text = $body['content'][0]['text'];
            $stop_reason = $body['stop_reason'] ?? 'unknown';
            $usage = $body['usage'] ?? array();
            
            // Log comprehensive info including stop reason
            AIAB_Logger::api_call('Anthropic', 'messages', array(
                'model' => $this->model,
                'prompt_preview' => substr($prompt, 0, 200) . '...'
            ), array(
                'status_code' => $status_code,
                'response_length' => strlen($result_text),
                'stop_reason' => $stop_reason,
                'usage' => $usage
            ), $duration, true);
            
            // WARN if response was cut off due to max_tokens
            if ($stop_reason === 'max_tokens') {
                AIAB_Logger::warning("⚠️ API response truncated due to max_tokens limit!", array(
                    'input_tokens' => $usage['input_tokens'] ?? 0,
                    'output_tokens' => $usage['output_tokens'] ?? 0,
                    'max_tokens' => $max_tokens,
                    'response_preview' => substr($result_text, -500) // Show end of response
                ));
            }
            
            // Log response preview for debugging content parsing issues
            AIAB_Logger::debug("API Response Preview", array(
                'first_500_chars' => substr($result_text, 0, 500),
                'last_200_chars' => substr($result_text, -200),
                'has_title_marker' => strpos($result_text, '---TITLE---') !== false,
                'has_content_marker' => strpos($result_text, '---CONTENT---') !== false
            ));
            
            return $result_text;
        }
        
        // Log the actual response body for debugging (raw_body already captured above)
        AIAB_Logger::error("Anthropic API returned unexpected format", array(
            'status_code' => $status_code,
            'raw_body_preview' => substr($raw_body, 0, 1000),
            'body_type' => gettype($body),
            'body_keys' => is_array($body) ? array_keys($body) : 'not_array',
            'has_content' => isset($body['content']),
            'content_value' => isset($body['content']) ? json_encode($body['content']) : 'not_set'
        ));
        
        AIAB_Logger::api_call('Anthropic', 'messages', $request_body, array(
            'status_code' => $status_code,
            'body' => $body
        ), $duration, false);
        
        throw new Exception('Unexpected API response format - check logs for details');
    }
    
    /**
     * Call OpenAI API
     */
    private function call_openai($prompt, $max_tokens) {
        $url = 'https://api.openai.com/v1/chat/completions';
        
        // Validate API key first
        if (empty($this->api_key) || strlen(trim($this->api_key)) < 10) {
            AIAB_Logger::error("OpenAI API key is missing or invalid", array(
                'key_length' => strlen($this->api_key ?? ''),
                'key_preview' => !empty($this->api_key) ? substr($this->api_key, 0, 10) . '...' : 'EMPTY'
            ));
            throw new Exception('OpenAI API key is not configured. Please check Settings → AI Provider.');
        }
        
        $system_prompt = $this->persona->generate_system_prompt();
        $system_prompt .= "\n\n=== SEO CONTENT WRITING RULES (RankMath 100/100 Optimization) ===\n";
        $system_prompt .= "You are writing SEO-optimized content optimized for RankMath SEO plugin.\n";
        $system_prompt .= "- Create COMPLETE titles under 60 characters (no truncated phrases!)\n";
        $system_prompt .= "- Keyword at beginning of title and in first paragraph\n";
        $system_prompt .= "- Include numbers and power words in title\n";
        $system_prompt .= "- ALL paragraphs under 120 words\n";
        $system_prompt .= "- Keyword density 1-1.5%, keyword in H2 subheadings\n";
        $system_prompt .= "- Proper H2 > H3 hierarchy, include Table of Contents\n";
        $system_prompt .= "- Meta description: keyword in first 120 chars, 150-160 total chars\n";
        $system_prompt .= "- NO citation markers [1][2][3], NO footnotes, NO bibliography sections\n";
        
        $request_body = array(
            'model' => $this->model,
            'max_tokens' => $max_tokens,
            'messages' => array(
                array('role' => 'system', 'content' => $system_prompt),
                array('role' => 'user', 'content' => $prompt)
            )
        );
        
        AIAB_Logger::debug("OpenAI API Request", array(
            'model' => $this->model,
            'max_tokens' => $max_tokens,
            'prompt_length' => strlen($prompt)
        ));
        
        $max_retries = 3;
        $retry_delay = 5; // Start with 5 seconds
        
        for ($attempt = 1; $attempt <= $max_retries; $attempt++) {
            $start_time = microtime(true);
            
            $response = wp_remote_post($url, array(
                'timeout' => 180, // Longer timeout for large content
                'headers' => array(
                    'Content-Type' => 'application/json',
                    'Authorization' => 'Bearer ' . $this->api_key
                ),
                'body' => json_encode($request_body)
            ));
            
            $duration = microtime(true) - $start_time;
            
            if (is_wp_error($response)) {
                AIAB_Logger::error("OpenAI request failed", array(
                    'error' => $response->get_error_message(),
                    'attempt' => $attempt
                ));
                throw new Exception('API request failed: ' . $response->get_error_message());
            }
            
            $status_code = wp_remote_retrieve_response_code($response);
            $raw_body = wp_remote_retrieve_body($response);
            $body = json_decode($raw_body, true);
            
            // Handle specific HTTP error codes
            if ($status_code === 401) {
                AIAB_Logger::error("🔴 OPENAI AUTH FAILED - Invalid API Key", array(
                    'status_code' => $status_code,
                    'api_key_preview' => substr($this->api_key, 0, 15) . '...'
                ));
                throw new Exception('Authentication failed: Your OpenAI API key is invalid. Please check Settings → AI Provider.');
            }
            
            if (isset($body['error'])) {
                $error_message = $body['error']['message'] ?? 'Unknown error';
                $error_type = $body['error']['type'] ?? '';
                $error_code = $body['error']['code'] ?? '';
                
                // Check for BUDGET/QUOTA errors - these need admin notification
                $is_budget_error = $this->is_budget_error($error_message, $error_type, $error_code, $status_code);
                
                if ($is_budget_error) {
                    AIAB_Logger::error("🔴 OPENAI BUDGET/QUOTA ERROR", array(
                        'status_code' => $status_code,
                        'error_type' => $error_type,
                        'error_code' => $error_code,
                        'error_message' => $error_message
                    ));
                    
                    // Send admin email notification
                    $this->send_budget_alert_email('OpenAI', $error_message);
                    
                    throw new Exception('Budget/Quota exceeded: ' . $error_message . ' - Admin has been notified.');
                }
                
                // Check for rate limit error (can retry)
                if (strpos($error_message, 'Rate limit') !== false || 
                    strpos($error_message, 'rate_limit') !== false ||
                    strpos($error_message, 'TPM') !== false ||
                    $status_code === 429) {
                    
                    if ($attempt < $max_retries) {
                        AIAB_Logger::warning("Rate limit hit, waiting {$retry_delay}s before retry {$attempt}/{$max_retries}");
                        sleep($retry_delay);
                        $retry_delay *= 2; // Exponential backoff
                        continue;
                    }
                }
                
                AIAB_Logger::error("OpenAI API error", array(
                    'status_code' => $status_code,
                    'error_type' => $error_type,
                    'error_message' => $error_message
                ));
                
                throw new Exception('OpenAI API error: ' . $error_message);
            }
            
            if (isset($body['choices'][0]['message']['content'])) {
                AIAB_Logger::info("OpenAI API success", array(
                    'duration' => round($duration, 2) . 's',
                    'response_length' => strlen($body['choices'][0]['message']['content'])
                ));
                return $body['choices'][0]['message']['content'];
            }
            
            // Log unexpected response format with details
            AIAB_Logger::error("OpenAI API returned unexpected format", array(
                'status_code' => $status_code,
                'raw_body_preview' => substr($raw_body, 0, 1000),
                'body_keys' => is_array($body) ? array_keys($body) : 'not_array'
            ));
            
            throw new Exception('Unexpected OpenAI API response format - check logs for details');
        }
        
        throw new Exception('API request failed after ' . $max_retries . ' retries');
    }
    
    /**
     * Check if error is budget/quota related
     */
    private function is_budget_error($message, $type, $code, $status_code) {
        $message_lower = strtolower($message);
        $type_lower = strtolower($type);
        $code_lower = strtolower($code);
        
        // Budget/quota keywords
        $budget_keywords = array(
            'quota', 'exceeded', 'billing', 'payment', 'insufficient',
            'budget', 'credit', 'balance', 'limit exceeded', 'usage limit',
            'spending', 'allowance', 'plan limit', 'subscription'
        );
        
        foreach ($budget_keywords as $keyword) {
            if (strpos($message_lower, $keyword) !== false) {
                return true;
            }
        }
        
        // Specific error codes
        $budget_codes = array(
            'insufficient_quota',
            'billing_hard_limit_reached',
            'budget_exceeded',
            'rate_limit_exceeded' // Sometimes indicates quota
        );
        
        if (in_array($code_lower, $budget_codes) || in_array($type_lower, $budget_codes)) {
            return true;
        }
        
        // HTTP 402 Payment Required
        if ($status_code === 402) {
            return true;
        }
        
        return false;
    }
    
    /**
     * Send email alert to admin when budget is exceeded
     */
    private function send_budget_alert_email($provider, $error_message) {
        // Check if we've already sent an email recently (within 1 hour)
        $last_sent = get_option('aiab_last_budget_email', 0);
        if ((time() - $last_sent) < 3600) {
            AIAB_Logger::debug("Budget alert email skipped - sent less than 1 hour ago");
            return;
        }
        
        $admin_email = get_option('admin_email');
        $site_name = get_bloginfo('name');
        $site_url = get_site_url();
        
        $subject = "⚠️ [{$site_name}] Eternal Auto Blogger - {$provider} Budget Alert";
        
        $message = "Hello,\n\n";
        $message .= "The Eternal Auto Blogger plugin on your site has encountered a budget/quota error with {$provider}.\n\n";
        $message .= "Error Details:\n";
        $message .= "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
        $message .= $error_message . "\n";
        $message .= "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";
        $message .= "What This Means:\n";
        $message .= "• Your {$provider} API budget has been exceeded or your account needs attention\n";
        $message .= "• The autoblogger will NOT be able to generate new content until this is resolved\n";
        $message .= "• Articles will be marked as 'auth_failed' and will not auto-retry\n\n";
        $message .= "What To Do:\n";
        $message .= "1. Log in to your {$provider} account and check your billing/usage\n";
        $message .= "2. Add credits or upgrade your plan if needed\n";
        $message .= "3. Go to WordPress Admin → Eternal Blogger → Settings to verify your API key\n";
        $message .= "4. Click 'Reset Failed Articles' on the Dashboard to retry failed articles\n\n";
        $message .= "Site: {$site_url}\n";
        $message .= "Time: " . current_time('mysql') . "\n\n";
        $message .= "This is an automated message from the Eternal Auto Blogger plugin.\n";
        $message .= "You will not receive another alert for at least 1 hour.\n";
        
        $headers = array('Content-Type: text/plain; charset=UTF-8');
        
        $sent = wp_mail($admin_email, $subject, $message, $headers);
        
        if ($sent) {
            update_option('aiab_last_budget_email', time());
            AIAB_Logger::warning("📧 Budget alert email sent to admin: {$admin_email}");
        } else {
            AIAB_Logger::error("Failed to send budget alert email to: {$admin_email}");
        }
    }
    
    /**
     * Call Perplexity API (OpenAI-compatible with higher token limits)
     */
    private function call_perplexity($prompt, $max_tokens) {
        $url = 'https://api.perplexity.ai/chat/completions';
        
        // Validate API key first
        if (empty($this->api_key) || strlen(trim($this->api_key)) < 10) {
            AIAB_Logger::error("Perplexity API key is missing or invalid", array(
                'key_length' => strlen($this->api_key ?? ''),
                'key_preview' => !empty($this->api_key) ? substr($this->api_key, 0, 10) . '...' : 'EMPTY'
            ));
            throw new Exception('Perplexity API key is not configured. Please check Settings → AI Provider.');
        }
        
        $system_prompt = $this->persona->generate_system_prompt();
        $system_prompt .= "\n\n=== SEO CONTENT WRITING RULES (RankMath 100/100 Optimization) ===\n";
        $system_prompt .= "You are writing SEO-optimized content optimized for RankMath SEO plugin.\n";
        $system_prompt .= "- Create COMPLETE titles under 60 characters (no truncated phrases!)\n";
        $system_prompt .= "- Keyword at beginning of title and in first paragraph\n";
        $system_prompt .= "- Include numbers and power words in title\n";
        $system_prompt .= "- ALL paragraphs under 120 words\n";
        $system_prompt .= "- Keyword density 1-1.5%, keyword in H2 subheadings\n";
        $system_prompt .= "- Proper H2 > H3 hierarchy, include Table of Contents\n";
        $system_prompt .= "- Meta description: keyword in first 120 chars, 150-160 total chars\n";
        $system_prompt .= "- Write comprehensive, detailed content - aim for the full word count requested\n";
        $system_prompt .= "- NO citation markers [1][2][3], NO footnotes, NO bibliography sections\n";
        
        $request_body = array(
            'model' => $this->model,
            'max_tokens' => $max_tokens,
            'messages' => array(
                array('role' => 'system', 'content' => $system_prompt),
                array('role' => 'user', 'content' => $prompt)
            )
        );
        
        AIAB_Logger::debug("Perplexity API Request", array(
            'model' => $this->model,
            'max_tokens' => $max_tokens,
            'prompt_length' => strlen($prompt)
        ));
        
        $max_retries = 3;
        $retry_delay = 5;
        $start_time = microtime(true);
        
        for ($attempt = 1; $attempt <= $max_retries; $attempt++) {
            $response = wp_remote_post($url, array(
                'timeout' => 180,
                'headers' => array(
                    'Content-Type' => 'application/json',
                    'Authorization' => 'Bearer ' . $this->api_key
                ),
                'body' => json_encode($request_body)
            ));
            
            $duration = microtime(true) - $start_time;
            
            if (is_wp_error($response)) {
                AIAB_Logger::error("Perplexity request failed", array(
                    'error' => $response->get_error_message(),
                    'attempt' => $attempt
                ));
                throw new Exception('API request failed: ' . $response->get_error_message());
            }
            
            $status_code = wp_remote_retrieve_response_code($response);
            $raw_body = wp_remote_retrieve_body($response);
            $body = json_decode($raw_body, true);
            
            // Handle specific HTTP error codes
            if ($status_code === 401) {
                AIAB_Logger::error("🔴 PERPLEXITY AUTH FAILED - Invalid API Key", array(
                    'status_code' => $status_code,
                    'api_key_preview' => substr($this->api_key, 0, 15) . '...',
                    'raw_response' => substr($raw_body, 0, 500)
                ));
                throw new Exception('Authentication failed: Your Perplexity API key is invalid. Please check Settings → AI Provider.');
            }
            
            if ($status_code === 403) {
                AIAB_Logger::error("🔴 PERPLEXITY ACCESS FORBIDDEN", array(
                    'status_code' => $status_code,
                    'raw_response' => substr($raw_body, 0, 500)
                ));
                throw new Exception('Access forbidden: Your Perplexity API key may be disabled.');
            }
            
            if ($status_code === 429) {
                AIAB_Logger::warning("Perplexity rate limit hit", array(
                    'attempt' => $attempt,
                    'will_retry' => $attempt < $max_retries
                ));
                if ($attempt < $max_retries) {
                    sleep($retry_delay);
                    $retry_delay *= 2;
                    continue;
                }
                throw new Exception('Rate limited: Too many Perplexity API requests.');
            }
            
            // Check for payment/budget errors
            if ($status_code === 402) {
                $error_msg = isset($body['error']['message']) ? $body['error']['message'] : $raw_body;
                AIAB_Logger::error("🔴 PERPLEXITY PAYMENT REQUIRED - Budget/Credit Issue", array(
                    'status_code' => $status_code,
                    'error_message' => $error_msg
                ));
                $this->send_budget_alert_email('Perplexity', $error_msg);
                throw new Exception('Budget/Payment required: ' . $error_msg . ' - Admin has been notified.');
            }
            
            if (isset($body['error'])) {
                $error_message = $body['error']['message'] ?? (is_string($body['error']) ? $body['error'] : json_encode($body['error']));
                
                // Check for budget-related errors
                if ($this->is_budget_error($error_message, $body['error']['type'] ?? '', $body['error']['code'] ?? '', $status_code)) {
                    AIAB_Logger::error("🔴 PERPLEXITY BUDGET ERROR", array(
                        'status_code' => $status_code,
                        'error' => $error_message
                    ));
                    $this->send_budget_alert_email('Perplexity', $error_message);
                    throw new Exception('Budget/Quota exceeded: ' . $error_message . ' - Admin has been notified.');
                }
                
                AIAB_Logger::error("Perplexity API error", array(
                    'status_code' => $status_code,
                    'error' => $error_message,
                    'attempt' => $attempt
                ));
                
                // Check for rate limit error
                if (strpos(strtolower($error_message), 'rate') !== false || 
                    strpos(strtolower($error_message), 'limit') !== false) {
                    
                    if ($attempt < $max_retries) {
                        AIAB_Logger::warning("Rate limit hit, waiting {$retry_delay}s before retry {$attempt}/{$max_retries}");
                        sleep($retry_delay);
                        $retry_delay *= 2;
                        continue;
                    }
                }
                
                throw new Exception('Perplexity API error: ' . $error_message);
            }
            
            if (isset($body['choices'][0]['message']['content'])) {
                AIAB_Logger::info("Perplexity API success", array(
                    'duration' => round($duration, 2) . 's',
                    'response_length' => strlen($body['choices'][0]['message']['content'])
                ));
                return $body['choices'][0]['message']['content'];
            }
            
            // Log unexpected response format with details
            AIAB_Logger::error("Perplexity API returned unexpected format", array(
                'status_code' => $status_code,
                'raw_body_preview' => substr($raw_body, 0, 1000),
                'body_keys' => is_array($body) ? array_keys($body) : 'not_array',
                'has_choices' => isset($body['choices']),
                'attempt' => $attempt
            ));
            
            throw new Exception('Unexpected Perplexity API response - check logs for details');
        }
        
        throw new Exception('Perplexity API request failed after ' . $max_retries . ' retries');
    }
    
    /**
     * Call Custom Server (Ollama) API - Your local AI server!
     * Uses NATIVE Ollama /api/chat endpoint (supports num_ctx properly)
     */
    private function call_walter($prompt, $max_tokens) {
        $base_url = rtrim(get_option('aiab_walter_url', 'http://localhost:11434'), '/');
        $url = $base_url . '/api/chat'; // Native Ollama endpoint (supports options properly)
        
        // Validate server URL
        if (empty($base_url) || $base_url === 'http://localhost:11434') {
            // Check if default is intended
            $configured_url = get_option('aiab_walter_url', '');
            if (empty($configured_url)) {
                AIAB_Logger::error("Custom Server URL not configured");
                throw new Exception('Custom Server URL is not configured. Please check Settings → AI Provider.');
            }
        }
        
        $model = get_option('aiab_walter_model', 'qwen2.5:32b-instruct-q4_K_M');
        
        $system_prompt = $this->persona->generate_system_prompt();
        $system_prompt .= "\n\n=== SEO CONTENT WRITING RULES (RankMath 100/100 Optimization) ===\n";
        $system_prompt .= "You are writing SEO-optimized content optimized for RankMath SEO plugin.\n";
        $system_prompt .= "- Create COMPLETE titles under 60 characters (no truncated phrases!)\n";
        $system_prompt .= "- Keyword at beginning of title and in first paragraph\n";
        $system_prompt .= "- Include numbers and power words in title\n";
        $system_prompt .= "- ALL paragraphs under 120 words\n";
        $system_prompt .= "- Keyword density 1-1.5%, keyword in H2 subheadings\n";
        $system_prompt .= "- Proper H2 > H3 hierarchy, include Table of Contents\n";
        $system_prompt .= "- Meta description: keyword in first 120 chars, 150-160 total chars\n";
        $system_prompt .= "- When data comparison is needed, use HTML tables with proper <table>, <thead>, <tbody>, <tr>, <th>, <td> tags\n";
        $system_prompt .= "- CRITICAL: Write LONG, comprehensive content - MINIMUM word count is specified in the prompt\n";
        $system_prompt .= "- Include 8-12 H2 sections for pillar articles, 6-8 for clusters\n";
        $system_prompt .= "- Each section should be 200-350 words with detailed explanations\n";
        $system_prompt .= "- DO NOT summarize or cut short - write the FULL article length requested\n";
        $system_prompt .= "- NO citation markers [1][2][3], NO footnotes, NO bibliography sections\n";
        
        // Native Ollama API format
        $request_body = array(
            'model' => $model,
            'messages' => array(
                array('role' => 'system', 'content' => $system_prompt),
                array('role' => 'user', 'content' => $prompt)
            ),
            'stream' => false,
            'options' => array(
                'num_predict' => $max_tokens,  // Max output tokens
                'num_ctx' => 32768,            // Context window - 32K for long articles
                'temperature' => 0.7,
                'top_p' => 0.9
            )
        );
        
        AIAB_Logger::debug("Custom Server (Ollama) API Request", array(
            'url' => $url,
            'model' => $model,
            'max_tokens' => $max_tokens,
            'num_ctx' => 32768,
            'prompt_length' => strlen($prompt)
        ));
        
        $max_retries = 3;
        $retry_delay = 10; // Longer delay for local inference
        $start_time = microtime(true);
        
        for ($attempt = 1; $attempt <= $max_retries; $attempt++) {
            $response = wp_remote_post($url, array(
                'timeout' => 1200, // 20 minutes - local CPU inference needs headroom
                'headers' => array(
                    'Content-Type' => 'application/json'
                ),
                'body' => json_encode($request_body)
            ));
            
            $duration = microtime(true) - $start_time;
            
            if (is_wp_error($response)) {
                $error_message = $response->get_error_message();
                AIAB_Logger::error("Custom Server request failed", array(
                    'error' => $error_message,
                    'attempt' => $attempt,
                    'url' => $url
                ));
                
                // Check if it's a connection error
                if (strpos($error_message, 'cURL error 7') !== false || 
                    strpos($error_message, 'Connection refused') !== false ||
                    strpos($error_message, 'couldn\'t connect') !== false) {
                    throw new Exception('Cannot connect to Custom Server at ' . $base_url . '. Is Ollama running?');
                }
                
                if ($attempt < $max_retries) {
                    AIAB_Logger::warning("Custom Server connection failed, waiting {$retry_delay}s before retry {$attempt}/{$max_retries}");
                    sleep($retry_delay);
                    continue;
                }
                
                throw new Exception('Custom Server API request failed: ' . $error_message);
            }
            
            $status_code = wp_remote_retrieve_response_code($response);
            $raw_body = wp_remote_retrieve_body($response);
            $body = json_decode($raw_body, true);
            
            // Handle specific HTTP error codes
            if ($status_code === 404) {
                AIAB_Logger::error("Custom Server model not found", array(
                    'status_code' => $status_code,
                    'model' => $model
                ));
                throw new Exception("Model '{$model}' not found on Custom Server. Run: ollama pull {$model}");
            }
            
            if ($status_code === 500 || $status_code === 503) {
                AIAB_Logger::warning("Custom Server error", array(
                    'status_code' => $status_code,
                    'attempt' => $attempt
                ));
                if ($attempt < $max_retries) {
                    sleep($retry_delay);
                    continue;
                }
                throw new Exception('Custom Server error. Is Ollama running and healthy?');
            }
            
            if (isset($body['error'])) {
                $error_message = is_string($body['error']) ? $body['error'] : ($body['error']['message'] ?? json_encode($body['error']));
                AIAB_Logger::error("Custom Server API error", array(
                    'status_code' => $status_code,
                    'error' => $error_message,
                    'attempt' => $attempt
                ));
                throw new Exception('Custom Server API error: ' . $error_message);
            }
            
            // OpenAI-compatible response format
            if (isset($body['choices'][0]['message']['content'])) {
                AIAB_Logger::info("Custom Server API success", array(
                    'duration' => round($duration, 2) . 's',
                    'response_length' => strlen($body['choices'][0]['message']['content']),
                    'model' => $model
                ));
                return $body['choices'][0]['message']['content'];
            }
            
            // Try Ollama native format as fallback
            if (isset($body['message']['content'])) {
                $content = $body['message']['content'];
                
                // Strip DeepSeek R1 thinking tags - these contain internal reasoning, not content
                $content = preg_replace('/<think>.*?<\/think>/s', '', $content);
                $content = preg_replace('/<thinking>.*?<\/thinking>/s', '', $content);
                $content = trim($content);
                
                AIAB_Logger::info("Custom Server API success (native format)", array(
                    'duration' => round($duration, 2) . 's',
                    'response_length' => strlen($content),
                    'model' => $model
                ));
                return $content;
            }
            
            // Log unexpected response format
            AIAB_Logger::error("Custom Server API returned unexpected format", array(
                'status_code' => $status_code,
                'raw_body_preview' => substr($raw_body, 0, 1000),
                'body_keys' => is_array($body) ? array_keys($body) : 'not_array',
                'attempt' => $attempt
            ));
            
            throw new Exception('Unexpected Custom Server API response - check logs for details');
        }
        
        throw new Exception('Custom Server API request failed after ' . $max_retries . ' retries');
    }
    
    /**
     * Write a complete article
     */
    public function write_article($keyword, $article_type, $is_pillar = false, $context = array()) {
        AIAB_Logger::log("Writing article: $keyword (type: $article_type, pillar: " . ($is_pillar ? 'yes' : 'no') . ")");
        
        $word_count = $is_pillar 
            ? get_option('aiab_pillar_word_count', 2500)
            : get_option('aiab_cluster_word_count', 1500);
        
        $prompt = $this->build_article_prompt($keyword, $article_type, $is_pillar, $word_count, $context);
        
        // Calculate max tokens based on target word count
        // With HTML formatting, we need roughly 1.5 tokens per word + overhead for formatting
        // Pillar articles need more room for comprehensive content
        $base_tokens = intval(get_option('aiab_max_tokens', 4096));
        
        if ($is_pillar) {
            // Pillar articles: 2500 words ≈ need 8000 tokens with HTML
            $max_tokens = max($base_tokens, 8192);
        } else {
            // Cluster articles: 1500 words ≈ need 5000 tokens with HTML
            $max_tokens = max($base_tokens, 6000);
        }
        
        // Provider-specific token limits
        if ($this->provider === 'anthropic') {
            // Claude supports up to 8192 output tokens
            $max_tokens = min($max_tokens, 8192);
        } elseif ($this->provider === 'perplexity') {
            // Perplexity models support very high token limits (up to 127k)
            // We'll use 8192 to match our needs without being excessive
            $max_tokens = min($max_tokens, 8192);
        } elseif ($this->provider === 'walter') {
            // Walter (Ollama) - balance between quality and memory usage
            // 32GB RAM with ~18GB model leaves limited headroom
            if ($is_pillar) {
                $max_tokens = 8192; // ~2000 words with HTML - good for pillar
            } else {
                $max_tokens = 6000; // ~1500 words with HTML - good for cluster
            }
            AIAB_Logger::debug("Custom Server mode: Using optimized token limit for 32GB RAM", array(
                'max_tokens' => $max_tokens,
                'target_words' => $word_count
            ));
        } else {
            // OpenAI GPT-4 models typically max at 4096 output tokens
            // For longer content with OpenAI, we'll request max and let it complete naturally
            $max_tokens = min($max_tokens, 4096);
            
            if ($is_pillar && $word_count > 1500) {
                AIAB_Logger::warning("OpenAI model limited to 4096 tokens - pillar may be shorter than target. Consider using Anthropic Claude or Perplexity for longer articles.");
            }
        }
        
        AIAB_Logger::debug("Generating article content", array(
            'keyword' => $keyword,
            'article_type' => $article_type,
            'is_pillar' => $is_pillar,
            'target_words' => $word_count,
            'max_tokens' => $max_tokens,
            'provider' => $this->provider
        ));
        
        // For Walter, we may need to retry if word count is too low
        $max_attempts = ($this->provider === 'walter') ? 2 : 1;
        $min_word_threshold = intval($word_count * 0.6); // At least 60% of target
        
        for ($attempt = 1; $attempt <= $max_attempts; $attempt++) {
            $content = $this->generate_completion($prompt, $max_tokens);
            
            // Parse the response (now WITHOUT final formatting - just extraction)
            $article_data = $this->parse_article_response($content, $keyword);
            
            // Check word count
            $actual_words = str_word_count(strip_tags($article_data['content']));
            
            if ($actual_words >= $min_word_threshold || $attempt >= $max_attempts) {
                if ($actual_words < $min_word_threshold) {
                    AIAB_Logger::warning("Article word count below threshold but max attempts reached", array(
                        'actual_words' => $actual_words,
                        'target_words' => $word_count,
                        'min_threshold' => $min_word_threshold
                    ));
                }
                break;
            }
            
            AIAB_Logger::info("Article too short, retrying with stronger word count emphasis", array(
                'actual_words' => $actual_words,
                'target_words' => $word_count,
                'attempt' => $attempt
            ));
            
            // Strengthen the prompt for retry
            $prompt = $this->build_article_prompt($keyword, $article_type, $is_pillar, $word_count, $context, true);
        }
        
        // ═══ ENFORCE KEYWORD DENSITY FOR RANKMATH 100/100 ═══
        // This ensures the exact focus keyword appears with proper density
        // MUST run BEFORE final formatting to avoid breaking HTML
        $this->enforce_keyword_density($article_data);
        
        // Also ensure title contains the keyword
        $article_data = $this->ensure_keyword_in_title($article_data);
        
        // ═══ FINAL HTML CLEANUP ═══
        // Fix any broken HTML tags from processing
        $article_data['content'] = $this->cleanup_html($article_data['content']);
        
        // Regenerate TOC after all modifications
        $article_data['content'] = $this->ensure_toc_exists($article_data['content']);
        
        return $article_data;
    }
    
    /**
     * Cleanup any broken HTML tags and ensure proper structure
     */
    private function cleanup_html($content) {
        // ═══ STRIP CITATION MARKERS ═══
        // Remove academic-style citation markers like [1], [2], [1][2], etc.
        $content = preg_replace('/\[\d+\]/', '', $content);
        // Remove multiple consecutive citation markers that might have spaces
        $content = preg_replace('/\[\d+\]\s*\[\d+\]/', '', $content);
        // Remove footnote-style markers like [*], [†], etc.
        $content = preg_replace('/\[[*†‡§]\]/', '', $content);
        // Remove "References" or "Bibliography" sections at the end
        $content = preg_replace('/<h[23][^>]*>\s*(References|Bibliography|Sources|Citations|Works Cited)\s*<\/h[23]>.*$/is', '', $content);
        // Clean up any leftover empty brackets
        $content = preg_replace('/\[\s*\]/', '', $content);
        
        // Fix any escaped HTML entities that shouldn't be escaped
        $content = str_replace('&lt;h2', '<h2', $content);
        $content = str_replace('&lt;/h2&gt;', '</h2>', $content);
        $content = str_replace('&lt;h3', '<h3', $content);
        $content = str_replace('&lt;/h3&gt;', '</h3>', $content);
        $content = str_replace('&lt;p&gt;', '<p>', $content);
        $content = str_replace('&lt;/p&gt;', '</p>', $content);
        $content = str_replace('&lt;strong&gt;', '<strong>', $content);
        $content = str_replace('&lt;/strong&gt;', '</strong>', $content);
        
        // Remove any raw HTML tag text that's showing (shouldn't happen but safety check)
        // This catches cases like "text</h2" showing in output
        $content = preg_replace('/<\/h2>(?=[^<]|$)/', '</h2>', $content);
        
        // Ensure proper H2 structure - fix broken tags
        $content = preg_replace('/<h2([^>]*)>([^<]*)<\/h2>/i', '<h2$1>$2</h2>', $content);
        
        // Remove duplicate closing tags
        $content = preg_replace('/<\/h2>\s*<\/h2>/', '</h2>', $content);
        $content = preg_replace('/<\/p>\s*<\/p>/', '</p>', $content);
        
        // Ensure H2 tags that got split are fixed
        // Pattern: text</h2 (missing closing bracket before </h2)
        $content = preg_replace('/([^>])<\/h2>/', '$1</h2>', $content);
        
        // Fix any H2 without proper closing
        $content = preg_replace('/<h2([^>]*)>([^<]+)(?!<\/h2>)(<[^h])/i', '<h2$1>$2</h2>$3', $content);
        
        return $content;
    }
    
    /**
     * Ensure TOC exists after all content modifications
     */
    private function ensure_toc_exists($content) {
        // Check if TOC already exists
        $has_toc = strpos($content, 'class="toc"') !== false 
                || strpos($content, 'table-of-contents') !== false
                || strpos($content, 'ez-toc') !== false
                || strpos($content, 'wp-block-table-of-contents') !== false
                || preg_match('/<nav[^>]*>.*?Table of Contents/is', $content);
        
        if ($has_toc) {
            return $content;
        }
        
        // Ensure H2 headings have IDs first
        $content = $this->add_heading_ids($content);
        
        // Generate and insert TOC
        $content = $this->generate_toc($content);
        
        return $content;
    }
    
    /**
     * Ensure the focus keyword appears at the BEGINNING of the title
     * 
     * CRITICAL FOR RANKMATH: Title must START with the focus keyword
     * RankMath checks: "Focus Keyword doesn't appear at the beginning of SEO title"
     * This means position 0, not just "first half"
     */
    private function ensure_keyword_in_title($article_data) {
        $keyword = isset($article_data['focus_keyword']) ? $article_data['focus_keyword'] : 
                   (isset($article_data['keyword']) ? $article_data['keyword'] : '');
        
        if (empty($keyword) || empty($article_data['title'])) {
            return $article_data;
        }
        
        $title = $article_data['title'];
        $keyword_lower = strtolower($keyword);
        $title_lower = strtolower($title);
        
        // Check if keyword is at the VERY BEGINNING of title (position 0)
        if (strpos($title_lower, $keyword_lower) === 0) {
            // Perfect - keyword is at the start
            AIAB_Logger::debug("Keyword already at beginning of title", array(
                'keyword' => $keyword,
                'title' => $title
            ));
            return $article_data;
        }
        
        // Keyword is NOT at the beginning - we MUST fix this
        $ucKeyword = ucwords($keyword);
        
        // Check if keyword exists elsewhere in title
        $keyword_pos = strpos($title_lower, $keyword_lower);
        
        if ($keyword_pos !== false) {
            // Keyword exists but not at start - restructure to move it to front
            // Remove keyword from current position
            $title_without_keyword = preg_replace('/\b' . preg_quote($keyword, '/') . '\b/i', '', $title);
            $title_without_keyword = trim(preg_replace('/\s+/', ' ', $title_without_keyword));
            $title_without_keyword = trim($title_without_keyword, ':- ');
            
            // Prepend keyword
            $new_title = "{$ucKeyword}: {$title_without_keyword}";
        } else {
            // Keyword not in title at all - prepend it
            $new_title = "{$ucKeyword}: {$title}";
        }
        
        // Ensure title is not too long (under 60 chars for SEO)
        if (strlen($new_title) > 60) {
            // Keep the keyword, truncate the rest
            $max_rest = 60 - strlen($ucKeyword) - 2; // 2 for ": "
            $rest = substr($new_title, strlen($ucKeyword) + 2);
            
            if ($max_rest > 10) {
                $rest = substr($rest, 0, $max_rest);
                // Don't cut in middle of word
                $rest = preg_replace('/\s+\S*$/', '', $rest);
                $new_title = "{$ucKeyword}: {$rest}";
            } else {
                // Keyword itself is too long, just use it
                $new_title = $ucKeyword;
            }
        }
        
        AIAB_Logger::info("✅ Fixed title to start with keyword", array(
            'original' => $title,
            'new_title' => $new_title,
            'keyword' => $keyword
        ));
        
        $article_data['title'] = $new_title;
        return $article_data;
    }
    
    /**
     * Build the article generation prompt
     */
    private function build_article_prompt($keyword, $article_type, $is_pillar, $word_count, $context, $emphasize_length = false) {
        // Budget mode uses a shorter, more concise prompt (saves ~1500 input tokens per article)
        if (get_option('aiab_budget_mode', 0)) {
            return $this->build_budget_article_prompt($keyword, $article_type, $is_pillar, $word_count, $context);
        }
        
        $prompt = "Write a comprehensive, well-researched article about: \"{$keyword}\"\n\n";
        
        $prompt .= "ARTICLE SPECIFICATIONS:\n";
        $prompt .= "- Article Type: " . $this->get_type_description($article_type) . "\n";
        $prompt .= "- Target Word Count: {$word_count} words (write a thorough, detailed article)\n";
        $prompt .= "- Main Topic: {$keyword}\n";
        
        if ($is_pillar) {
            $prompt .= "- This is a COMPREHENSIVE GUIDE - make it thorough and authoritative\n";
            $prompt .= "- Cover all major aspects of the topic in depth\n";
            $prompt .= "- This will serve as a main reference article\n";
        } else {
            $prompt .= "- This is a FOCUSED ARTICLE - go deep on this specific subtopic\n";
            $prompt .= "- Be detailed and thorough on this particular aspect\n";
        }
        
        if (!empty($context['pillar_keyword'])) {
            $prompt .= "- This article relates to: {$context['pillar_keyword']}\n";
        }
        
        if (!empty($context['related_topics'])) {
            $prompt .= "- Related topics: " . implode(', ', $context['related_topics']) . "\n";
        }
        
        $prompt .= "\n═══════════════════════════════════════════════════════════════\n";
        $prompt .= "WRITING GUIDELINES\n";
        $prompt .= "═══════════════════════════════════════════════════════════════\n\n";
        
        // Randomize suggested number and power word to prevent repetitive titles
        $suggested_numbers = array(3, 5, 6, 8, 9, 10, 11, 12);
        $suggested_number = $suggested_numbers[array_rand($suggested_numbers)];
        
        $power_words_positive = array('Essential', 'Ultimate', 'Best', 'Proven', 'Effective', 'Complete', 'Expert', 'Powerful', 'Key', 'Top', 'Smart', 'Quick', 'Simple', 'Easy');
        $power_words_negative = array('Critical', 'Dangerous', 'Common', 'Costly', 'Worst', 'Surprising', 'Hidden', 'Overlooked');
        $all_power_words = array_merge($power_words_positive, $power_words_negative);
        $suggested_power = $all_power_words[array_rand($all_power_words)];
        
        $prompt .= "TITLE (SEO-OPTIMIZED):\n";
        $prompt .= "- ⚠️ MAXIMUM 55 characters - count carefully!\n";
        $prompt .= "- ⚠️ NO COLONS (:) allowed in title - they cause truncation issues\n";
        $prompt .= "- ⚠️ Must be a COMPLETE sentence or phrase - no cut-off words\n";
        $prompt .= "- Start with the main topic \"{$keyword}\"\n";
        $prompt .= "- Keep it simple and direct\n";
        $prompt .= "- GOOD examples (simple, complete, no colons):\n";
        $prompt .= "  • '{$keyword} Guide for {$suggested_number} {$suggested_power} Results'\n";
        $prompt .= "  • '{$suggested_number} {$suggested_power} {$keyword} Tips That Work'\n";
        $prompt .= "  • 'How to Master {$keyword} in {$suggested_number} Steps'\n";
        $prompt .= "  • '{$keyword} Explained for Beginners'\n";
        $prompt .= "  • 'The {$suggested_power} {$keyword} Strategy Guide'\n";
        $prompt .= "- BAD examples (DO NOT USE):\n";
        $prompt .= "  • '{$keyword}: 9 Best' ❌ (has colon, incomplete)\n";
        $prompt .= "  • '{$keyword} Guide: Master' ❌ (has colon, cut off)\n";
        $prompt .= "  • 'Everything About {$keyword}: 10 Expert' ❌ (too long, colon, cut off)\n\n";
        
        $prompt .= "CONTENT STRUCTURE:\n";
        $prompt .= "- Start with an engaging introduction that addresses the topic directly\n";
        $prompt .= "- Mention \"{$keyword}\" in the FIRST paragraph (first 10% of content)\n";
        $prompt .= "- Organize content with clear H2 headings for main sections\n";
        $prompt .= "- Use H3 subheadings where helpful for organization\n";
        
        // Emphatic word count instructions (especially important for local models)
        if ($emphasize_length || $this->provider === 'walter') {
            $prompt .= "\n⚠️⚠️⚠️ CRITICAL WORD COUNT REQUIREMENT ⚠️⚠️⚠️\n";
            $prompt .= "- MINIMUM REQUIRED: {$word_count} words - this is NON-NEGOTIABLE\n";
            $prompt .= "- Your article MUST be at least {$word_count} words long\n";
            $prompt .= "- Short articles will be REJECTED - write comprehensively\n";
            $prompt .= "- Aim for 6-10 detailed H2 sections, each with 150-300 words\n";
            $prompt .= "- Include examples, explanations, and practical details\n";
            $prompt .= "- DO NOT summarize or cut short - elaborate fully on each point\n\n";
        } else {
            $prompt .= "- Write at least {$word_count} words for thorough coverage\n\n";
        }
        
        $prompt .= "⚠️ CRITICAL SEO REQUIREMENT - KEYWORD IN HEADINGS:\n";
        $prompt .= "- The main topic/keyword is: \"{$keyword}\"\n";
        $prompt .= "- Include this keyword (or close variation) in AT LEAST 3-4 of your H2 headings\n";
        $prompt .= "- Example H2: <h2 id=\"understanding-{$this->slugify($keyword)}\">Understanding {$keyword}</h2>\n";
        $prompt .= "- This is REQUIRED for SEO scoring - do not skip this!\n\n";
        
        $prompt .= "⚠️ KEYWORD DENSITY REQUIREMENT (CRITICAL FOR SEO):\n";
        $prompt .= "- Primary keyword: \"{$keyword}\"\n";
        $prompt .= "- Target density: 1.0-1.5% (keyword appears every 70-100 words)\n";
        $prompt .= "- For a {$word_count} word article: use the EXACT phrase at least " . max(10, intval($word_count / 100)) . "-" . max(15, intval($word_count / 70)) . " times\n";
        $prompt .= "- ALSO use keyword variations and partial matches (e.g., if keyword is 'termite mound temperature', also use 'termite mounds', 'mound temperature', 'temperature control')\n";
        $prompt .= "- Place keyword in: first paragraph, last paragraph, and distributed throughout\n";
        $prompt .= "- Avoid awkward repetition - integrate naturally into sentences\n\n";
        
        $prompt .= "HEADING HIERARCHY:\n";
        $prompt .= "- The article title is H1 (handled separately, NOT in your content)\n";
        $prompt .= "- Your content starts directly with introduction paragraphs\n";
        if ($this->provider === 'walter' || $emphasize_length) {
            $section_count = $is_pillar ? '8-12' : '6-8';
            $prompt .= "- Use H2 for ALL main sections - MUST have {$section_count} main sections minimum\n";
            $prompt .= "- Each H2 section should be 200-400 words with detailed content\n";
        } else {
            $prompt .= "- Use H2 for ALL main sections (aim for 6-10 main sections)\n";
        }
        $prompt .= "- Use H3 for subsections within H2 sections\n";
        $prompt .= "- Do NOT include H1 in your content\n";
        $prompt .= "- Do NOT use H4 or deeper levels\n\n";
        
        $prompt .= "READABILITY:\n";
        $prompt .= "- Keep paragraphs short and focused (3-5 sentences each)\n";
        $prompt .= "- Use clear, accessible language\n";
        $prompt .= "- Include transition words between ideas (however, therefore, additionally, etc.)\n";
        $prompt .= "- Vary sentence length for better flow\n";
        $prompt .= "- Break complex topics into digestible sections\n\n";
        
        $prompt .= "META DESCRIPTION:\n";
        $prompt .= "- Write a compelling summary (150-160 characters)\n";
        $prompt .= "- Include \"{$keyword}\" in the first 120 characters\n";
        $prompt .= "- Make it informative and engaging\n\n";
        
        $prompt .= "ADDITIONAL ELEMENTS:\n";
        $prompt .= "- ⚠️ DO NOT use citation markers, footnotes, or bracketed numbers like [1], [2], [3]\n";
        $prompt .= "- ⚠️ DO NOT add references or bibliography sections\n";
        $prompt .= "- If mentioning sources, integrate them naturally in the text (e.g., 'According to Harvard research...')\n";
        $prompt .= "- Include practical tips or takeaways where relevant\n";
        $prompt .= "- Provide image alt text that MUST include \"{$keyword}\" (for SEO)\n";
        $prompt .= "  Format: \"[Keyword] - descriptive image description\" (max 125 chars)\n\n";
        
        // Add localization reminder if persona has localities
        if ($this->persona->has_localities()) {
            $localities = $this->persona->get_localities();
            $primary_locality = $localities[0];
            $localization = $this->persona->get_localization_data($primary_locality);
            
            $prompt .= "═══════════════════════════════════════════════════════════════\n";
            $prompt .= "⚠️ LOCALIZATION REQUIREMENTS (CRITICAL)\n";
            $prompt .= "═══════════════════════════════════════════════════════════════\n";
            $prompt .= "This article is for: " . $this->persona->get_localities_display() . "\n\n";
            $prompt .= "YOU MUST USE THESE LOCAL CONVENTIONS:\n";
            $prompt .= "• Currency: {$localization['currency_symbol']} ({$localization['currency_name']})\n";
            $prompt .= "  - When mentioning ANY prices, costs, fees, salaries, or budgets, use {$localization['currency_symbol']}\n";
            $prompt .= "  - Format: {$localization['currency_format']}\n";
            $prompt .= "• Measurements: {$localization['measurement']} system\n";
            $prompt .= "• Temperature: {$localization['temperature']}\n";
            $prompt .= "• Spelling: {$localization['spelling']}\n";
            $prompt .= "• Date format: {$localization['date_format']}\n\n";
            $prompt .= "DO NOT use currencies from other regions (no $, €, £ unless that IS the local currency)\n";
            $prompt .= "DO NOT use American spellings if British English is specified (and vice versa)\n\n";
        }
        
        $prompt .= "═══════════════════════════════════════════════════════════════\n";
        $prompt .= "TABLE OF CONTENTS\n";
        $prompt .= "═══════════════════════════════════════════════════════════════\n";
        $prompt .= "- Include a Table of Contents after the introduction\n";
        $prompt .= "- Use ONLY HTML format - DO NOT use Markdown syntax like [text](#link)\n";
        $prompt .= "- Each H2 heading must have an id attribute for linking\n";
        $prompt .= "- Use this EXACT HTML format (compatible with SEO plugins):\n";
        $prompt .= '  <div class="wp-block-table-of-contents">' . "\n";
        $prompt .= '  <nav class="ez-toc-container">' . "\n";
        $prompt .= "  <p class=\"ez-toc-title\">Table of Contents</p>\n";
        $prompt .= '  <ul class="ez-toc-list">' . "\n";
        $prompt .= '    <li><a href="#section-id">Section Title</a></li>' . "\n";
        $prompt .= "  </ul>\n";
        $prompt .= "  </nav>\n";
        $prompt .= "  </div>\n";
        $prompt .= '  <h2 id="section-id">Section Title</h2>' . "\n\n";
        
        $prompt .= "═══════════════════════════════════════════════════════════════\n";
        $prompt .= "ARTICLE STRUCTURE\n";
        $prompt .= "═══════════════════════════════════════════════════════════════\n";
        
        if ($this->provider === 'walter' || $emphasize_length) {
            $section_count = $is_pillar ? '8-12' : '6-8';
            $words_per_section = $is_pillar ? '250-350' : '200-250';
            $prompt .= "⚠️ COMPREHENSIVE CONTENT REQUIRED - Follow this structure:\n\n";
            $prompt .= "1. Introduction (3-4 paragraphs, ~200 words - hook the reader and preview content)\n";
            $prompt .= "2. Table of Contents with navigation links\n";
            $prompt .= "3. Main Content: EXACTLY {$section_count} H2 sections\n";
            $prompt .= "   - Each H2 section MUST be {$words_per_section} words\n";
            $prompt .= "   - Include examples, case studies, or practical applications\n";
            $prompt .= "   - Use H3 subsections to break down complex topics\n";
            $prompt .= "   - Short, readable paragraphs (3-4 sentences each)\n";
            $prompt .= "4. Expert Tips section (5-8 actionable tips, ~200 words)\n";
            $prompt .= "5. FAQ section with 3-5 questions (~150 words)\n";
            $prompt .= "6. Conclusion (~150 words summarizing key takeaways)\n\n";
            $prompt .= "TOTAL MINIMUM: {$word_count} words - DO NOT write less!\n\n";
        } else {
            $prompt .= "1. Introduction (2-3 paragraphs introducing the topic)\n";
            $prompt .= "2. Table of Contents with navigation links\n";
            $prompt .= "3. Main Content: 6-10 H2 sections with H3 subsections as needed\n";
            $prompt .= "   - Each section thoroughly covers one aspect\n";
            $prompt .= "   - Short, readable paragraphs throughout\n";
            $prompt .= "4. Expert tips or key takeaways section\n";
            $prompt .= "5. Conclusion summarizing main points\n\n";
        }
        
        $prompt .= "═══════════════════════════════════════════════════════════════\n";
        $prompt .= "OUTPUT FORMAT (Follow Exactly)\n";
        $prompt .= "═══════════════════════════════════════════════════════════════\n";
        $prompt .= "IMPORTANT: Use ONLY HTML tags. Do NOT use any Markdown syntax.\n";
        $prompt .= "- NO Markdown links like [text](#link) - use <a href=\"#id\">text</a>\n";
        $prompt .= "- NO Markdown headers like ## Title - use <h2>Title</h2>\n";
        $prompt .= "- NO Markdown lists like - item - use <ul><li>item</li></ul>\n\n";
        $prompt .= "Return the article in this exact format with these markers:\n";
        $prompt .= "---TITLE---\n[Your article title]\n";
        $prompt .= "---META---\n[Meta description 150-160 characters]\n";
        $prompt .= "---EXCERPT---\n[2-3 sentence summary]\n";
        $prompt .= "---CONTENT---\n[Full HTML article: intro → TOC → H2/H3 sections with ids → conclusion]\n";
        $prompt .= "---TAGS---\n[Comma-separated relevant tags]\n";
        $prompt .= "---ALT_TEXT---\n[Image alt starting with \"{$keyword}\" - format: \"Keyword - description\"]\n";
        
        return $prompt;
    }
    
    /**
     * 💰 BUDGET MODE: Shorter article prompt (saves ~1500 input tokens per article)
     * Still produces quality content but with less verbose instructions
     */
    private function build_budget_article_prompt($keyword, $article_type, $is_pillar, $word_count, $context) {
        AIAB_Logger::debug("💰 Using budget mode prompt (shorter)", array('keyword' => $keyword));
        
        $prompt = "Write an SEO-optimized article about: \"{$keyword}\"\n\n";
        $prompt .= "Type: " . ($is_pillar ? 'Comprehensive Guide' : 'Focused Article') . "\n";
        $prompt .= "Words: {$word_count}\n";
        $prompt .= "Format: HTML only (no Markdown)\n\n";
        
        $prompt .= "REQUIREMENTS:\n";
        $prompt .= "1. Title: Under 55 chars, NO COLONS, complete phrase. Example: 'VPS Hosting Guide With 9 Expert Tips'\n";
        $prompt .= "2. Meta: 150-160 chars, keyword in first 120 chars\n";
        $prompt .= "3. Content: Intro → TOC → H2 sections (with id attributes) → Conclusion\n";
        $prompt .= "4. Keyword density: 1-1.5% (keyword every 70-100 words)\n";
        $prompt .= "5. Include keyword in 3-4 H2 headings\n";
        $prompt .= "6. NO citation markers [1][2][3], NO footnotes, NO bibliography\n\n";
        
        $prompt .= "TOC FORMAT:\n";
        $prompt .= '<div class="wp-block-table-of-contents"><nav class="ez-toc-container">' . "\n";
        $prompt .= '<p class="ez-toc-title">Table of Contents</p><ul class="ez-toc-list">' . "\n";
        $prompt .= '<li><a href="#section-id">Section</a></li></ul></nav></div>' . "\n\n";
        
        $prompt .= "OUTPUT FORMAT:\n";
        $prompt .= "---TITLE---\n[Title]\n";
        $prompt .= "---META---\n[Meta description]\n";
        $prompt .= "---EXCERPT---\n[2-3 sentence summary]\n";
        $prompt .= "---CONTENT---\n[HTML article]\n";
        $prompt .= "---TAGS---\n[Tags]\n";
        $prompt .= "---ALT_TEXT---\n[Image alt text - MUST start with \"{$keyword}\"]\n";
        
        return $prompt;
    }
    
    /**
     * Simple slugify for use in prompts
     */
    private function slugify($text) {
        $text = strtolower($text);
        $text = preg_replace('/[^a-z0-9\s-]/', '', $text);
        $text = preg_replace('/\s+/', '-', trim($text));
        $text = preg_replace('/-+/', '-', $text);
        return substr($text, 0, 30);
    }
    
    /**
     * Get article type description for prompt
     */
    private function get_type_description($type) {
        $descriptions = array(
            // Pillar Content
            'pillar' => 'COMPREHENSIVE GUIDE - Write a thorough, authoritative guide that covers this broad topic completely. This should be the definitive resource on the subject. Include multiple sections, expert insights, and make it a complete reference. Target 2500+ words.',
            
            // Supporting/Cluster Content
            'supporting' => 'FOCUSED ARTICLE - Write a detailed article that covers one specific subtopic thoroughly. Go deep on this narrow subject with practical information.',
            
            // How-to Guides
            'how_to' => 'HOW-TO GUIDE - Write a step-by-step tutorial that solves a specific problem. Use numbered steps, clear instructions, and practical tips. Include a materials/requirements section if relevant. Make it actionable so readers can follow along immediately.',
            
            // Listicles
            'listicle' => 'LIST ARTICLE - Write a numbered list article covering multiple points. Use a clear number in the structure (e.g., "7 Signs...", "10 Ways..."). Make each point clear with descriptive headers.',
            
            // Comparison Articles
            'comparison' => 'COMPARISON ARTICLE - Write an objective comparison with clear pros and cons for each option. Use a structured format with side-by-side analysis. Include a verdict or recommendation at the end.',
            
            // Best-of / Review Articles
            'review' => 'REVIEW ARTICLE - Write a helpful review or "best of" roundup. Include specific recommendations with pros and cons. Be objective and helpful.',
            
            // Problem-Solution Articles
            'problem_solution' => 'PROBLEM-SOLUTION ARTICLE - Start by acknowledging the reader\'s challenge, explain what causes the problem, then provide clear solutions. Make it practical and actionable.',
            
            // Question-Based (Q&A)
            'question' => 'Q&A ARTICLE - Directly answer the question in the first paragraph. Then expand with detailed explanation. Provide clear, thorough answers.',
            
            // Local Articles
            'local_seo' => 'REGIONAL ARTICLE - Write content with geographical relevance. Include specific references to the region (UAE, Dubai, Middle East, etc.). Mention local regulations, climate factors, or regional considerations.',
            
            // Cost/Pricing Articles
            'cost' => 'PRICING GUIDE - Provide specific cost ranges, factors that affect pricing, and what to expect. Include a pricing table or breakdown if relevant.',
            
            // Glossary/Definition Articles
            'glossary' => 'DEFINITION ARTICLE - Define the term clearly in the first sentence. Then expand with context, examples, why it matters, and related concepts.',
            
            // Buyer\'s Guides
            'buyers_guide' => 'BUYER\'S GUIDE - Help readers make purchasing decisions. Cover what to look for, features that matter, common mistakes to avoid, and specific recommendations.',
            
            // Case Studies
            'case_study' => 'CASE STUDY - Write a real-world example of a problem and solution. Use a narrative structure: the challenge, the approach, the solution, the results.',
            
            // Seasonal/Trend Articles
            'seasonal' => 'SEASONAL ARTICLE - Tie the topic to specific time periods, weather, or events. Reference how seasons or trends affect this topic.',
            
            // General Guide
            'guide' => 'DETAILED GUIDE - Write a thorough explanation of the topic. Include practical insights, tips, and takeaways.',
            
            // Informational (fallback)
            'informational' => 'INFORMATIONAL ARTICLE - Educate the reader thoroughly on this topic. Provide valuable insights, explain concepts clearly, and include actionable takeaways.',
        );
        
        return isset($descriptions[$type]) ? $descriptions[$type] : $descriptions['informational'];
    }
    
    /**
     * Parse the AI response into structured article data
     */
    private function parse_article_response($content, $keyword) {
        $article = array(
            'title' => '',
            'meta_description' => '',
            'excerpt' => '',
            'content' => '',
            'tags' => array(),
            'alt_text' => '',
            'word_count' => 0,
            'keyword' => $keyword, // Primary keyword passed in
            'focus_keyword' => '' // Will be optimized below
        );
        
        // Log raw response length for debugging
        $raw_length = strlen($content);
        AIAB_Logger::debug("Parsing AI response", array(
            'raw_length' => $raw_length,
            'has_title_marker' => strpos($content, '---TITLE---') !== false,
            'has_content_marker' => strpos($content, '---CONTENT---') !== false
        ));
        
        // Parse title - try multiple patterns
        if (preg_match('/---TITLE---\s*\n(.+?)(?=\n---|\n\n---)/s', $content, $matches)) {
            $article['title'] = trim($matches[1]);
        } elseif (preg_match('/---TITLE---\s*\n(.+?)$/m', $content, $matches)) {
            $article['title'] = trim($matches[1]);
        } else {
            // Fallback: look for first H1 or strong heading
            if (preg_match('/<h1[^>]*>(.+?)<\/h1>/i', $content, $matches)) {
                $article['title'] = trim(strip_tags($matches[1]));
            } elseif (preg_match('/^#\s+(.+)$/m', $content, $matches)) {
                $article['title'] = trim($matches[1]);
            } else {
                // Create a short SEO-friendly title from the keyword
                // Don't just use the full keyword - truncate intelligently
                $title_keyword = $keyword;
                
                // If keyword has a colon (subtitle), use only the first part
                if (strpos($title_keyword, ':') !== false) {
                    $title_keyword = trim(explode(':', $title_keyword)[0]);
                }
                
                // Limit to first 8 words max
                $words = explode(' ', $title_keyword);
                if (count($words) > 8) {
                    $title_keyword = implode(' ', array_slice($words, 0, 8));
                }
                
                // Ensure under 60 chars
                if (strlen($title_keyword) > 55) {
                    $title_keyword = substr($title_keyword, 0, 55);
                    $last_space = strrpos($title_keyword, ' ');
                    if ($last_space > 30) {
                        $title_keyword = substr($title_keyword, 0, $last_space);
                    }
                }
                
                $article['title'] = ucwords($title_keyword);
                AIAB_Logger::warning("Title fallback used - created from keyword", array(
                    'original_keyword' => substr($keyword, 0, 100) . (strlen($keyword) > 100 ? '...' : ''),
                    'generated_title' => $article['title']
                ));
            }
        }
        
        // SANITIZE TITLE: Remove markdown, asterisks, and enforce length limit
        $article['title'] = preg_replace('/\*+/', '', $article['title']); // Remove asterisks
        $article['title'] = preg_replace('/_{2,}/', '', $article['title']); // Remove underscores
        $article['title'] = preg_replace('/`+/', '', $article['title']); // Remove backticks
        $article['title'] = preg_replace('/#+\s*/', '', $article['title']); // Remove markdown headers
        $article['title'] = trim($article['title']);
        
        // Store original title before any modifications
        $original_title = $article['title'];
        
        // First check: Is the title already incomplete (before truncation)?
        $needs_regeneration = $this->is_title_incomplete($article['title']);
        
        // Second check: Will truncation make it incomplete?
        if (!$needs_regeneration && strlen($article['title']) > 60) {
            // Simulate what truncation would produce
            $truncated = substr($article['title'], 0, 60);
            $last_space = strrpos($truncated, ' ');
            if ($last_space > 40) {
                $simulated_truncated = substr($truncated, 0, $last_space);
            } else {
                $simulated_truncated = $truncated;
            }
            
            // Check if truncated version would be incomplete
            if ($this->is_title_incomplete($simulated_truncated)) {
                $needs_regeneration = true;
                AIAB_Logger::debug("Title would be incomplete after truncation, will regenerate", array(
                    'original' => $original_title,
                    'would_truncate_to' => $simulated_truncated
                ));
            } else {
                // Truncation is safe - use the truncated version
                $article['title'] = $simulated_truncated;
                AIAB_Logger::debug("📏 Article title safely truncated to 60 chars: " . $article['title']);
            }
        }
        
        // Note: Regeneration will happen AFTER content is parsed (needs content for context)
        // Store flag for later
        $article['_needs_title_regeneration'] = $needs_regeneration;
        $article['_original_title'] = $original_title;
        
        // Parse meta description
        if (preg_match('/---META---\s*\n(.+?)(?=\n---|\n\n)/s', $content, $matches)) {
            $article['meta_description'] = trim(substr($matches[1], 0, 160));
        } elseif (preg_match('/---META---\s*\n(.+?)$/m', $content, $matches)) {
            $article['meta_description'] = trim(substr($matches[1], 0, 160));
        }
        
        // Parse excerpt
        if (preg_match('/---EXCERPT---\s*\n(.+?)(?=\n---|\n\n---)/s', $content, $matches)) {
            $article['excerpt'] = trim($matches[1]);
        } elseif (preg_match('/---EXCERPT---\s*\n(.+?)$/m', $content, $matches)) {
            $article['excerpt'] = trim($matches[1]);
        }
        
        // Parse content - this is the most important part
        $parsed_content = '';
        
        // Method 1: Standard format with ---CONTENT---
        if (preg_match('/---CONTENT---\s*\n(.+?)(?=\n---TAGS---|\n---ALT_TEXT---|$)/s', $content, $matches)) {
            $parsed_content = trim($matches[1]);
        }
        
        // Method 2: Content extends to end (no closing marker)
        if (empty($parsed_content) && preg_match('/---CONTENT---\s*\n(.+)$/s', $content, $matches)) {
            $parsed_content = trim($matches[1]);
            // Remove any trailing markers
            $parsed_content = preg_replace('/\n---[A-Z_]+---.*$/s', '', $parsed_content);
        }
        
        // Method 3: No markers at all - use the whole response but clean it
        if (empty($parsed_content) || strlen($parsed_content) < 500) {
            AIAB_Logger::warning("Content markers not found or content too short, using fallback parsing");
            
            // If there's structured content, try to extract it
            if (strpos($content, '<h2') !== false || strpos($content, '## ') !== false) {
                // Content has headings - it's probably the article body
                $parsed_content = $content;
                
                // Remove any marker lines at the start
                $parsed_content = preg_replace('/^---[A-Z_]+---\s*\n.*?\n(?=<|#|\w)/s', '', $parsed_content);
            } else {
                // Use everything as fallback
                $parsed_content = $content;
            }
        }
        
        // Clean any remaining markers from content
        $parsed_content = $this->strip_output_markers($parsed_content);
        
        $article['content'] = $parsed_content;
        
        // Parse tags
        if (preg_match('/---TAGS---\s*\n(.+?)(?=\n---|$)/s', $content, $matches)) {
            $tags = explode(',', $matches[1]);
            $article['tags'] = array_map('trim', array_filter($tags));
        }
        
        // Parse alt text
        if (preg_match('/---ALT_TEXT---\s*\n(.+?)(?=\n---|$)/s', $content, $matches)) {
            $article['alt_text'] = trim($matches[1]);
        } else {
            $article['alt_text'] = "Featured image for: " . $article['title'];
        }
        
        // Calculate word count BEFORE formatting
        $raw_word_count = str_word_count(strip_tags($article['content']));
        
        // Safety check - if content is suspiciously short, log a warning
        if ($raw_word_count < 100) {
            AIAB_Logger::error("Article content is too short!", array(
                'word_count' => $raw_word_count,
                'content_preview' => substr($article['content'], 0, 500),
                'raw_response_preview' => substr($content, 0, 1000)
            ));
        }
        
        // Clean up content - ensure proper heading structure
        $article['content'] = $this->format_content($article['content']);
        
        // Recalculate word count after formatting
        $article['word_count'] = str_word_count(strip_tags($article['content']));
        
        AIAB_Logger::debug("Article parsed", array(
            'title' => $article['title'],
            'word_count' => $article['word_count'],
            'has_meta' => !empty($article['meta_description']),
            'tags_count' => count($article['tags'])
        ));
        
        // Extract and optimize the focus keyword for RankMath
        $article['focus_keyword'] = $this->optimize_focus_keyword($keyword, $article['title'], $article['content']);
        
        // ═══ TITLE REGENERATION (if needed) ═══
        // Now that we have content and focus keyword, regenerate incomplete titles
        if (!empty($article['_needs_title_regeneration']) && !empty($article['content'])) {
            $original_title = isset($article['_original_title']) ? $article['_original_title'] : $article['title'];
            
            AIAB_Logger::info("🔄 Triggering title regeneration", array(
                'original_title' => $original_title,
                'focus_keyword' => $article['focus_keyword'],
                'content_length' => strlen($article['content'])
            ));
            
            $article['title'] = $this->regenerate_title_from_content(
                $article['content'],
                $article['focus_keyword'],
                $original_title
            );
            
            // Update alt text if it references the old title
            if (strpos($article['alt_text'], $original_title) !== false || 
                strpos($article['alt_text'], 'Featured image for:') !== false) {
                $article['alt_text'] = $article['focus_keyword'] . " - " . $article['title'];
            }
        }
        
        // Clean up internal flags before returning
        unset($article['_needs_title_regeneration']);
        unset($article['_original_title']);
        
        return $article;
    }
    
    /**
     * Strip any remaining output markers and trailing TOC from content
     * This catches cases where markers leak through the parsing
     */
    private function strip_output_markers($content) {
        if (empty($content)) {
            return $content;
        }
        
        // Remove any ---MARKER--- lines and everything after common markers
        $markers_to_strip = array(
            '---TAGS---',
            '---ALT_TEXT---',
            '---META---',
            '---EXCERPT---',
            '---TITLE---',
            '---END---',
            '---CONTENT---',
        );
        
        foreach ($markers_to_strip as $marker) {
            // Remove the marker and everything after it
            $pos = strpos($content, $marker);
            if ($pos !== false) {
                $content = substr($content, 0, $pos);
            }
        }
        
        // Remove any trailing "Table of Contents" section at the END of content
        // This pattern matches TOC that appears after the main content
        $content = preg_replace('/\n+Table of Contents\s*\n+(\s*[\*\-•]\s*[^\n]+\n*)+$/is', '', $content);
        
        // Also remove if it's using numbered list format
        $content = preg_replace('/\n+Table of Contents\s*\n+(\s*\d+\.\s*[^\n]+\n*)+$/is', '', $content);
        
        // Remove any other common AI artifacts at the end
        $content = preg_replace('/\n+---+\s*$/s', '', $content);  // Trailing dashes
        $content = preg_replace('/\n+\*\*\*+\s*$/s', '', $content);  // Trailing asterisks
        
        // Trim whitespace
        $content = trim($content);
        
        return $content;
    }
    
    /**
     * Optimize the focus keyword for RankMath SEO
     * 
     * IMPORTANT: RankMath needs the EXACT same keyword to appear in:
     * - Title (at beginning)
     * - URL
     * - Meta description
     * - Content
     * 
     * So we must be VERY conservative about modifying the keyword.
     * Only remove question prefixes, nothing else.
     */
    private function optimize_focus_keyword($original_keyword, $title, $content) {
        // Start with the original keyword, just lowercase and trimmed
        $focus_keyword = strtolower(trim($original_keyword));
        
        // Only remove question patterns from the START - nothing else!
        $question_prefixes = array(
            'what is ', 'what are ', 'what does ', 'what do ',
            'how to ', 'how do ', 'how does ', 'how can ',
            'why is ', 'why are ', 'why does ', 'why do ',
            'when to ', 'when should ', 'when do ',
            'where to ', 'where can ', 'where do ',
            'which is ', 'which are ',
            'who is ', 'who are ',
            'can you ', 'can i ', 'can we ',
            'should i ', 'should you ', 'should we ',
            'is it ', 'is there ', 'is this ',
            'are there ', 'are these ',
            'does it ', 'does this ',
            'do i ', 'do you ', 'do we ',
            'will it ', 'will this ',
        );
        
        foreach ($question_prefixes as $prefix) {
            if (strpos($focus_keyword, $prefix) === 0) {
                $focus_keyword = substr($focus_keyword, strlen($prefix));
                break; // Only remove one prefix
            }
        }
        
        // Remove trailing question mark only
        $focus_keyword = rtrim($focus_keyword, '?');
        
        // Trim and normalize whitespace
        $focus_keyword = trim($focus_keyword);
        $focus_keyword = preg_replace('/\s+/', ' ', $focus_keyword);
        
        // That's it - don't try to shorten or modify further!
        // The keyword must remain intact for RankMath matching
        
        AIAB_Logger::debug("Focus keyword optimized", array(
            'original' => $original_keyword,
            'optimized' => $focus_keyword
        ));
        
        return $focus_keyword;
    }
    
    /**
     * ═══════════════════════════════════════════════════════════════
     * KEYWORD DENSITY ENFORCEMENT (RankMath 100/100 Score)
     * ═══════════════════════════════════════════════════════════════
     * 
     * This method ensures the EXACT focus keyword appears in content
     * with proper density (1-1.5%) for RankMath SEO scoring.
     * 
     * RankMath checks:
     * ✓ Keyword in first paragraph (first 10% of content)
     * ✓ Keyword in H2 subheadings (2-3 times)
     * ✓ Keyword density 1-1.5% (keyword every 70-100 words)
     * ✓ Keyword in conclusion/final paragraph
     */
    public function enforce_keyword_density(&$article) {
        $focus_keyword = isset($article['focus_keyword']) ? $article['focus_keyword'] : 
                        (isset($article['keyword']) ? $article['keyword'] : '');
        
        if (empty($focus_keyword) || empty($article['content'])) {
            return;
        }
        
        $content = $article['content'];
        $word_count = str_word_count(strip_tags($content));
        
        // Calculate current keyword density
        $keyword_count = $this->count_keyword_occurrences($content, $focus_keyword);
        $current_density = ($word_count > 0) ? ($keyword_count / $word_count) * 100 : 0;
        
        // Target: 1.0-1.5% density
        $target_count_min = max(1, intval($word_count / 100)); // 1%
        $target_count_max = max(2, intval($word_count / 70));  // ~1.4%
        
        AIAB_Logger::debug("Keyword density analysis", array(
            'keyword' => $focus_keyword,
            'current_count' => $keyword_count,
            'current_density' => round($current_density, 2) . '%',
            'word_count' => $word_count,
            'target_range' => "{$target_count_min}-{$target_count_max} occurrences",
            'needs_enforcement' => $keyword_count < $target_count_min
        ));
        
        // 1. Ensure keyword in FIRST paragraph (critical for RankMath)
        $content = $this->ensure_keyword_in_first_paragraph($content, $focus_keyword);
        
        // 2. Ensure keyword in 2-3 H2 headings (using safe replacement)
        $content = $this->ensure_keyword_in_h2_headings($content, $focus_keyword);
        
        // 3. If density still low, inject keyword throughout body
        $current_count = $this->count_keyword_occurrences($content, $focus_keyword);
        if ($current_count < $target_count_min) {
            $content = $this->inject_keyword_throughout_content($content, $focus_keyword, $target_count_min, $target_count_max);
        }
        
        // 4. Ensure keyword in LAST paragraph/conclusion
        $content = $this->ensure_keyword_in_conclusion($content, $focus_keyword);
        
        // Recalculate and log final density
        $final_count = $this->count_keyword_occurrences($content, $focus_keyword);
        $final_density = ($word_count > 0) ? ($final_count / $word_count) * 100 : 0;
        
        AIAB_Logger::info("✅ Keyword density enforced", array(
            'keyword' => $focus_keyword,
            'before_count' => $keyword_count,
            'after_count' => $final_count,
            'density' => round($final_density, 2) . '%'
        ));
        
        $article['content'] = $content;
    }
    
    /**
     * Count exact keyword occurrences (case-insensitive, whole word)
     */
    private function count_keyword_occurrences($content, $keyword) {
        if (empty($keyword)) return 0;
        
        // Strip HTML for accurate counting
        $text = strip_tags($content);
        
        // Count case-insensitive occurrences (with word boundaries for accuracy)
        $pattern = '/\b' . preg_quote($keyword, '/') . '\b/iu';
        preg_match_all($pattern, $text, $matches);
        
        return count($matches[0]);
    }
    
    /**
     * Ensure keyword appears in FIRST paragraph
     * RankMath: "Keyword in first 10% of content"
     */
    private function ensure_keyword_in_first_paragraph($content, $keyword) {
        // Find the first <p> tag content
        if (!preg_match('/<p[^>]*>(.*?)<\/p>/is', $content, $match)) {
            return $content;
        }
        
        $full_match = $match[0];
        $first_p_content = $match[1];
        
        // Check if keyword already in first paragraph
        if (stripos($first_p_content, $keyword) !== false) {
            return $content;
        }
        
        // Build new first paragraph with keyword at the start
        $ucKeyword = ucwords($keyword);
        $new_first_p_content = "Understanding <strong>{$ucKeyword}</strong> is essential. " . $first_p_content;
        
        // Replace just the inner content, preserving the p tags
        $new_p_tag = str_replace($first_p_content, $new_first_p_content, $full_match);
        
        // Replace only the first occurrence
        $content = preg_replace('/' . preg_quote($full_match, '/') . '/', $new_p_tag, $content, 1);
        
        AIAB_Logger::debug("Injected keyword into first paragraph", array(
            'keyword' => $keyword
        ));
        
        return $content;
    }
    
    /**
     * Ensure keyword appears in 2-3 H2 headings
     * RankMath: "Keyword in subheading(s)"
     * 
     * SAFE IMPLEMENTATION: Uses preg_replace_callback to avoid offset issues
     */
    private function ensure_keyword_in_h2_headings($content, $keyword) {
        // First, count how many H2s already have the keyword
        preg_match_all('/<h2[^>]*>(.+?)<\/h2>/is', $content, $all_matches);
        
        if (empty($all_matches[0])) {
            return $content;
        }
        
        $total_h2s = count($all_matches[0]);
        $headings_with_keyword = 0;
        
        foreach ($all_matches[1] as $heading_text) {
            if (stripos(strip_tags($heading_text), $keyword) !== false) {
                $headings_with_keyword++;
            }
        }
        
        // Target: 2-3 H2s with keyword
        $target_h2s = min(3, max(2, $total_h2s - 2));
        
        if ($headings_with_keyword >= $target_h2s) {
            return $content;
        }
        
        $modifications_needed = $target_h2s - $headings_with_keyword;
        $modified = 0;
        $skip_patterns = array('table of contents', 'conclusion', 'introduction', 'faq', 'summary');
        
        // Use callback for safe replacement
        $ucKeyword = ucwords($keyword);
        
        $content = preg_replace_callback(
            '/<h2([^>]*)>(.+?)<\/h2>/is',
            function($match) use ($keyword, $ucKeyword, &$modified, $modifications_needed, $skip_patterns) {
                // Already modified enough
                if ($modified >= $modifications_needed) {
                    return $match[0];
                }
                
                $attrs = $match[1];
                $heading_text = strip_tags($match[2]);
                
                // Skip if already has keyword
                if (stripos($heading_text, $keyword) !== false) {
                    return $match[0];
                }
                
                // Skip TOC/Conclusion/etc headings
                foreach ($skip_patterns as $skip) {
                    if (stripos($heading_text, $skip) !== false) {
                        return $match[0];
                    }
                }
                
                // Add keyword to heading - prepend for clarity
                $new_heading = "{$ucKeyword}: {$heading_text}";
                
                // Keep it reasonable length
                if (strlen($new_heading) > 80) {
                    $new_heading = "{$ucKeyword} - " . substr($heading_text, 0, 60);
                }
                
                $modified++;
                return "<h2{$attrs}>{$new_heading}</h2>";
            },
            $content
        );
        
        if ($modified > 0) {
            AIAB_Logger::debug("Added keyword to H2 headings", array(
                'keyword' => $keyword,
                'headings_modified' => $modified
            ));
        }
        
        return $content;
    }
    
    /**
     * Inject keyword throughout content to hit target density
     */
    private function inject_keyword_throughout_content($content, $keyword, $target_min, $target_max) {
        $current_count = $this->count_keyword_occurrences($content, $keyword);
        $injections_needed = max(0, $target_min - $current_count);
        
        if ($injections_needed === 0) {
            return $content;
        }
        
        // Find all paragraphs
        preg_match_all('/<p[^>]*>(.+?)<\/p>/is', $content, $matches, PREG_SET_ORDER);
        
        if (count($matches) < 3) {
            return $content;
        }
        
        // Natural keyword insertion phrases
        $insertion_phrases = array(
            "This relates directly to {KEYWORD}.",
            "When considering {KEYWORD}, this becomes clear.",
            "The importance of {KEYWORD} is evident here.",
            "Understanding {KEYWORD} helps with this aspect.",
            "{KEYWORD} factors into this consideration.",
        );
        
        $injected = 0;
        $ucKeyword = ucwords($keyword);
        
        // Skip first and last paragraphs (handled separately)
        // Spread injections across middle paragraphs
        $middle_paragraphs = array_slice($matches, 1, -1);
        $step = max(1, floor(count($middle_paragraphs) / max(1, $injections_needed)));
        
        for ($i = 0; $i < count($middle_paragraphs) && $injected < $injections_needed; $i += $step) {
            $match = $middle_paragraphs[$i];
            $full_tag = $match[0];
            $p_content = $match[1];
            
            // Skip if already has keyword
            if (stripos($p_content, $keyword) !== false) {
                continue;
            }
            
            // Skip very short paragraphs
            if (str_word_count(strip_tags($p_content)) < 20) {
                continue;
            }
            
            // Choose insertion phrase
            $phrase = str_replace('{KEYWORD}', "<strong>{$ucKeyword}</strong>", $insertion_phrases[$injected % count($insertion_phrases)]);
            
            // Append phrase to paragraph
            $new_content = rtrim($p_content) . " " . $phrase;
            $new_tag = str_replace($p_content, $new_content, $full_tag);
            
            // Replace in content (only first occurrence of this exact paragraph)
            $content = preg_replace('/' . preg_quote($full_tag, '/') . '/', $new_tag, $content, 1);
            
            $injected++;
        }
        
        AIAB_Logger::debug("Injected keyword into body paragraphs", array(
            'keyword' => $keyword,
            'injections' => $injected
        ));
        
        return $content;
    }
    
    /**
     * Ensure keyword appears in conclusion/final paragraph
     */
    private function ensure_keyword_in_conclusion($content, $keyword) {
        // Find the last paragraph
        preg_match_all('/<p[^>]*>(.+?)<\/p>/is', $content, $matches, PREG_SET_ORDER);
        
        if (empty($matches)) {
            return $content;
        }
        
        $last_match = end($matches);
        $full_tag = $last_match[0];
        $last_p_content = $last_match[1];
        
        // Check if keyword already in last paragraph
        if (stripos($last_p_content, $keyword) !== false) {
            return $content;
        }
        
        // Add keyword to conclusion naturally
        $ucKeyword = ucwords($keyword);
        $conclusion_addition = " Understanding <strong>{$ucKeyword}</strong> is key to success in this area.";
        
        $new_content = rtrim($last_p_content) . $conclusion_addition;
        $new_tag = str_replace($last_p_content, $new_content, $full_tag);
        
        // Replace last occurrence
        $pos = strrpos($content, $full_tag);
        if ($pos !== false) {
            $content = substr_replace($content, $new_tag, $pos, strlen($full_tag));
        }
        
        AIAB_Logger::debug("Added keyword to conclusion", array('keyword' => $keyword));
        
        return $content;
    }
    
    /**
     * Format content with proper HTML structure
     */
    private function format_content($content) {
        // FIRST: Remove any raw Markdown ToC links that weren't converted to HTML
        // These show up as: - [Section Title](#anchor)
        // We remove them because there should be an HTML TOC already
        $content = preg_replace('/^[-*]\s*\[([^\]]+)\]\(#[^\)]+\)\s*$/m', '', $content);
        
        // Also remove numbered markdown links: 1. [Section Title](#anchor)
        $content = preg_replace('/^\d+\.\s*\[([^\]]+)\]\(#[^\)]+\)\s*$/m', '', $content);
        
        // Convert any inline markdown links to HTML: [text](url) -> <a href="url">text</a>
        $content = preg_replace('/\[([^\]]+)\]\(([^\)]+)\)/', '<a href="$2">$1</a>', $content);
        
        // IMPORTANT: Remove ALL H1 tags - WordPress uses the post title as H1
        // This prevents duplicate titles showing on the page
        $content = preg_replace('/<h1[^>]*>.*?<\/h1>\s*/is', '', $content);
        
        // Also remove markdown H1 (# Title)
        $content = preg_replace('/^#\s+.+$/m', '', $content);
        
        // Convert any remaining markdown headings
        $content = preg_replace('/^## (.+)$/m', '<h2>$1</h2>', $content);
        $content = preg_replace('/^### (.+)$/m', '<h3>$1</h3>', $content);
        $content = preg_replace('/^#### (.+)$/m', '<h3>$1</h3>', $content); // H4 becomes H3
        $content = preg_replace('/^##### (.+)$/m', '<h3>$1</h3>', $content); // H5 becomes H3
        
        // Convert H4, H5, H6 HTML tags to H3 (flatten structure)
        $content = preg_replace('/<h4([^>]*)>/i', '<h3$1>', $content);
        $content = preg_replace('/<\/h4>/i', '</h3>', $content);
        $content = preg_replace('/<h5([^>]*)>/i', '<h3$1>', $content);
        $content = preg_replace('/<\/h5>/i', '</h3>', $content);
        $content = preg_replace('/<h6([^>]*)>/i', '<h3$1>', $content);
        $content = preg_replace('/<\/h6>/i', '</h3>', $content);
        
        // Ensure all H2 headings have ID attributes for anchor links
        $content = $this->add_heading_ids($content);
        
        // Check if TOC exists, if not generate one
        // Check for multiple TOC formats recognized by SEO plugins
        $has_toc = strpos($content, 'class="toc"') !== false 
                || strpos($content, 'table-of-contents') !== false
                || strpos($content, 'article-toc') !== false
                || strpos($content, 'rank-math-toc') !== false
                || strpos($content, 'ez-toc') !== false
                || strpos($content, 'wp-block-table-of-contents') !== false
                || strpos($content, 'lwptoc') !== false
                || strpos($content, 'Table of Contents') !== false;
        
        if (!$has_toc) {
            $content = $this->generate_toc($content);
        }
        
        // Convert markdown bold/italic if present
        $content = preg_replace('/\*\*(.+?)\*\*/', '<strong>$1</strong>', $content);
        $content = preg_replace('/\*(.+?)\*/', '<em>$1</em>', $content);
        
        // Ensure paragraphs are wrapped
        $paragraphs = preg_split('/\n\s*\n/', $content);
        $formatted = array();
        
        foreach ($paragraphs as $p) {
            $p = trim($p);
            if (empty($p)) continue;
            
            // Don't wrap if it's already a block element
            if (preg_match('/^<(h[1-6]|ul|ol|blockquote|div|p|table|nav)/i', $p)) {
                $formatted[] = $p;
            } else {
                $formatted[] = '<p>' . $p . '</p>';
            }
        }
        
        // Remove any empty paragraphs
        $content = implode("\n\n", $formatted);
        $content = preg_replace('/<p>\s*<\/p>/', '', $content);
        
        // Trim any leading/trailing whitespace
        $content = trim($content);
        
        return $content;
    }
    
    /**
     * Add ID attributes to H2 headings for anchor links
     */
    private function add_heading_ids($content) {
        $counter = 1;
        
        // Find all H2 tags without IDs and add them
        $content = preg_replace_callback(
            '/<h2(?![^>]*\bid=)([^>]*)>(.+?)<\/h2>/is',
            function($matches) use (&$counter) {
                $attrs = $matches[1];
                $text = $matches[2];
                $id = 'section-' . $counter;
                $counter++;
                return '<h2 id="' . $id . '"' . $attrs . '>' . $text . '</h2>';
            },
            $content
        );
        
        return $content;
    }
    
    /**
     * Generate Table of Contents from H2 headings
     * Uses classes that SEO plugins (RankMath, etc.) recognize
     */
    private function generate_toc($content) {
        // Extract all H2 headings with their IDs
        preg_match_all('/<h2[^>]*id=["\']([^"\']+)["\'][^>]*>(.+?)<\/h2>/is', $content, $matches, PREG_SET_ORDER);
        
        if (count($matches) < 2) {
            // Not enough sections for a TOC
            return $content;
        }
        
        // Build TOC HTML using classes recognized by SEO plugins
        // ez-toc classes are recognized by RankMath as "Easy Table of Contents" plugin format
        $toc = '<div class="wp-block-table-of-contents">' . "\n";
        $toc .= '<nav class="ez-toc-container">' . "\n";
        $toc .= '<p class="ez-toc-title">Table of Contents</p>' . "\n";
        $toc .= '<ul class="ez-toc-list">' . "\n";
        
        foreach ($matches as $match) {
            $id = $match[1];
            $text = strip_tags($match[2]);
            
            // Skip TOC heading itself
            if (stripos($text, 'table of contents') !== false || stripos($text, 'in this article') !== false) {
                continue;
            }
            
            $toc .= '  <li class="ez-toc-page-1"><a class="ez-toc-link" href="#' . esc_attr($id) . '">' . esc_html($text) . '</a></li>' . "\n";
        }
        
        $toc .= '</ul>' . "\n";
        $toc .= '</nav>' . "\n";
        $toc .= '</div>' . "\n\n";
        
        // Find where to insert TOC (after first paragraph, before first H2)
        $first_h2_pos = strpos($content, '<h2');
        
        if ($first_h2_pos !== false) {
            // Insert TOC before first H2
            $content = substr($content, 0, $first_h2_pos) . $toc . substr($content, $first_h2_pos);
        } else {
            // No H2 found, prepend TOC
            $content = $toc . $content;
        }
        
        return $content;
    }
    
    /**
     * Generate a title for an article (RankMath optimized)
     */
    public function generate_title($keyword, $article_type) {
        $prompt = "Generate ONE compelling SEO-optimized title for an article about: \"{$keyword}\"\n";
        $prompt .= "Article type: {$article_type}\n\n";
        $prompt .= "REQUIREMENTS FOR RANKMATH 100/100 SCORE:\n";
        $prompt .= "1. Put the keyword \"{$keyword}\" at the BEGINNING (first half of title)\n";
        $prompt .= "2. Include a NUMBER (e.g., 7, 10, 5, 2024, 2025)\n";
        $prompt .= "3. Include a POWER WORD: Ultimate, Essential, Proven, Complete, Expert, Powerful, Simple, Easy, Best, Top, Amazing, Incredible\n";
        $prompt .= "4. Include SENTIMENT: positive (best, amazing, perfect) OR negative (avoid, never, mistakes, worst)\n";
        $prompt .= "5. Keep under 60 characters total\n";
        $prompt .= "6. Make it compelling and click-worthy\n\n";
        $prompt .= "GOOD EXAMPLES:\n";
        $prompt .= "- \"Pest Control: 7 Proven Methods for Amazing Results\"\n";
        $prompt .= "- \"Indoor Air Quality: 10 Essential Tips You Need Now\"\n";
        $prompt .= "- \"Dubai Pest Control: 5 Mistakes to Avoid in 2025\"\n\n";
        $prompt .= "Respond with ONLY the title, nothing else. No quotes or punctuation at the end.";
        
        $title = $this->generate_completion($prompt, 100);
        return trim($title, "\"'\n ");
    }
    
    /**
     * Check if a title appears incomplete/truncated
     * 
     * Detects patterns like:
     * - "Top Kvm Vps Providers For Developers 2025: 9 Best" (ends with adjective, missing noun)
     * - "Gpu Server Cost Optimization Strategies: 9 Proven" (ends with adjective, missing noun)
     * - Titles ending with numbers followed by incomplete words
     * - Titles ending with colons or other incomplete punctuation
     */
    private function is_title_incomplete($title) {
        $title = trim($title);
        
        // ═══ PATTERN GROUP A: AI META-COMMENTARY DETECTION ═══
        // These indicate the AI leaked its thinking/reasoning into the title
        
        // Pattern A0: Contains a colon - these cause truncation issues
        // We want simple titles without colons
        if (strpos($title, ':') !== false) {
            AIAB_Logger::debug("Title contains colon - will regenerate for cleaner format", array('title' => $title));
            return true;
        }
        
        // Pattern A1: Contains forum/system tags
        if (preg_match('/\[(closed|solved|duplicate|migrated|locked|answered|updated|edit|note)\]/i', $title)) {
            AIAB_Logger::debug("Title detected as AI artifact (forum tag)", array('title' => $title));
            return true;
        }
        
        // Pattern A2: Starts with first-person conversational phrases (AI talking, not titling)
        $ai_conversation_starts = array(
            '/^I\s+(appreciate|understand|need|would|will|can|cannot|don\'t|do not|think|believe|recommend|suggest|notice|see|found|have|am|was)\b/i',
            '/^(This is|That is|Here is|There is)\s+(fundamentally|not|a|an|the|my|basically|essentially)/i',
            '/^(The issue|The problem|The thing|My concern|My recommendation|My suggestion)[:is]/i',
            '/^What I\'d?\s+(recommend|suggest|say|think|do|propose)/i',
            '/^(Let me|Allow me|I\'ll|I will)\s+/i',
            '/^(Based on|According to)\s+(my|the|this|your)/i',
        );
        
        foreach ($ai_conversation_starts as $pattern) {
            if (preg_match($pattern, $title)) {
                AIAB_Logger::debug("Title detected as AI conversation (starts with AI phrase)", array('title' => $title));
                return true;
            }
        }
        
        // Pattern A3: Contains AI self-reference phrases
        $ai_self_reference = array(
            'my expertise',
            'not aligned with',
            'outside my',
            'beyond my',
            'my knowledge',
            'my understanding',
            'I cannot',
            'I can\'t',
            'I don\'t',
            'I do not',
            'expertise areas',
            'content interests',
            'persona',
            'not in content',
            'mismatched audience',
            'outside core expertise',
            'fundamentally not',
            'need to clarify',
            'let me explain',
            'I should note',
            'I must point out',
            'it\'s worth noting',
            'important to note',
        );
        
        foreach ($ai_self_reference as $phrase) {
            if (stripos($title, $phrase) !== false) {
                AIAB_Logger::debug("Title detected as AI meta-commentary (contains self-reference)", array(
                    'title' => $title,
                    'phrase' => $phrase
                ));
                return true;
            }
        }
        
        // Pattern A4: Title is just a label/header format (AI organizing its thoughts)
        if (preg_match('/^(The\s+)?(Issue|Problem|Solution|Answer|Response|Recommendation|Suggestion|Summary|Overview|Conclusion|Note|Warning|Update|Edit):\s*$/i', $title)) {
            AIAB_Logger::debug("Title detected as AI label (organizing thoughts)", array('title' => $title));
            return true;
        }
        
        // Pattern A5: Contains "100+" or similar quantifiers that suggest meta-discussion
        if (preg_match('/\d{2,}\+\s*(expertise|topics?|areas?|categories|interests?)/i', $title)) {
            AIAB_Logger::debug("Title detected as AI meta (quantifier + meta term)", array('title' => $title));
            return true;
        }
        
        // ═══ PATTERN GROUP B: TRUNCATION/INCOMPLETE DETECTION ═══
        
        // Pattern 1: Ends with JUST a number (e.g., ": 10", ": 8", "2025: 10")
        // This is the most common truncation pattern!
        if (preg_match('/:\s*\d+\s*$/', $title)) {
            AIAB_Logger::debug("Title detected as incomplete (ends with colon+number)", array('title' => $title));
            return true;
        }
        
        // Pattern 2: Ends with question mark followed by number (e.g., "Should I ? 9")
        if (preg_match('/\?\s*\d+\s*$/', $title)) {
            AIAB_Logger::debug("Title detected as incomplete (ends with ?+number)", array('title' => $title));
            return true;
        }
        
        // Pattern 3: Ends with a number followed by an adjective (missing the noun)
        // e.g., "9 Best", "10 Proven", "5 Essential"
        $incomplete_adjective_patterns = array(
            '/:\s*\d+\s+(Best|Top|Proven|Essential|Key|Easy|Simple|Quick|Smart|Effective|Powerful|Amazing|Great|Ultimate|Critical|Important|Common|Worst|Surprising)\s*$/i',
            '/\s+\d+\s+(Best|Top|Proven|Essential|Key|Easy|Simple|Quick|Smart|Effective|Powerful|Amazing|Great|Ultimate|Critical|Important|Common|Worst|Surprising)\s*$/i'
        );
        
        foreach ($incomplete_adjective_patterns as $pattern) {
            if (preg_match($pattern, $title)) {
                AIAB_Logger::debug("Title detected as incomplete (ends with number+adjective)", array('title' => $title));
                return true;
            }
        }
        
        // Pattern 4: "Number + adjective + in/for/of + year" - noun still missing!
        // e.g., "9 Best in 2025", "10 Top for 2024", "5 Essential of 2025"
        // These look complete but are actually "9 Best [THINGS] in 2025" with noun missing
        if (preg_match('/\d+\s+(Best|Top|Proven|Essential|Key|Easy|Simple|Quick|Smart|Effective|Powerful|Amazing|Great|Ultimate|Critical|Important|Common|Worst|Surprising)\s+(in|for|of)\s+\d{4}\s*$/i', $title)) {
            AIAB_Logger::debug("Title detected as incomplete (number+adjective+year, missing noun)", array('title' => $title));
            return true;
        }
        
        // Pattern 5: Ends with a standalone number after whitespace (e.g., "Gaming: 8", "Methods 5")
        // But NOT years like "2025" or "2024" which are valid endings
        if (preg_match('/\s+(\d{1,2})\s*$/', $title, $matches)) {
            $num = intval($matches[1]);
            // Single/double digit numbers (1-99) at end are likely truncated list counts
            if ($num < 100) {
                AIAB_Logger::debug("Title detected as incomplete (ends with standalone number)", array('title' => $title, 'number' => $num));
                return true;
            }
        }
        
        // Pattern 6: Ends with a colon (clearly incomplete)
        if (preg_match('/:\s*$/', $title)) {
            AIAB_Logger::debug("Title detected as incomplete (ends with colon)", array('title' => $title));
            return true;
        }
        
        // Pattern 7: Ends with articles or prepositions (clearly cut off)
        if (preg_match('/\s+(the|a|an|to|for|with|in|on|of|and|or|that|which|how)\s*$/i', $title)) {
            AIAB_Logger::debug("Title detected as incomplete (ends with article/preposition)", array('title' => $title));
            return true;
        }
        
        // Pattern 8: Ends with a dash or hyphen
        if (preg_match('/[-–—]\s*$/', $title)) {
            AIAB_Logger::debug("Title detected as incomplete (ends with dash)", array('title' => $title));
            return true;
        }
        
        // Pattern 9: Title looks like it was mid-sentence (ends with common verbs)
        if (preg_match('/\s+(is|are|can|will|should|must|do|does|have|has|get|make|use|need)\s*$/i', $title)) {
            AIAB_Logger::debug("Title detected as incomplete (ends with verb)", array('title' => $title));
            return true;
        }
        
        // Pattern 10: Ends with "I" followed by punctuation or number (broken question format)
        if (preg_match('/\s+I\s*[\?\d]\s*\d*\s*$/i', $title)) {
            AIAB_Logger::debug("Title detected as incomplete (broken question format)", array('title' => $title));
            return true;
        }
        
        // Pattern 11: Contains double punctuation (parsing artifacts)
        // e.g., "::", "??", "..", "--" (not intentional ellipsis)
        if (preg_match('/::|\?\?|\.\.(?!\.)|--/', $title)) {
            AIAB_Logger::debug("Title detected as incomplete (double punctuation artifact)", array('title' => $title));
            return true;
        }
        
        // Pattern 12: Ends with words that typically need a following noun
        // e.g., "General" (needs Guide/Overview), "Complete" (needs Guide), "Ultimate" (needs Guide)
        $needs_noun_words = array(
            'General',      // needs Guide, Overview, Tips
            'Complete',     // needs Guide, List, Overview  
            'Ultimate',     // needs Guide, List
            'Comprehensive',// needs Guide, Overview
            'Detailed',     // needs Guide, Analysis
            'Quick',        // needs Guide, Tips, Start
            'Brief',        // needs Overview, Guide
            'Full',         // needs Guide, List, Review
            'Basic',        // needs Guide, Overview, Tutorial
            'Advanced',     // needs Guide, Tutorial, Tips
            'Beginner',     // needs Guide, Tutorial
            'Expert',       // needs Guide, Tips, Advice
            'Professional', // needs Guide, Tips
            'Practical',    // needs Guide, Tips, Advice
            'Essential',    // needs Guide, Tips, List
            'Definitive',   // needs Guide, List
            'In-Depth',     // needs Guide, Analysis, Review
            'Step-by-Step', // needs Guide, Tutorial
            'Introductory', // needs Guide, Overview
        );
        
        foreach ($needs_noun_words as $word) {
            if (preg_match('/\b' . preg_quote($word, '/') . '\s*$/i', $title)) {
                AIAB_Logger::debug("Title detected as incomplete (ends with word needing noun)", array(
                    'title' => $title,
                    'word' => $word
                ));
                return true;
            }
        }
        
        // Pattern 13: Ends with incomplete compound adjective phrases (missing their noun)
        // e.g., "Virtual Private" (missing Server), "High Performance" (missing Computing)
        $incomplete_compounds = array(
            'Virtual Private',      // needs Server/Network
            'High Performance',     // needs Computing/Server
            'Open Source',          // needs Software/Project
            'Real Time',            // needs Data/Processing
            'Machine Learning',     // usually complete, but check context
            'Artificial Intelligence', // usually complete
            'Low Cost',             // needs Solution/Option
            'High Quality',         // needs Service/Product
            'Full Stack',           // needs Developer/Development
            'Cross Platform',       // needs App/Development
            'Cloud Based',          // needs Solution/Service
            'Data Driven',          // needs Approach/Strategy
            'Cost Effective',       // needs Solution/Method
            'User Friendly',        // needs Interface/Design
            'Mobile Friendly',      // needs Design/Website
            'Search Engine',        // needs Optimization/Marketing
            'Social Media',         // needs Marketing/Strategy
            'Content Management',   // needs System
            'Customer Relationship', // needs Management
            'Supply Chain',         // needs Management
            'What Is a Virtual Private', // specific broken pattern
            'What Is a',            // incomplete question
            'How to Get',           // incomplete
            'How to Use',           // incomplete  
            'How to Make',          // incomplete
            'Why You Should',       // incomplete
            'Why You Need',         // incomplete
            'What You Need',        // incomplete
            'Everything You Need',  // incomplete (needs "to Know")
        );
        
        foreach ($incomplete_compounds as $compound) {
            if (preg_match('/\b' . preg_quote($compound, '/') . '\s*$/i', $title)) {
                AIAB_Logger::debug("Title detected as incomplete (ends with incomplete compound)", array(
                    'title' => $title,
                    'compound' => $compound
                ));
                return true;
            }
        }
        
        // Pattern 14: Ends with a single common adjective (likely missing noun)
        // But only if preceded by "a" or "an" article (indicates incomplete noun phrase)
        if (preg_match('/\b(a|an)\s+(Virtual|Private|Digital|Online|Remote|Local|Global|Modern|Traditional|Basic|Advanced|Simple|Complex|New|Old|Big|Small|Fast|Slow|Good|Bad|Free|Paid|Premium|Professional|Personal|Public|Secure|Custom|Standard|Special|General|Specific|Primary|Secondary|Main|Key|Core|Essential|Critical|Important|Major|Minor|Full|Partial|Complete|Incomplete|Total|Single|Multiple|Various|Different|Similar|Same|Other|Another|Next|Last|First|Final|Initial)\s*$/i', $title)) {
            AIAB_Logger::debug("Title detected as incomplete (article + adjective, missing noun)", array('title' => $title));
            return true;
        }
        
        return false;
    }
    
    /**
     * Regenerate title based on article content and focus keyword
     * 
     * Called when the original title is detected as incomplete/truncated.
     * This creates a COMPLETE, sensible title under 60 characters.
     */
    private function regenerate_title_from_content($content, $focus_keyword, $original_title = '') {
        // Extract a content summary for context (first 300 words)
        $plain_content = strip_tags($content);
        $words = explode(' ', $plain_content);
        $content_summary = implode(' ', array_slice($words, 0, 300));
        
        // Extract H2 headings for topic context
        preg_match_all('/<h2[^>]*>([^<]+)<\/h2>/i', $content, $h2_matches);
        $h2_topics = !empty($h2_matches[1]) ? implode(', ', array_slice($h2_matches[1], 0, 5)) : '';
        
        $prompt = "Generate a NEW title for this article.\n\n";
        $prompt .= "FOCUS KEYWORD: \"{$focus_keyword}\"\n\n";
        
        if (!empty($original_title)) {
            $prompt .= "ORIGINAL TITLE (broken/incomplete): \"{$original_title}\"\n";
            $prompt .= "Create a BETTER version that makes sense.\n\n";
        }
        
        if (!empty($h2_topics)) {
            $prompt .= "ARTICLE SECTIONS: {$h2_topics}\n\n";
        }
        
        $prompt .= "ARTICLE EXCERPT:\n" . substr($content_summary, 0, 500) . "\n\n";
        
        $prompt .= "═══ STRICT REQUIREMENTS ═══\n";
        $prompt .= "1. MAXIMUM 55 characters (count them!)\n";
        $prompt .= "2. NO COLONS (:) - they cause problems\n";
        $prompt .= "3. Must be a COMPLETE phrase that makes sense\n";
        $prompt .= "4. Start with or include \"{$focus_keyword}\"\n";
        $prompt .= "5. Simple and direct - no fancy punctuation\n\n";
        
        $prompt .= "═══ GOOD EXAMPLES (no colons, complete) ═══\n";
        $prompt .= "✅ \"VPS Hosting Guide With 7 Expert Tips\"\n";
        $prompt .= "✅ \"9 Proven GPU Server Cost Saving Methods\"\n";
        $prompt .= "✅ \"How to Choose the Best Cloud Provider\"\n";
        $prompt .= "✅ \"Complete NVMe VPS Setup Tutorial 2025\"\n";
        $prompt .= "✅ \"Essential Linux Server Security Guide\"\n\n";
        
        $prompt .= "═══ BAD EXAMPLES (never do this) ═══\n";
        $prompt .= "❌ \"VPS Hosting: 7 Best\" (colon + incomplete)\n";
        $prompt .= "❌ \"GPU Servers: Expert\" (colon + cut off)\n";
        $prompt .= "❌ \"Everything You Need to Know About: 9\" (nonsense)\n\n";
        
        $prompt .= "Respond with ONLY the title. No quotes. No explanation.";
        
        AIAB_Logger::info("🔄 Regenerating incomplete title", array(
            'original' => $original_title,
            'keyword' => $focus_keyword
        ));
        
        $new_title = $this->generate_completion($prompt, 80);
        $new_title = trim($new_title, "\"'\n ");
        
        // Remove any markdown formatting that slipped through
        $new_title = preg_replace('/\*+/', '', $new_title);
        $new_title = preg_replace('/#+\s*/', '', $new_title);
        
        // Remove colons if AI still included them
        $new_title = str_replace(':', ' -', $new_title);
        $new_title = preg_replace('/\s+/', ' ', $new_title);
        $new_title = trim($new_title, ' -');
        
        // Final safety check - if still over 55, truncate smartly
        if (strlen($new_title) > 55) {
            $truncated = substr($new_title, 0, 55);
            $last_space = strrpos($truncated, ' ');
            if ($last_space > 35) {
                $new_title = substr($truncated, 0, $last_space);
            }
            
            // If still incomplete after truncation, use simple fallback
            if ($this->is_title_incomplete($new_title)) {
                AIAB_Logger::warning("Regenerated title still incomplete, using fallback", array(
                    'attempted' => $new_title
                ));
                // Fallback: Create a simple but complete title (NO COLON)
                $new_title = ucwords($focus_keyword) . ' Complete Guide';
                if (strlen($new_title) > 55) {
                    $new_title = ucwords($focus_keyword) . ' Guide';
                }
                if (strlen($new_title) > 55) {
                    // Keyword itself is too long, just truncate it
                    $new_title = substr(ucwords($focus_keyword), 0, 45) . ' Guide';
                }
            }
        }
        
        // Verify the new title is complete
        if ($this->is_title_incomplete($new_title)) {
            AIAB_Logger::warning("Regenerated title still looks incomplete, using safe fallback", array(
                'attempted' => $new_title,
                'keyword' => $focus_keyword
            ));
            // Use a guaranteed-complete fallback (NO COLONS)
            $new_title = ucwords($focus_keyword) . ' Essential Guide 2025';
            if (strlen($new_title) > 55) {
                $new_title = ucwords($focus_keyword) . ' Complete Guide';
            }
            if (strlen($new_title) > 55) {
                $new_title = ucwords($focus_keyword) . ' Guide';
            }
            if (strlen($new_title) > 55) {
                $new_title = substr(ucwords($focus_keyword), 0, 45) . ' Guide';
            }
        }
        
        AIAB_Logger::info("✅ Title regenerated successfully", array(
            'original' => $original_title,
            'new_title' => $new_title,
            'length' => strlen($new_title)
        ));
        
        return $new_title;
    }
    
    /**
     * Generate meta description (RankMath optimized)
     */
    public function generate_meta_description($title, $keyword) {
        $prompt = "Write a meta description for an article titled: \"{$title}\"\n";
        $prompt .= "Primary keyword: {$keyword}\n\n";
        $prompt .= "REQUIREMENTS FOR RANKMATH 100/100 SCORE:\n";
        $prompt .= "1. Include \"{$keyword}\" in the FIRST 120 characters - CRITICAL!\n";
        $prompt .= "2. Total length: exactly 150-160 characters (no more, no less)\n";
        $prompt .= "3. Include a call-to-action: Learn, Discover, Find out, Get, Read, Explore\n";
        $prompt .= "4. Make it compelling and relevant to searchers\n";
        $prompt .= "5. Create urgency or curiosity\n\n";
        $prompt .= "Respond with ONLY the meta description, nothing else.";
        
        $meta = $this->generate_completion($prompt, 100);
        return substr(trim($meta), 0, 160);
    }
}
