<?php
/**
 * Article Publisher
 * Handles publishing articles to WordPress
 */

if (!defined('ABSPATH')) {
    exit;
}

class AIAB_Article_Publisher {
    
    private $persona;
    
    public function __construct(AIAB_Persona $persona = null) {
        $this->persona = $persona;
    }
    
    /**
     * Publish an article to WordPress
     */
    public function publish($article_data, $article_id = null) {
        AIAB_Logger::log("Publishing article: " . $article_data['title']);
        
        $post_status = get_option('aiab_post_status', 'publish');
        $default_category = get_option('aiab_default_category', 1);
        $default_author = get_option('aiab_author_id', 1);
        
        // Use persona's author if set, otherwise use default
        $author_id = $default_author;
        if ($this->persona && $this->persona->get_wp_author_id()) {
            $author_id = $this->persona->get_wp_author_id();
        }
        
        // Use persona's categories if set, otherwise use default
        $post_categories = array($default_category);
        if ($this->persona) {
            $persona_categories = $this->persona->get_categories();
            if (!empty($persona_categories)) {
                // Randomly select one category from persona's assigned categories
                $post_categories = array($persona_categories[array_rand($persona_categories)]);
            }
        }
        
        // Get focus keyword
        $focus_keyword = isset($article_data['focus_keyword']) && !empty($article_data['focus_keyword']) 
            ? $article_data['focus_keyword'] 
            : (isset($article_data['keyword']) ? $article_data['keyword'] : '');
        
        // ═══════════════════════════════════════════════════════════════
        // CRITICAL: Ensure title STARTS with focus keyword BEFORE insert
        // This ensures both post_title and rank_math_title match
        // ═══════════════════════════════════════════════════════════════
        $optimized_title = $this->ensure_seo_title_starts_with_keyword($article_data['title'], $focus_keyword);
        $article_data['title'] = $optimized_title; // Update for later use
        
        // Prepare post data
        $post_data = array(
            'post_title' => wp_strip_all_tags($optimized_title),
            'post_content' => $article_data['content'],
            'post_excerpt' => isset($article_data['excerpt']) ? $article_data['excerpt'] : '',
            'post_status' => $post_status,
            'post_author' => $author_id,
            'post_type' => 'post',
            'post_category' => $post_categories,
        );
        
        // Generate SEO-optimized short slug from focus keyword
        $slug = $this->generate_seo_slug($article_data);
        if (!empty($slug)) {
            $post_data['post_name'] = $slug;
        }
        
        AIAB_Logger::info("Publishing article with keyword-optimized title", array(
            'original_title' => isset($article_data['title']) ? $article_data['title'] : 'N/A',
            'optimized_title' => $optimized_title,
            'focus_keyword' => $focus_keyword,
            'slug' => $slug,
            'keyword_at_start' => (stripos($optimized_title, $focus_keyword) === 0) ? 'YES' : 'NO'
        ));
        
        // Insert the post
        $post_id = wp_insert_post($post_data, true);
        
        if (is_wp_error($post_id)) {
            throw new Exception('Failed to create post: ' . $post_id->get_error_message());
        }
        
        // Set tags
        if (!empty($article_data['tags'])) {
            wp_set_post_tags($post_id, $article_data['tags']);
        }
        
        // Set featured image if provided
        if (!empty($article_data['featured_image_id'])) {
            set_post_thumbnail($post_id, $article_data['featured_image_id']);
            
            // IMPORTANT: Update image alt text to include focus keyword (RankMath requirement)
            if (!empty($focus_keyword)) {
                $this->update_image_alt_with_keyword($article_data['featured_image_id'], $focus_keyword, $article_data);
            }
        }
        
        // Set SEO meta (RankMath compatible)
        $this->set_seo_meta($post_id, $article_data);
        
        // Update article record if provided
        if ($article_id) {
            $this->update_article_record($article_id, $post_id, $article_data);
        }
        
        // Set custom meta for tracking
        update_post_meta($post_id, '_aiab_generated', 1);
        update_post_meta($post_id, '_aiab_article_id', $article_id);
        update_post_meta($post_id, '_aiab_keyword', $article_data['keyword'] ?? '');
        update_post_meta($post_id, '_aiab_persona_id', $this->persona ? $this->persona->get_id() : 0);
        update_post_meta($post_id, '_aiab_generated_at', current_time('mysql'));
        
        AIAB_Logger::log("Article published successfully", 'info', array(
            'post_id' => $post_id,
            'title' => $article_data['title']
        ));
        
        return $post_id;
    }
    
    /**
     * Generate SEO-optimized slug from focus keyword
     * 
     * CRITICAL FOR RANKMATH: The URL MUST contain the EXACT focus keyword
     * RankMath checks for exact match, so we cannot remove words like "for", "the" etc.
     * if they are part of the focus keyword.
     */
    private function generate_seo_slug($article_data) {
        // Use focus keyword if available, otherwise use main keyword
        $focus_keyword = isset($article_data['focus_keyword']) && !empty($article_data['focus_keyword'])
            ? $article_data['focus_keyword']
            : (isset($article_data['keyword']) ? $article_data['keyword'] : '');
        
        if (empty($focus_keyword)) {
            return '';
        }
        
        // IMPORTANT: Keep the focus keyword INTACT - only convert to URL format
        // DO NOT remove words like "for", "the", "a" - RankMath needs exact match!
        $slug = strtolower(trim($focus_keyword));
        
        // Only remove question prefixes at the START (these won't be in the keyword usually)
        $slug = preg_replace('/^(what is |what are |how to |how do |why is |when to |where to |who is |can you |should i )/i', '', $slug);
        
        // Remove trailing question mark
        $slug = rtrim($slug, '?');
        
        // Remove special characters EXCEPT spaces and hyphens
        $slug = preg_replace('/[^a-z0-9\s-]/', '', $slug);
        
        // Replace spaces with hyphens
        $slug = preg_replace('/\s+/', '-', trim($slug));
        
        // Remove multiple consecutive hyphens
        $slug = preg_replace('/-+/', '-', $slug);
        
        // Trim hyphens from ends
        $slug = trim($slug, '-');
        
        // Limit length but try to keep complete keyword
        if (strlen($slug) > 75) {
            $slug = substr($slug, 0, 75);
            // Don't cut in middle of a word
            $slug = preg_replace('/-[^-]*$/', '', $slug);
        }
        
        // Ensure we have something
        if (empty($slug) || strlen($slug) < 3) {
            // Fallback to sanitized focus keyword
            $slug = sanitize_title($focus_keyword);
        }
        
        AIAB_Logger::debug("Generated SEO slug from focus keyword", array(
            'focus_keyword' => $focus_keyword,
            'slug' => $slug,
            'length' => strlen($slug)
        ));
        
        return $slug;
    }
    
    /**
     * Set SEO meta fields (RankMath & Yoast compatible)
     * Optimized for RankMath 100/100 score
     */
    private function set_seo_meta($post_id, $article_data) {
        // Use focus_keyword if available, fallback to keyword
        $focus_keyword = isset($article_data['focus_keyword']) && !empty($article_data['focus_keyword']) 
            ? $article_data['focus_keyword'] 
            : (isset($article_data['keyword']) ? $article_data['keyword'] : '');
            
        $meta_description = isset($article_data['meta_description']) ? $article_data['meta_description'] : '';
        $title = isset($article_data['title']) ? $article_data['title'] : '';
        
        // ═══════════════════════════════════════════════════════════════
        // CRITICAL: Ensure SEO title STARTS with focus keyword
        // RankMath requires keyword at position 0 of SEO title
        // ═══════════════════════════════════════════════════════════════
        $seo_title = $this->ensure_seo_title_starts_with_keyword($title, $focus_keyword);
        
        AIAB_Logger::debug("Setting SEO meta for post $post_id", array(
            'focus_keyword' => $focus_keyword,
            'original_title' => $title,
            'seo_title' => $seo_title,
            'meta_description_length' => strlen($meta_description)
        ));
        
        // ALWAYS set RankMath meta fields (they work even if RankMath isn't detected yet)
        // RankMath reads these post_meta values directly
        update_post_meta($post_id, 'rank_math_focus_keyword', $focus_keyword);
        update_post_meta($post_id, 'rank_math_title', $seo_title);
        
        // Meta Description - ensure it's optimized
        $optimized_description = $this->optimize_meta_description($meta_description, $focus_keyword);
        update_post_meta($post_id, 'rank_math_description', $optimized_description);
        
        // Pillar Content flag
        update_post_meta($post_id, 'rank_math_pillar_content', 
            isset($article_data['is_pillar']) && $article_data['is_pillar'] ? 'on' : 'off');
        
        // Robots meta (index, follow)
        update_post_meta($post_id, 'rank_math_robots', array('index', 'follow'));
        
        // Cornerstone content (for pillar articles)
        if (isset($article_data['is_pillar']) && $article_data['is_pillar']) {
            update_post_meta($post_id, 'rank_math_cornerstone', 'on');
        }
        
        // Twitter/Social meta
        update_post_meta($post_id, 'rank_math_twitter_title', $seo_title);
        update_post_meta($post_id, 'rank_math_twitter_description', $optimized_description);
        
        // OpenGraph meta
        update_post_meta($post_id, 'rank_math_facebook_title', $seo_title);
        update_post_meta($post_id, 'rank_math_facebook_description', $optimized_description);
        
        // Set schema if RankMath is active (this requires RankMath API)
        // Check for RankMath with multiple detection methods
        $rankmath_active = class_exists('RankMath') 
            || class_exists('RankMath\\Helper')
            || class_exists('\\RankMath\\Helper')
            || defined('RANK_MATH_VERSION')
            || defined('RANK_MATH_FILE');
            
        if ($rankmath_active) {
            $article_type = isset($article_data['article_type']) ? $article_data['article_type'] : 'informational';
            $this->set_rankmath_schema($post_id, $article_data, $article_type);
        }
        
        AIAB_Logger::info("✅ RankMath SEO meta set for post $post_id", array(
            'focus_keyword' => $focus_keyword,
            'seo_title' => $seo_title,
            'keyword_at_start' => (stripos($seo_title, $focus_keyword) === 0) ? 'YES' : 'NO',
            'rankmath_detected' => $rankmath_active ? 'yes' : 'no (meta still set)'
        ));
        
        // Yoast SEO (fallback)
        if (defined('WPSEO_VERSION')) {
            update_post_meta($post_id, '_yoast_wpseo_focuskw', $focus_keyword);
            update_post_meta($post_id, '_yoast_wpseo_metadesc', $meta_description);
            update_post_meta($post_id, '_yoast_wpseo_title', $seo_title);
            
            // Cornerstone content
            if (isset($article_data['is_pillar']) && $article_data['is_pillar']) {
                update_post_meta($post_id, '_yoast_wpseo_is_cornerstone', '1');
            }
        }
        
        // All in One SEO (fallback)
        if (class_exists('AIOSEOP_Core')) {
            update_post_meta($post_id, '_aioseop_keywords', $focus_keyword);
            update_post_meta($post_id, '_aioseop_description', $meta_description);
            update_post_meta($post_id, '_aioseop_title', $title);
        }
        
        // Always add JSON-LD schema to post content (works without any SEO plugin)
        $this->inject_schema_markup($post_id, $article_data);
    }
    
    /**
     * Update featured image alt text to include focus keyword
     * RankMath requires focus keyword in image alt for 100/100 score
     */
    private function update_image_alt_with_keyword($image_id, $focus_keyword, $article_data) {
        if (empty($image_id) || empty($focus_keyword)) {
            return;
        }
        
        // Get current alt text
        $current_alt = get_post_meta($image_id, '_wp_attachment_image_alt', true);
        
        // Check if focus keyword is already in alt text (case insensitive)
        if (!empty($current_alt) && stripos($current_alt, $focus_keyword) !== false) {
            AIAB_Logger::debug("Image alt already contains focus keyword", array(
                'image_id' => $image_id,
                'alt' => $current_alt
            ));
            return;
        }
        
        // Build new alt text with focus keyword
        // Format: "Focus Keyword - descriptive text"
        if (!empty($current_alt)) {
            // Prepend focus keyword to existing alt
            $new_alt = ucwords($focus_keyword) . ' - ' . $current_alt;
        } elseif (!empty($article_data['alt_text'])) {
            // Use provided alt_text with focus keyword
            $new_alt = ucwords($focus_keyword) . ' - ' . $article_data['alt_text'];
        } else {
            // Generate alt from title and keyword
            $title = isset($article_data['title']) ? $article_data['title'] : '';
            $new_alt = ucwords($focus_keyword) . ' - Featured image for ' . $title;
        }
        
        // Limit alt text length (125 chars recommended for accessibility)
        if (strlen($new_alt) > 125) {
            $new_alt = substr($new_alt, 0, 122) . '...';
        }
        
        // Update the alt text
        update_post_meta($image_id, '_wp_attachment_image_alt', $new_alt);
        
        // Also update the attachment title to include keyword (helps with image SEO)
        $attachment = get_post($image_id);
        if ($attachment) {
            $new_title = ucwords($focus_keyword);
            if (!empty($article_data['title'])) {
                $new_title .= ' - ' . $article_data['title'];
            }
            
            wp_update_post(array(
                'ID' => $image_id,
                'post_title' => substr($new_title, 0, 200)
            ));
        }
        
        AIAB_Logger::info("✅ Updated image alt text with focus keyword", array(
            'image_id' => $image_id,
            'focus_keyword' => $focus_keyword,
            'new_alt' => $new_alt
        ));
    }
    
    /**
     * Optimize meta description for RankMath scoring
     * Ensures keyword in first 120 chars, length 150-160 chars
     */
    private function optimize_meta_description($description, $focus_keyword) {
        $description = trim($description);
        
        // If empty, generate one
        if (empty($description)) {
            return '';
        }
        
        // Check if focus keyword is in the description
        $keyword_pos = stripos($description, $focus_keyword);
        
        // If keyword is not in description, prepend it
        if ($keyword_pos === false) {
            // Try to naturally incorporate keyword
            $description = ucfirst($focus_keyword) . ': ' . lcfirst($description);
        } elseif ($keyword_pos > 120) {
            // Keyword is too far - try to move it earlier
            // Remove keyword from current position and prepend
            $description = preg_replace('/\b' . preg_quote($focus_keyword, '/') . '\b/i', '', $description, 1);
            $description = ucfirst($focus_keyword) . ' - ' . trim($description);
        }
        
        // Ensure length is 150-160 characters
        if (strlen($description) > 160) {
            // Truncate at word boundary
            $description = substr($description, 0, 157);
            $description = preg_replace('/\s+\S*$/', '', $description) . '...';
        } elseif (strlen($description) < 150) {
            // Too short - add a call to action if there's room
            $cta_options = array(
                ' Learn more here.',
                ' Find out now.',
                ' Discover today.',
                ' Read the full guide.'
            );
            foreach ($cta_options as $cta) {
                if (strlen($description . $cta) <= 160) {
                    $description .= $cta;
                    break;
                }
            }
        }
        
        return $description;
    }
    
    /**
     * ═══════════════════════════════════════════════════════════════
     * ENSURE SEO TITLE STARTS WITH FOCUS KEYWORD
     * ═══════════════════════════════════════════════════════════════
     * 
     * RankMath REQUIRES the focus keyword to appear at the BEGINNING
     * of the SEO title (position 0). This is a strict requirement.
     * 
     * This is the FINAL enforcement - guaranteed to work.
     */
    private function ensure_seo_title_starts_with_keyword($title, $focus_keyword) {
        if (empty($focus_keyword) || empty($title)) {
            return $title;
        }
        
        $keyword_lower = strtolower(trim($focus_keyword));
        $title_lower = strtolower($title);
        
        // ═══ RULE 1: If title already STARTS with keyword - perfect, return as-is ═══
        if (strpos($title_lower, $keyword_lower) === 0) {
            AIAB_Logger::debug("SEO title already starts with keyword", array(
                'title' => $title,
                'keyword' => $focus_keyword
            ));
            return $this->clean_title($title);
        }
        
        // ═══ RULE 2: If keyword is in first 50% of title - good enough for SEO ═══
        $keyword_pos = stripos($title_lower, $keyword_lower);
        $halfway = strlen($title) / 2;
        
        if ($keyword_pos !== false && $keyword_pos < $halfway) {
            AIAB_Logger::debug("SEO title has keyword in first half - acceptable", array(
                'title' => $title,
                'keyword' => $focus_keyword,
                'position' => $keyword_pos
            ));
            return $this->clean_title($title);
        }
        
        // ═══ RULE 3: Keyword is missing or too far back - need to restructure ═══
        $ucKeyword = ucwords($focus_keyword);
        
        // For very long keywords (>25 chars), use abbreviated version
        if (strlen($ucKeyword) > 25) {
            // Take first 2-3 significant words
            $keyword_words = explode(' ', $ucKeyword);
            if (count($keyword_words) > 2) {
                $ucKeyword = implode(' ', array_slice($keyword_words, 0, 3));
            }
        }
        
        // Clean up the original title - remove colons and awkward patterns
        $clean_title = $title;
        $clean_title = preg_replace('/\s*:\s*:/', ':', $clean_title); // Fix double colons
        $clean_title = preg_replace('/\s*:\s*$/', '', $clean_title); // Remove trailing colon
        $clean_title = preg_replace('/^\s*:\s*/', '', $clean_title); // Remove leading colon
        $clean_title = trim($clean_title, ':- ');
        
        // Remove keyword from title if it exists (to avoid duplication)
        if ($keyword_pos !== false) {
            $clean_title = preg_replace('/\b' . preg_quote($focus_keyword, '/') . '\b/i', '', $clean_title, 1);
            $clean_title = preg_replace('/\s+/', ' ', $clean_title);
            $clean_title = trim($clean_title, ':- ');
        }
        
        // Extract useful parts from the original title
        // Look for numbers and power words to preserve
        $has_number = preg_match('/\b(\d+)\b/', $clean_title, $number_match);
        $number = $has_number ? $number_match[1] : '';
        
        $power_words = array('Essential', 'Ultimate', 'Best', 'Proven', 'Effective', 'Complete', 
                            'Critical', 'Key', 'Top', 'Simple', 'Quick', 'Easy', 'Expert',
                            'Powerful', 'Smart', 'Dangerous', 'Common', 'Costly', 'Surprising');
        $found_power = '';
        foreach ($power_words as $pw) {
            if (stripos($clean_title, $pw) !== false) {
                $found_power = $pw;
                break;
            }
        }
        
        // Build new title - prioritize making sense over exact format
        $max_length = 60;
        $new_title = '';
        
        // Strategy 1: "Number Power Keyword Tips/Ways/Methods"
        if ($number && $found_power && strlen($ucKeyword) <= 35) {
            $suffix_options = array('Tips', 'Ways', 'Methods', 'Steps', 'Strategies', 'Solutions');
            $suffix = $suffix_options[array_rand($suffix_options)];
            $new_title = "{$number} {$found_power} {$ucKeyword} {$suffix}";
        }
        // Strategy 2: "Keyword: Number Power Tips"
        elseif ($number && strlen($ucKeyword) <= 30) {
            $found_power = $found_power ?: 'Essential';
            $new_title = "{$ucKeyword}: {$number} {$found_power} Tips";
        }
        // Strategy 3: "Keyword Guide" for very long keywords
        elseif (strlen($ucKeyword) > 30) {
            $new_title = "{$ucKeyword} Guide";
        }
        // Strategy 4: Simple "Keyword: Descriptive Phrase"
        else {
            // Get a short meaningful phrase from original
            $short_phrase = $this->extract_meaningful_phrase($clean_title, $max_length - strlen($ucKeyword) - 2);
            if (!empty($short_phrase)) {
                $new_title = "{$ucKeyword}: {$short_phrase}";
            } else {
                $new_title = "{$ucKeyword}: Complete Guide";
            }
        }
        
        // Final cleanup and length check
        $new_title = $this->clean_title($new_title);
        
        // If still too long, simplify further
        if (strlen($new_title) > $max_length) {
            if ($number) {
                $new_title = "{$number} {$ucKeyword} Tips";
            } else {
                $new_title = "{$ucKeyword} Guide";
            }
            $new_title = substr($new_title, 0, $max_length);
        }
        
        AIAB_Logger::info("✅ Restructured SEO title", array(
            'original' => $title,
            'fixed' => $new_title,
            'keyword' => $focus_keyword
        ));
        
        return $new_title;
    }
    
    /**
     * Clean up title - fix double colons, trim, proper casing
     */
    private function clean_title($title) {
        // Fix double colons
        $title = preg_replace('/\s*:\s*:\s*/', ': ', $title);
        // Fix colon with no space after
        $title = preg_replace('/:\s*(\w)/', ': $1', $title);
        // Remove trailing punctuation except ?
        $title = preg_replace('/[:.]+$/', '', $title);
        // Fix multiple spaces
        $title = preg_replace('/\s+/', ' ', $title);
        return trim($title);
    }
    
    /**
     * Extract a meaningful phrase from text that fits within max length
     */
    private function extract_meaningful_phrase($text, $max_length) {
        if (empty($text) || $max_length < 10) {
            return '';
        }
        
        // Remove common filler words at start
        $text = preg_replace('/^(how to|what is|why|when|the|a|an)\s+/i', '', $text);
        $text = trim($text);
        
        if (strlen($text) <= $max_length) {
            return ucfirst($text);
        }
        
        // Cut at word boundary
        $short = substr($text, 0, $max_length);
        $short = preg_replace('/\s+\S*$/', '', $short);
        
        // Don't return fragments that are too short
        if (strlen($short) < 10) {
            return '';
        }
        
        return ucfirst(trim($short));
    }
    
    /**
     * Update article record in our database
     */
    private function update_article_record($article_id, $post_id, $article_data) {
        global $wpdb;
        $table = AIAB_Database::get_table('articles');
        
        $wpdb->update(
            $table,
            array(
                'wp_post_id' => $post_id,
                'title' => $article_data['title'],
                'slug' => get_post_field('post_name', $post_id),
                'content' => $article_data['content'],
                'excerpt' => isset($article_data['excerpt']) ? $article_data['excerpt'] : '',
                'meta_description' => isset($article_data['meta_description']) ? $article_data['meta_description'] : '',
                'word_count' => isset($article_data['word_count']) ? $article_data['word_count'] : str_word_count(strip_tags($article_data['content'])),
                'status' => 'published',
                'published_at' => current_time('mysql')
            ),
            array('id' => $article_id)
        );
    }
    
    /**
     * Update article content (for adding internal links)
     */
    public function update_article_content($post_id, $new_content) {
        $result = wp_update_post(array(
            'ID' => $post_id,
            'post_content' => $new_content
        ), true);
        
        if (is_wp_error($result)) {
            throw new Exception('Failed to update post: ' . $result->get_error_message());
        }
        
        return true;
    }
    
    /**
     * Bulk publish articles from a sphere
     */
    public function publish_sphere(AIAB_Thought_Sphere $sphere) {
        $articles = $sphere->get_articles('written');
        $published = array();
        
        foreach ($articles as $article) {
            try {
                $article_data = array(
                    'title' => $article->title,
                    'content' => $article->content,
                    'excerpt' => $article->excerpt,
                    'meta_description' => $article->meta_description,
                    'keyword' => $article->keyword,
                    'tags' => json_decode($article->internal_links, true) ?: array(),
                    'is_pillar' => (bool) $article->is_pillar,
                    'featured_image_id' => $article->featured_image_id,
                    'word_count' => $article->word_count,
                    'article_type' => $article->article_type ?? 'informational'
                );
                
                $post_id = $this->publish($article_data, $article->id);
                $published[] = $post_id;
                
                $sphere->increment_published();
                
            } catch (Exception $e) {
                AIAB_Logger::log("Failed to publish article: " . $e->getMessage(), 'error', array(
                    'article_id' => $article->id
                ));
            }
        }
        
        return $published;
    }
    
    /**
     * Schedule a post for future publication
     */
    public function schedule_post($article_data, $publish_date, $article_id = null) {
        $article_data['post_status'] = 'future';
        
        $post_data = array(
            'post_title' => wp_strip_all_tags($article_data['title']),
            'post_content' => $article_data['content'],
            'post_excerpt' => isset($article_data['excerpt']) ? $article_data['excerpt'] : '',
            'post_status' => 'future',
            'post_date' => $publish_date,
            'post_date_gmt' => get_gmt_from_date($publish_date),
            'post_author' => get_option('aiab_author_id', 1),
            'post_type' => 'post',
        );
        
        $post_id = wp_insert_post($post_data, true);
        
        if (is_wp_error($post_id)) {
            throw new Exception('Failed to schedule post: ' . $post_id->get_error_message());
        }
        
        // Set meta
        $this->set_seo_meta($post_id, $article_data);
        
        if ($article_id) {
            global $wpdb;
            $table = AIAB_Database::get_table('articles');
            $wpdb->update(
                $table,
                array('wp_post_id' => $post_id, 'status' => 'scheduled'),
                array('id' => $article_id)
            );
        }
        
        return $post_id;
    }
    
    /**
     * Get category by name or create it
     */
    public function get_or_create_category($category_name) {
        $category = get_category_by_slug(sanitize_title($category_name));
        
        if ($category) {
            return $category->term_id;
        }
        
        $result = wp_insert_term($category_name, 'category');
        
        if (is_wp_error($result)) {
            return get_option('default_category', 1);
        }
        
        return $result['term_id'];
    }
    
    /**
     * Set RankMath schema based on article type
     */
    private function set_rankmath_schema($post_id, $article_data, $article_type) {
        // RankMath stores schema type in rank_math_rich_snippet
        $schema_type = $this->get_schema_type_for_article($article_type);
        
        // Only set the schema TYPE - let RankMath generate the actual schema data
        // Setting rank_math_schema_Article with our own format causes JavaScript errors
        // because RankMath expects specific internal properties like 'isPrimary', 'metadata', etc.
        update_post_meta($post_id, 'rank_math_rich_snippet', $schema_type);
        
        // Set type-specific hints that RankMath uses
        $title = isset($article_data['title']) ? $article_data['title'] : '';
        
        switch ($schema_type) {
            case 'howto':
                update_post_meta($post_id, 'rank_math_snippet_howto_name', $title);
                break;
            case 'faq':
                // RankMath auto-extracts FAQs from content
                break;
            case 'review':
                update_post_meta($post_id, 'rank_math_snippet_review_worst', '1');
                update_post_meta($post_id, 'rank_math_snippet_review_best', '5');
                break;
        }
        
        // DO NOT set rank_math_schema_Article - let RankMath build it
    }
    
    /**
     * Get appropriate schema type for article type
     */
    private function get_schema_type_for_article($article_type) {
        $schema_map = array(
            'how_to' => 'howto',
            'question' => 'faq',
            'review' => 'review',
            'listicle' => 'article',
            'comparison' => 'article',
            'problem_solution' => 'article',
            'local_seo' => 'article',
            'cost' => 'article',
            'glossary' => 'article',
            'buyers_guide' => 'article',
            'case_study' => 'article',
            'seasonal' => 'article',
            'guide' => 'article',
            'pillar' => 'article',
            'supporting' => 'article',
            'informational' => 'article',
        );
        
        return isset($schema_map[$article_type]) ? $schema_map[$article_type] : 'article';
    }
    
    /**
     * Inject JSON-LD schema markup into post meta (works without SEO plugin)
     */
    private function inject_schema_markup($post_id, $article_data) {
        $article_type = isset($article_data['article_type']) ? $article_data['article_type'] : 'informational';
        $schema = $this->generate_schema_json($post_id, $article_data, $article_type);
        
        if (!empty($schema)) {
            update_post_meta($post_id, '_aiab_schema_json', $schema);
        }
    }
    
    /**
     * Generate JSON-LD schema based on article type
     */
    private function generate_schema_json($post_id, $article_data, $article_type) {
        $site_name = get_bloginfo('name');
        $site_url = home_url();
        $post_url = get_permalink($post_id);
        
        // Use persona's author if available, otherwise use default
        $author_id = get_option('aiab_author_id', 1);
        if ($this->persona && $this->persona->get_wp_author_id()) {
            $author_id = $this->persona->get_wp_author_id();
        }
        $author_name = get_the_author_meta('display_name', $author_id);
        
        $published_date = get_the_date('c', $post_id);
        $modified_date = get_the_modified_date('c', $post_id);
        
        $base_schema = array(
            '@context' => 'https://schema.org',
            'headline' => $article_data['title'],
            'description' => isset($article_data['meta_description']) ? $article_data['meta_description'] : '',
            'author' => array(
                '@type' => 'Person',
                'name' => $author_name,
            ),
            'publisher' => array(
                '@type' => 'Organization',
                'name' => $site_name,
                'url' => $site_url,
            ),
            'datePublished' => $published_date,
            'dateModified' => $modified_date,
            'mainEntityOfPage' => array(
                '@type' => 'WebPage',
                '@id' => $post_url,
            ),
        );
        
        // Generate type-specific schema
        switch ($article_type) {
            case 'how_to':
                $schema = $this->generate_howto_schema($base_schema, $article_data);
                break;
                
            case 'question':
                $schema = $this->generate_faq_schema($base_schema, $article_data);
                break;
                
            case 'review':
                $schema = $this->generate_review_schema($base_schema, $article_data);
                break;
                
            case 'listicle':
                $schema = $this->generate_itemlist_schema($base_schema, $article_data);
                break;
                
            default:
                $schema = array_merge(array('@type' => 'Article'), $base_schema);
                break;
        }
        
        return json_encode($schema, JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT);
    }
    
    /**
     * Generate HowTo schema for how-to articles
     */
    private function generate_howto_schema($base_schema, $article_data) {
        $schema = array_merge(array('@type' => 'HowTo'), $base_schema);
        $schema['name'] = $article_data['title'];
        
        // Extract steps from content if possible
        $content = isset($article_data['content']) ? $article_data['content'] : '';
        $steps = $this->extract_steps_from_content($content);
        
        if (!empty($steps)) {
            $schema['step'] = $steps;
        }
        
        return $schema;
    }
    
    /**
     * Generate FAQ schema for Q&A articles
     */
    private function generate_faq_schema($base_schema, $article_data) {
        $schema = array(
            '@context' => 'https://schema.org',
            '@type' => 'FAQPage',
            'mainEntity' => array()
        );
        
        // Extract questions from content
        $content = isset($article_data['content']) ? $article_data['content'] : '';
        $faqs = $this->extract_faqs_from_content($content);
        
        if (!empty($faqs)) {
            $schema['mainEntity'] = $faqs;
        } else {
            // Fallback: use title as question
            $schema['mainEntity'][] = array(
                '@type' => 'Question',
                'name' => $article_data['title'],
                'acceptedAnswer' => array(
                    '@type' => 'Answer',
                    'text' => isset($article_data['excerpt']) ? $article_data['excerpt'] : substr(strip_tags($content), 0, 300)
                )
            );
        }
        
        return $schema;
    }
    
    /**
     * Generate Review schema for review articles
     */
    private function generate_review_schema($base_schema, $article_data) {
        $schema = array_merge(array('@type' => 'Review'), $base_schema);
        $schema['itemReviewed'] = array(
            '@type' => 'Thing',
            'name' => $article_data['keyword']
        );
        $schema['reviewRating'] = array(
            '@type' => 'Rating',
            'ratingValue' => '4',
            'bestRating' => '5',
            'worstRating' => '1'
        );
        
        return $schema;
    }
    
    /**
     * Generate ItemList schema for listicles
     */
    private function generate_itemlist_schema($base_schema, $article_data) {
        $schema = array_merge(array('@type' => 'Article'), $base_schema);
        
        // Add ItemList for the list items
        $content = isset($article_data['content']) ? $article_data['content'] : '';
        $items = $this->extract_list_items_from_content($content);
        
        if (!empty($items)) {
            $schema['mainEntity'] = array(
                '@type' => 'ItemList',
                'itemListElement' => $items
            );
        }
        
        return $schema;
    }
    
    /**
     * Extract steps from how-to content
     */
    private function extract_steps_from_content($content) {
        $steps = array();
        
        // Look for numbered lists or H2/H3 headings
        preg_match_all('/<h[23][^>]*>(?:(?:Step\s*)?(\d+)[\.\:\)]?\s*)?(.+?)<\/h[23]>/is', $content, $matches, PREG_SET_ORDER);
        
        $position = 1;
        foreach ($matches as $match) {
            $heading_text = strip_tags($match[2]);
            
            // Skip TOC heading
            if (stripos($heading_text, 'table of contents') !== false) {
                continue;
            }
            
            $steps[] = array(
                '@type' => 'HowToStep',
                'position' => $position,
                'name' => $heading_text,
                'text' => $heading_text
            );
            $position++;
        }
        
        return array_slice($steps, 0, 10); // Limit to 10 steps
    }
    
    /**
     * Extract FAQs from content (questions in H2/H3)
     */
    private function extract_faqs_from_content($content) {
        $faqs = array();
        
        // Look for question headings (start with who, what, when, where, why, how, can, does, is)
        preg_match_all('/<h[23][^>]*>([^<]*\?)<\/h[23]>\s*<p>([^<]+)/is', $content, $matches, PREG_SET_ORDER);
        
        foreach ($matches as $match) {
            $question = strip_tags(trim($match[1]));
            $answer = strip_tags(trim($match[2]));
            
            if (!empty($question) && !empty($answer)) {
                $faqs[] = array(
                    '@type' => 'Question',
                    'name' => $question,
                    'acceptedAnswer' => array(
                        '@type' => 'Answer',
                        'text' => substr($answer, 0, 500)
                    )
                );
            }
        }
        
        return array_slice($faqs, 0, 10); // Limit to 10 FAQs
    }
    
    /**
     * Extract list items from content for listicles
     */
    private function extract_list_items_from_content($content) {
        $items = array();
        
        // Look for H2/H3 headings that might be list items
        preg_match_all('/<h[23][^>]*>(?:(\d+)[\.\)]\s*)?(.+?)<\/h[23]>/is', $content, $matches, PREG_SET_ORDER);
        
        $position = 1;
        foreach ($matches as $match) {
            $item_text = strip_tags($match[2]);
            
            // Skip TOC heading
            if (stripos($item_text, 'table of contents') !== false || 
                stripos($item_text, 'conclusion') !== false ||
                stripos($item_text, 'introduction') !== false) {
                continue;
            }
            
            $items[] = array(
                '@type' => 'ListItem',
                'position' => $position,
                'name' => $item_text
            );
            $position++;
        }
        
        return array_slice($items, 0, 15); // Limit to 15 items
    }
}
