<?php
/**
 * Database Handler
 * Creates and manages custom tables for Eternal Auto Blogger
 */

if (!defined('ABSPATH')) {
    exit;
}

class AIAB_Database {
    
    /**
     * Create all required tables
     */
    public static function create_tables() {
        global $wpdb;
        
        $charset_collate = $wpdb->get_charset_collate();
        
        // MySQL 5.5 compatible - no DEFAULT CURRENT_TIMESTAMP
        // Use varchar(191) for indexed fields (191*4=764 bytes < 767 byte limit with utf8mb4)
        
        // Personas table - The AI Authors
        $table_personas = $wpdb->prefix . 'aiab_personas';
        $sql_personas = "CREATE TABLE $table_personas (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            name varchar(255) NOT NULL,
            slug varchar(191) NOT NULL,
            wp_author_id bigint(20) unsigned DEFAULT NULL,
            categories longtext,
            interests longtext,
            passions longtext,
            expertise_areas longtext,
            localities longtext,
            voice_style varchar(100) DEFAULT 'professional',
            tone varchar(100) DEFAULT 'informative',
            knowledge_sources longtext,
            writing_guidelines longtext,
            avoid_topics longtext,
            personal_story longtext,
            target_audience varchar(255),
            language varchar(50) DEFAULT 'English',
            is_active tinyint(1) DEFAULT 1,
            total_articles int(11) DEFAULT 0,
            total_spheres int(11) DEFAULT 0,
            created_at datetime DEFAULT NULL,
            updated_at datetime DEFAULT NULL,
            PRIMARY KEY (id),
            UNIQUE KEY slug (slug),
            KEY is_active (is_active),
            KEY wp_author_id (wp_author_id)
        ) $charset_collate;";
        
        // Thought Spheres table - Article clusters
        $table_spheres = $wpdb->prefix . 'aiab_thought_spheres';
        $sql_spheres = "CREATE TABLE $table_spheres (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            persona_id bigint(20) unsigned NOT NULL,
            pillar_keyword varchar(255) NOT NULL,
            pillar_title varchar(255),
            search_volume int(11) DEFAULT 0,
            difficulty_score int(11) DEFAULT 0,
            status varchar(50) DEFAULT 'planned',
            phase varchar(50) DEFAULT 'research',
            total_articles int(11) DEFAULT 7,
            published_articles int(11) DEFAULT 0,
            sphere_data longtext,
            research_data longtext,
            error_log longtext,
            started_at datetime DEFAULT NULL,
            completed_at datetime DEFAULT NULL,
            created_at datetime DEFAULT NULL,
            updated_at datetime DEFAULT NULL,
            PRIMARY KEY (id),
            KEY persona_id (persona_id),
            KEY status (status),
            KEY pillar_keyword (pillar_keyword(191)),
            KEY updated_at (updated_at)
        ) $charset_collate;";
        
        // Articles table - Individual articles in spheres
        $table_articles = $wpdb->prefix . 'aiab_articles';
        $sql_articles = "CREATE TABLE $table_articles (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            sphere_id bigint(20) unsigned NOT NULL,
            persona_id bigint(20) unsigned NOT NULL,
            wp_post_id bigint(20) unsigned DEFAULT NULL,
            article_type varchar(50) NOT NULL,
            is_pillar tinyint(1) DEFAULT 0,
            keyword varchar(255) NOT NULL,
            title varchar(255),
            slug varchar(255),
            content longtext,
            excerpt text,
            meta_description varchar(160),
            featured_image_id bigint(20) unsigned DEFAULT NULL,
            featured_image_alt varchar(255),
            internal_links longtext,
            external_links longtext,
            word_count int(11) DEFAULT 0,
            seo_score int(11) DEFAULT 0,
            status varchar(50) DEFAULT 'planned',
            retry_count int(11) DEFAULT 0,
            last_error varchar(500) DEFAULT NULL,
            published_at datetime DEFAULT NULL,
            created_at datetime DEFAULT NULL,
            updated_at datetime DEFAULT NULL,
            PRIMARY KEY (id),
            KEY sphere_id (sphere_id),
            KEY persona_id (persona_id),
            KEY wp_post_id (wp_post_id),
            KEY is_pillar (is_pillar),
            KEY status (status),
            KEY updated_at (updated_at)
        ) $charset_collate;";
        
        // Topic History - Prevents duplicates
        $table_history = $wpdb->prefix . 'aiab_topic_history';
        $sql_history = "CREATE TABLE $table_history (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            persona_id bigint(20) unsigned NOT NULL,
            keyword varchar(255) NOT NULL,
            keyword_hash varchar(32) NOT NULL,
            sphere_id bigint(20) unsigned DEFAULT NULL,
            written_at datetime DEFAULT NULL,
            PRIMARY KEY (id),
            UNIQUE KEY persona_keyword (persona_id, keyword_hash),
            KEY persona_id (persona_id),
            KEY keyword_hash (keyword_hash)
        ) $charset_collate;";
        
        // Research Cache - Stores API results
        $table_research = $wpdb->prefix . 'aiab_research_cache';
        $sql_research = "CREATE TABLE $table_research (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            query_hash varchar(32) NOT NULL,
            query_text varchar(191) NOT NULL,
            source varchar(50) NOT NULL,
            results longtext,
            expires_at datetime DEFAULT NULL,
            created_at datetime DEFAULT NULL,
            PRIMARY KEY (id),
            UNIQUE KEY query_hash (query_hash),
            KEY expires_at (expires_at)
        ) $charset_collate;";
        
        // Activity Log
        $table_log = $wpdb->prefix . 'aiab_log';
        $sql_log = "CREATE TABLE $table_log (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            level varchar(20) DEFAULT 'info',
            message text NOT NULL,
            context longtext,
            persona_id bigint(20) unsigned DEFAULT NULL,
            sphere_id bigint(20) unsigned DEFAULT NULL,
            created_at datetime DEFAULT NULL,
            PRIMARY KEY (id),
            KEY level (level),
            KEY persona_id (persona_id),
            KEY created_at (created_at)
        ) $charset_collate;";
        
        // Queue table for async processing
        $table_queue = $wpdb->prefix . 'aiab_queue';
        $sql_queue = "CREATE TABLE $table_queue (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            job_type varchar(50) NOT NULL,
            job_data longtext NOT NULL,
            priority int(11) DEFAULT 10,
            attempts int(11) DEFAULT 0,
            max_attempts int(11) DEFAULT 3,
            status varchar(20) DEFAULT 'pending',
            error_message text,
            scheduled_at datetime DEFAULT NULL,
            started_at datetime DEFAULT NULL,
            completed_at datetime DEFAULT NULL,
            created_at datetime DEFAULT NULL,
            PRIMARY KEY (id),
            KEY status (status),
            KEY priority (priority),
            KEY scheduled_at (scheduled_at)
        ) $charset_collate;";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        
        dbDelta($sql_personas);
        dbDelta($sql_spheres);
        dbDelta($sql_articles);
        dbDelta($sql_history);
        dbDelta($sql_research);
        dbDelta($sql_log);
        dbDelta($sql_queue);
        
        // Store database version
        update_option('aiab_db_version', AIAB_VERSION);
    }
    
    /**
     * Get table name with prefix
     */
    public static function get_table($table) {
        global $wpdb;
        return $wpdb->prefix . 'aiab_' . $table;
    }
    
    /**
     * Check if topic already written
     */
    public static function is_topic_written($persona_id, $keyword) {
        global $wpdb;
        $table = self::get_table('topic_history');
        $hash = md5(strtolower(trim($keyword)));
        
        $exists = $wpdb->get_var($wpdb->prepare(
            "SELECT id FROM $table WHERE persona_id = %d AND keyword_hash = %s",
            $persona_id,
            $hash
        ));
        
        return !empty($exists);
    }
    
    /**
     * Record written topic
     */
    public static function record_topic($persona_id, $keyword, $sphere_id = null) {
        global $wpdb;
        $table = self::get_table('topic_history');
        
        return $wpdb->insert($table, array(
            'persona_id' => $persona_id,
            'keyword' => $keyword,
            'keyword_hash' => md5(strtolower(trim($keyword))),
            'sphere_id' => $sphere_id
        ));
    }
    
    /**
     * Get cached research
     */
    public static function get_research_cache($query, $source) {
        global $wpdb;
        $table = self::get_table('research_cache');
        $hash = md5($query . $source);
        
        $result = $wpdb->get_row($wpdb->prepare(
            "SELECT results FROM $table WHERE query_hash = %s AND expires_at > NOW()",
            $hash
        ));
        
        return $result ? json_decode($result->results, true) : null;
    }
    
    /**
     * Set research cache
     */
    public static function set_research_cache($query, $source, $results, $hours = 24) {
        global $wpdb;
        $table = self::get_table('research_cache');
        $hash = md5($query . $source);
        
        $wpdb->replace($table, array(
            'query_hash' => $hash,
            'query_text' => $query,
            'source' => $source,
            'results' => json_encode($results),
            'expires_at' => date('Y-m-d H:i:s', strtotime("+{$hours} hours"))
        ));
    }
    
    /**
     * Clean expired cache
     */
    public static function clean_expired_cache() {
        global $wpdb;
        $table = self::get_table('research_cache');
        $wpdb->query("DELETE FROM $table WHERE expires_at < NOW()");
    }
    
    /**
     * Add job to queue
     */
    public static function add_to_queue($job_type, $job_data, $priority = 10) {
        global $wpdb;
        $table = self::get_table('queue');
        
        return $wpdb->insert($table, array(
            'job_type' => $job_type,
            'job_data' => json_encode($job_data),
            'priority' => $priority,
            'status' => 'pending'
        ));
    }
    
    /**
     * Get next queue job
     */
    public static function get_next_job() {
        global $wpdb;
        $table = self::get_table('queue');
        
        return $wpdb->get_row(
            "SELECT * FROM $table 
             WHERE status = 'pending' 
             AND scheduled_at <= NOW() 
             AND attempts < max_attempts 
             ORDER BY priority ASC, created_at ASC 
             LIMIT 1"
        );
    }
    
    /**
     * Update job status
     */
    public static function update_job($id, $status, $error = null) {
        global $wpdb;
        $table = self::get_table('queue');
        
        $data = array('status' => $status);
        
        if ($status === 'processing') {
            $data['started_at'] = current_time('mysql');
            $data['attempts'] = $wpdb->get_var($wpdb->prepare(
                "SELECT attempts FROM $table WHERE id = %d", $id
            )) + 1;
        } elseif ($status === 'completed') {
            $data['completed_at'] = current_time('mysql');
        } elseif ($status === 'failed') {
            $data['error_message'] = $error;
        }
        
        return $wpdb->update($table, $data, array('id' => $id));
    }
    
    /**
     * Run database migrations for existing installations
     */
    public static function run_migrations() {
        global $wpdb;
        $table_personas = self::get_table('personas');
        
        // Check if wp_author_id column exists
        $column_exists = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = %s AND TABLE_NAME = %s AND COLUMN_NAME = 'wp_author_id'",
            DB_NAME,
            $table_personas
        ));
        
        if (empty($column_exists)) {
            // Add wp_author_id column
            $wpdb->query("ALTER TABLE $table_personas ADD COLUMN wp_author_id bigint(20) unsigned DEFAULT NULL AFTER slug");
            $wpdb->query("ALTER TABLE $table_personas ADD INDEX wp_author_id (wp_author_id)");
            AIAB_Logger::info("Migration: Added wp_author_id column to personas table");
        }
        
        // Check if categories column exists
        $column_exists = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = %s AND TABLE_NAME = %s AND COLUMN_NAME = 'categories'",
            DB_NAME,
            $table_personas
        ));
        
        if (empty($column_exists)) {
            // Add categories column
            $wpdb->query("ALTER TABLE $table_personas ADD COLUMN categories longtext AFTER wp_author_id");
            AIAB_Logger::info("Migration: Added categories column to personas table");
        }
        
        // Check if localities column exists (added in v1.8.0)
        $column_exists = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = %s AND TABLE_NAME = %s AND COLUMN_NAME = 'localities'",
            DB_NAME,
            $table_personas
        ));
        
        if (empty($column_exists)) {
            $wpdb->query("ALTER TABLE $table_personas ADD COLUMN localities longtext AFTER expertise_areas");
            AIAB_Logger::info("Migration: Added localities column to personas table");
        }
        
        // Check if personal_story column exists
        $column_exists = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = %s AND TABLE_NAME = %s AND COLUMN_NAME = 'personal_story'",
            DB_NAME,
            $table_personas
        ));
        
        if (empty($column_exists)) {
            $wpdb->query("ALTER TABLE $table_personas ADD COLUMN personal_story longtext AFTER avoid_topics");
            AIAB_Logger::info("Migration: Added personal_story column to personas table");
        }
        
        // Check if target_audience column exists
        $column_exists = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = %s AND TABLE_NAME = %s AND COLUMN_NAME = 'target_audience'",
            DB_NAME,
            $table_personas
        ));
        
        if (empty($column_exists)) {
            $wpdb->query("ALTER TABLE $table_personas ADD COLUMN target_audience varchar(255) AFTER personal_story");
            AIAB_Logger::info("Migration: Added target_audience column to personas table");
        }
        
        // v1.7.5: Add updated_at to thought_spheres table
        $table_spheres = self::get_table('thought_spheres');
        $column_exists = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = %s AND TABLE_NAME = %s AND COLUMN_NAME = 'updated_at'",
            DB_NAME,
            $table_spheres
        ));
        
        if (empty($column_exists)) {
            $wpdb->query("ALTER TABLE $table_spheres ADD COLUMN updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP AFTER created_at");
            $wpdb->query("ALTER TABLE $table_spheres ADD INDEX updated_at (updated_at)");
            // Initialize existing rows
            $wpdb->query("UPDATE $table_spheres SET updated_at = created_at WHERE updated_at IS NULL");
            AIAB_Logger::info("Migration: Added updated_at column to thought_spheres table");
        }
        
        // v1.7.5: Add updated_at to articles table
        $table_articles = self::get_table('articles');
        $column_exists = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = %s AND TABLE_NAME = %s AND COLUMN_NAME = 'updated_at'",
            DB_NAME,
            $table_articles
        ));
        
        if (empty($column_exists)) {
            $wpdb->query("ALTER TABLE $table_articles ADD COLUMN updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP AFTER created_at");
            $wpdb->query("ALTER TABLE $table_articles ADD INDEX updated_at (updated_at)");
            // Initialize existing rows
            $wpdb->query("UPDATE $table_articles SET updated_at = created_at WHERE updated_at IS NULL");
            AIAB_Logger::info("Migration: Added updated_at column to articles table");
        }
    }
}
