<?php
/**
 * Image Generator
 * Generates featured images using AI image APIs
 */

if (!defined('ABSPATH')) {
    exit;
}

class AIAB_Image_Generator {
    
    private $provider;
    private $api_key;
    
    public function __construct() {
        $this->provider = get_option('aiab_image_provider', 'dalle');
        
        if ($this->provider === 'dalle') {
            $this->api_key = get_option('aiab_openai_api_key', '');
        } elseif ($this->provider === 'stability') {
            $this->api_key = get_option('aiab_stability_api_key', '');
        } else {
            $this->api_key = ''; // Disabled
        }
    }
    
    /**
     * Generate a featured image for an article
     */
    public function generate($title, $keyword, $alt_text = '') {
        // Check if image generation is disabled
        if ($this->provider === 'none') {
            AIAB_Logger::debug("Image generation disabled in settings");
            return null;
        }
        
        if (empty($this->api_key)) {
            $provider_name = $this->provider === 'dalle' ? 'OpenAI' : 'Stability AI';
            AIAB_Logger::warning("Image generation skipped: {$provider_name} API key not configured", array(
                'provider' => $this->provider,
                'hint' => "Add your {$provider_name} API key in Eternal Blogger → Settings → API Keys"
            ));
            return null;
        }
        
        AIAB_Logger::info("Generating featured image for: $title", array(
            'provider' => $this->provider,
            'keyword' => $keyword
        ));
        
        // Build the image prompt
        $prompt = $this->build_prompt($title, $keyword);
        
        try {
            if ($this->provider === 'dalle') {
                $image_url = $this->generate_dalle($prompt);
            } else {
                $image_url = $this->generate_stability($prompt);
            }
            
            if ($image_url) {
                // Download and add to media library
                $attachment_id = $this->save_to_media_library($image_url, $title, $alt_text ?: $title);
                
                AIAB_Logger::log("Featured image generated", 'info', array(
                    'attachment_id' => $attachment_id
                ));
                
                return $attachment_id;
            }
        } catch (Exception $e) {
            AIAB_Logger::log("Image generation failed: " . $e->getMessage(), 'error');
        }
        
        return null;
    }
    
    /**
     * Build an image generation prompt
     */
    private function build_prompt($title, $keyword) {
        $style = get_option('aiab_image_style', 'professional');
        
        $style_descriptions = array(
            'professional' => 'professional, clean, modern business style',
            'artistic' => 'artistic, creative, visually striking',
            'minimalist' => 'minimalist, simple, clean with lots of white space',
            'photographic' => 'photorealistic, like a professional photograph',
            'illustration' => 'digital illustration style, colorful and engaging'
        );
        
        $style_desc = isset($style_descriptions[$style]) ? $style_descriptions[$style] : $style_descriptions['professional'];
        
        $prompt = "Create a featured blog image for an article about: {$keyword}. ";
        $prompt .= "Article title: {$title}. ";
        $prompt .= "Style: {$style_desc}. ";
        $prompt .= "The image should be suitable for a blog header, visually appealing, and relevant to the topic. ";
        $prompt .= "No text in the image. High quality, 16:9 aspect ratio.";
        
        return $prompt;
    }
    
    /**
     * Generate image using DALL-E
     */
    private function generate_dalle($prompt) {
        $url = 'https://api.openai.com/v1/images/generations';
        
        $response = wp_remote_post($url, array(
            'timeout' => 60,
            'headers' => array(
                'Content-Type' => 'application/json',
                'Authorization' => 'Bearer ' . $this->api_key
            ),
            'body' => json_encode(array(
                'model' => 'dall-e-3',
                'prompt' => $prompt,
                'n' => 1,
                'size' => '1792x1024', // 16:9 ish
                'quality' => 'standard'
            ))
        ));
        
        if (is_wp_error($response)) {
            throw new Exception('DALL-E API request failed: ' . $response->get_error_message());
        }
        
        $body = json_decode(wp_remote_retrieve_body($response), true);
        
        if (isset($body['error'])) {
            throw new Exception('DALL-E error: ' . $body['error']['message']);
        }
        
        if (isset($body['data'][0]['url'])) {
            return $body['data'][0]['url'];
        }
        
        throw new Exception('Unexpected DALL-E response format');
    }
    
    /**
     * Generate image using Stability AI
     */
    private function generate_stability($prompt) {
        $url = 'https://api.stability.ai/v1/generation/stable-diffusion-xl-1024-v1-0/text-to-image';
        
        // SDXL valid dimensions (16:9 ish = 1344x768, 1:1 = 1024x1024)
        // Using 1344x768 for blog featured images (landscape)
        $response = wp_remote_post($url, array(
            'timeout' => 60,
            'headers' => array(
                'Content-Type' => 'application/json',
                'Authorization' => 'Bearer ' . $this->api_key,
                'Accept' => 'application/json'
            ),
            'body' => json_encode(array(
                'text_prompts' => array(
                    array('text' => $prompt, 'weight' => 1)
                ),
                'cfg_scale' => 7,
                'height' => 768,
                'width' => 1344,
                'samples' => 1,
                'steps' => 30
            ))
        ));
        
        if (is_wp_error($response)) {
            throw new Exception('Stability API request failed: ' . $response->get_error_message());
        }
        
        $body = json_decode(wp_remote_retrieve_body($response), true);
        
        if (isset($body['message'])) {
            throw new Exception('Stability error: ' . $body['message']);
        }
        
        if (isset($body['artifacts'][0]['base64'])) {
            // Save base64 to temp file and return path
            $temp_file = wp_tempnam('stability_');
            file_put_contents($temp_file, base64_decode($body['artifacts'][0]['base64']));
            return $temp_file;
        }
        
        throw new Exception('Unexpected Stability response format');
    }
    
    /**
     * Save image to WordPress media library
     */
    private function save_to_media_library($image_source, $title, $alt_text) {
        require_once(ABSPATH . 'wp-admin/includes/file.php');
        require_once(ABSPATH . 'wp-admin/includes/media.php');
        require_once(ABSPATH . 'wp-admin/includes/image.php');
        
        // Generate filename
        $filename = sanitize_title($title) . '-' . time() . '.png';
        
        // Handle URL vs local file
        if (filter_var($image_source, FILTER_VALIDATE_URL)) {
            // Download from URL
            $temp_file = download_url($image_source, 60);
            
            if (is_wp_error($temp_file)) {
                throw new Exception('Failed to download image: ' . $temp_file->get_error_message());
            }
        } else {
            // Local file (from Stability base64)
            $temp_file = $image_source;
        }
        
        // Prepare file array for media_handle_sideload
        $file_array = array(
            'name' => $filename,
            'tmp_name' => $temp_file
        );
        
        // Upload to media library
        $attachment_id = media_handle_sideload($file_array, 0, $title);
        
        // Clean up temp file
        @unlink($temp_file);
        
        if (is_wp_error($attachment_id)) {
            throw new Exception('Failed to add to media library: ' . $attachment_id->get_error_message());
        }
        
        // Set alt text
        update_post_meta($attachment_id, '_wp_attachment_image_alt', $alt_text);
        
        // Mark as AI generated
        update_post_meta($attachment_id, '_aiab_generated', 1);
        
        return $attachment_id;
    }
    
    /**
     * Generate image from existing stock photo (placeholder/fallback)
     */
    public function get_placeholder_image($keyword) {
        // Use Unsplash Source API as fallback
        $url = 'https://source.unsplash.com/1600x900/?' . urlencode($keyword);
        
        try {
            return $this->save_to_media_library($url, $keyword . ' featured image', $keyword);
        } catch (Exception $e) {
            AIAB_Logger::log("Placeholder image failed: " . $e->getMessage(), 'warning');
            return null;
        }
    }
}
