<?php
/**
 * Link Manager
 * Handles internal and external link injection
 */

if (!defined('ABSPATH')) {
    exit;
}

class AIAB_Link_Manager {
    
    /**
     * Inject internal links into article content
     */
    public function inject_internal_links($content, $links, $anchor_strategy = 'natural') {
        if (empty($links)) {
            return $content;
        }
        
        AIAB_Logger::log("Injecting " . count($links) . " internal links");
        
        foreach ($links as $link) {
            $url = $this->get_proper_permalink($link['wp_post_id']);
            $title = $link['title'];
            
            // Try to find natural anchor text in content
            $anchor = $this->find_anchor_text($content, $title, $link['type']);
            
            if ($anchor && strpos($content, '<a href') === false || strpos($content, $anchor) !== false) {
                // Create the link HTML
                $link_html = '<a href="' . esc_url($url) . '" title="' . esc_attr($title) . '">' . $anchor . '</a>';
                
                // Replace first occurrence that's not already linked
                $content = $this->replace_unlinked_text($content, $anchor, $link_html);
            } else {
                // Fallback: Add contextual link at end of a relevant paragraph
                $content = $this->append_contextual_link($content, $url, $title);
            }
        }
        
        return $content;
    }
    
    /**
     * Get proper permalink (ensures pretty URL, not ?p=ID format)
     */
    private function get_proper_permalink($post_id) {
        // Get the post object to ensure we have all data
        $post = get_post($post_id);
        
        if (!$post) {
            return home_url('/?p=' . $post_id);
        }
        
        // Try to get the slug directly
        $slug = $post->post_name;
        
        // If post has a proper slug, build the URL using it
        if (!empty($slug) && $slug !== $post_id && !is_numeric($slug)) {
            // Get the permalink structure
            $permalink = get_permalink($post);
            
            // Double-check it's not the ugly format
            if (strpos($permalink, '?p=') === false) {
                return $permalink;
            }
            
            // Fallback: construct URL from slug
            return trailingslashit(home_url()) . trailingslashit($slug);
        }
        
        // Last resort: use standard get_permalink
        return get_permalink($post);
    }
    
    /**
     * Find natural anchor text in content
     */
    private function find_anchor_text($content, $title, $type) {
        // Extract key phrases from title
        $title_words = explode(' ', strtolower($title));
        $content_lower = strtolower($content);
        
        // Try exact title match first
        if (stripos($content, $title) !== false) {
            return $title;
        }
        
        // Try partial matches (3+ word phrases from title)
        for ($length = min(5, count($title_words)); $length >= 3; $length--) {
            for ($start = 0; $start <= count($title_words) - $length; $start++) {
                $phrase = implode(' ', array_slice($title_words, $start, $length));
                if (stripos($content, $phrase) !== false) {
                    // Find the actual case version in content
                    preg_match('/' . preg_quote($phrase, '/') . '/i', $content, $matches);
                    if (!empty($matches[0])) {
                        return $matches[0];
                    }
                }
            }
        }
        
        // Try key individual words (nouns typically)
        $stop_words = array('the', 'a', 'an', 'in', 'on', 'at', 'to', 'for', 'of', 'and', 'or', 'is', 'are', 'was', 'were', 'be', 'been', 'how', 'what', 'why', 'when', 'where', 'which', 'who', 'your', 'my', 'our', 'their');
        
        $key_words = array_filter($title_words, function($word) use ($stop_words) {
            return strlen($word) > 3 && !in_array($word, $stop_words);
        });
        
        foreach ($key_words as $word) {
            // Find word with context (2-3 words around it)
            $pattern = '/(\w+\s+)?' . preg_quote($word, '/') . '(\s+\w+)?/i';
            if (preg_match($pattern, $content, $matches)) {
                return trim($matches[0]);
            }
        }
        
        return null;
    }
    
    /**
     * Replace text only if it's not already inside a link
     */
    private function replace_unlinked_text($content, $search, $replace) {
        // Split content by existing links
        $parts = preg_split('/(<a[^>]*>.*?<\/a>)/is', $content, -1, PREG_SPLIT_DELIM_CAPTURE);
        
        $replaced = false;
        foreach ($parts as &$part) {
            // Skip if this part is a link
            if (preg_match('/^<a[^>]*>/i', $part)) {
                continue;
            }
            
            // Replace first occurrence
            if (!$replaced && stripos($part, $search) !== false) {
                $part = preg_replace('/' . preg_quote($search, '/') . '/i', $replace, $part, 1);
                $replaced = true;
            }
        }
        
        return implode('', $parts);
    }
    
    /**
     * Append a contextual link to content
     */
    private function append_contextual_link($content, $url, $title) {
        $link_text = "Read more: <a href=\"" . esc_url($url) . "\">" . esc_html($title) . "</a>";
        
        // Find a good place to insert (after a paragraph in the middle-ish)
        $paragraphs = preg_split('/(<\/p>)/i', $content, -1, PREG_SPLIT_DELIM_CAPTURE);
        $total = count($paragraphs);
        
        if ($total > 4) {
            // Insert after ~60% of content
            $insert_at = (int) ($total * 0.6);
            $insert_at = $insert_at - ($insert_at % 2); // Ensure we're after a </p>
            
            $before = array_slice($paragraphs, 0, $insert_at + 1);
            $after = array_slice($paragraphs, $insert_at + 1);
            
            return implode('', $before) . "\n<p><em>" . $link_text . "</em></p>\n" . implode('', $after);
        }
        
        // Fallback: add before last paragraph
        return preg_replace('/<\/p>\s*$/', "</p>\n<p><em>" . $link_text . "</em></p>", $content);
    }
    
    /**
     * Inject external link from configured sources
     */
    public function inject_external_link($content, $keyword) {
        $external_sources = get_option('aiab_external_link_sources', '');
        
        if (empty($external_sources)) {
            return $content;
        }
        
        $sources = array_filter(array_map('trim', explode("\n", $external_sources)));
        
        if (empty($sources)) {
            return $content;
        }
        
        // Pick a random source
        $source_url = $sources[array_rand($sources)];
        
        // Find relevant anchor or use generic
        $anchor = $this->find_external_anchor($content, $keyword);
        
        if ($anchor) {
            $link_html = '<a href="' . esc_url($source_url) . '" target="_blank" rel="noopener noreferrer">' . $anchor . '</a>';
            $content = $this->replace_unlinked_text($content, $anchor, $link_html);
        } else {
            // Add as "Learn more" link
            $link_html = '<p><em>For more information, visit <a href="' . esc_url($source_url) . '" target="_blank" rel="noopener noreferrer">this resource</a>.</em></p>';
            
            // Insert before conclusion or at end
            if (preg_match('/<h[23][^>]*>.*?(conclusion|summary|final|wrap)/i', $content)) {
                $content = preg_replace('/(<h[23][^>]*>.*?(conclusion|summary|final|wrap))/i', $link_html . "\n$1", $content, 1);
            } else {
                $content .= "\n" . $link_html;
            }
        }
        
        AIAB_Logger::log("External link injected", 'info', array(
            'source' => $source_url
        ));
        
        return $content;
    }
    
    /**
     * Find anchor text for external link
     */
    private function find_external_anchor($content, $keyword) {
        // Look for phrases like "according to", "research shows", "studies indicate"
        $patterns = array(
            '/according to ([^,.<]+)/i',
            '/research (shows|indicates|suggests)/i',
            '/studies (show|indicate|suggest)/i',
            '/experts (say|recommend|suggest)/i',
        );
        
        foreach ($patterns as $pattern) {
            if (preg_match($pattern, $content, $matches)) {
                return $matches[0];
            }
        }
        
        // Look for keyword phrase
        $keyword_words = explode(' ', $keyword);
        if (count($keyword_words) >= 2) {
            $phrase = implode(' ', array_slice($keyword_words, 0, 3));
            if (stripos($content, $phrase) !== false) {
                return $phrase;
            }
        }
        
        return null;
    }
    
    /**
     * Process all links for a sphere
     */
    public function process_sphere_links(AIAB_Thought_Sphere $sphere) {
        AIAB_Logger::log("Processing links for sphere: " . $sphere->get_pillar_keyword());
        
        // Generate link map
        $link_map = $sphere->generate_link_map();
        
        $publisher = new AIAB_Article_Publisher();
        
        // Process pillar article links
        $pillar = $sphere->get_pillar_article();
        if ($pillar && $pillar->wp_post_id) {
            $content = get_post_field('post_content', $pillar->wp_post_id);
            
            // Add links to all cluster articles
            if (!empty($link_map['pillar_links'])) {
                $content = $this->inject_internal_links($content, $link_map['pillar_links']);
            }
            
            // Add external link
            $content = $this->inject_external_link($content, $pillar->keyword);
            
            // Update post
            $publisher->update_article_content($pillar->wp_post_id, $content);
            
            // Update article record
            $this->update_article_links($pillar->id, $link_map['pillar_links'], 'internal');
        }
        
        // Process cluster articles
        $clusters = $sphere->get_cluster_articles();
        foreach ($clusters as $cluster) {
            if (!$cluster->wp_post_id) continue;
            
            $content = get_post_field('post_content', $cluster->wp_post_id);
            
            // Add links (to pillar and adjacent clusters)
            if (!empty($link_map['cluster_links'][$cluster->id])) {
                $content = $this->inject_internal_links($content, $link_map['cluster_links'][$cluster->id]);
            }
            
            // Add external link
            $content = $this->inject_external_link($content, $cluster->keyword);
            
            // Update post
            $publisher->update_article_content($cluster->wp_post_id, $content);
            
            // Update article record
            $this->update_article_links($cluster->id, $link_map['cluster_links'][$cluster->id] ?? array(), 'internal');
        }
        
        AIAB_Logger::log("Sphere links processed successfully");
    }
    
    /**
     * Update article links in database
     */
    private function update_article_links($article_id, $links, $type = 'internal') {
        global $wpdb;
        $table = AIAB_Database::get_table('articles');
        
        $field = $type === 'internal' ? 'internal_links' : 'external_links';
        
        $wpdb->update(
            $table,
            array($field => json_encode($links)),
            array('id' => $article_id)
        );
    }
    
    /**
     * Get link statistics for a sphere
     */
    public function get_sphere_link_stats($sphere_id) {
        global $wpdb;
        $table = AIAB_Database::get_table('articles');
        
        $articles = $wpdb->get_results($wpdb->prepare(
            "SELECT id, title, internal_links, external_links FROM $table WHERE sphere_id = %d",
            $sphere_id
        ));
        
        $stats = array(
            'total_articles' => count($articles),
            'internal_links' => 0,
            'external_links' => 0,
            'articles_without_links' => 0
        );
        
        foreach ($articles as $article) {
            $internal = json_decode($article->internal_links, true) ?: array();
            $external = json_decode($article->external_links, true) ?: array();
            
            $stats['internal_links'] += count($internal);
            $stats['external_links'] += count($external);
            
            if (empty($internal) && empty($external)) {
                $stats['articles_without_links']++;
            }
        }
        
        return $stats;
    }
}
