<?php
/**
 * Logger & Debug System
 * Comprehensive logging for debugging and monitoring
 */

if (!defined('ABSPATH')) {
    exit;
}

class AIAB_Logger {
    
    private static $instance = null;
    private static $start_time = null;
    private static $memory_start = null;
    private static $current_context = array();
    
    // Log levels
    const LEVEL_DEBUG = 'debug';
    const LEVEL_INFO = 'info';
    const LEVEL_WARNING = 'warning';
    const LEVEL_ERROR = 'error';
    const LEVEL_CRITICAL = 'critical';
    
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
            self::$start_time = microtime(true);
            self::$memory_start = memory_get_usage();
        }
        return self::$instance;
    }
    
    /**
     * Main logging method
     */
    public static function log($message, $level = 'info', $context = array()) {
        global $wpdb;
        
        // Check if table exists first to avoid errors
        $table = AIAB_Database::get_table('log');
        $table_exists = $wpdb->get_var("SHOW TABLES LIKE '$table'") === $table;
        
        if (!$table_exists) {
            // Table doesn't exist - try to create it, or just use error_log
            error_log("AIAB: Log table missing, attempting to create...");
            AIAB_Database::create_tables();
            
            // Check again
            $table_exists = $wpdb->get_var("SHOW TABLES LIKE '$table'") === $table;
            if (!$table_exists) {
                // Still doesn't exist, just use error_log
                error_log("AIAB [$level]: $message");
                return;
            }
        }
        
        // Add performance metrics to context
        $context['_performance'] = array(
            'memory_usage' => self::format_bytes(memory_get_usage()),
            'memory_peak' => self::format_bytes(memory_get_peak_usage()),
            'elapsed_time' => self::$start_time ? round(microtime(true) - self::$start_time, 4) . 's' : 'N/A',
        );
        
        // Add request info for debugging
        if (empty($context['_request'])) {
            $context['_request'] = array(
                'url' => isset($_SERVER['REQUEST_URI']) ? $_SERVER['REQUEST_URI'] : 'CLI',
                'method' => isset($_SERVER['REQUEST_METHOD']) ? $_SERVER['REQUEST_METHOD'] : 'CLI',
                'is_cron' => defined('DOING_CRON') && DOING_CRON,
                'is_ajax' => defined('DOING_AJAX') && DOING_AJAX,
            );
        }
        
        // Merge with current context
        $context = array_merge(self::$current_context, $context);
        
        // Write to database (suppress errors)
        $wpdb->suppress_errors(true);
        $result = $wpdb->insert($table, array(
            'level' => $level,
            'message' => $message,
            'context' => !empty($context) ? json_encode($context, JSON_PRETTY_PRINT) : null,
            'persona_id' => isset($context['persona_id']) ? $context['persona_id'] : null,
            'sphere_id' => isset($context['sphere_id']) ? $context['sphere_id'] : null,
        ));
        $wpdb->suppress_errors(false);
        
        // If insert failed, fallback to error_log
        if ($result === false) {
            error_log("AIAB [$level]: $message");
        }
        
        // Also write to error_log and file for debugging
        $debug_enabled = get_option('aiab_debug_mode', false);
        if ($debug_enabled || $level === 'error' || $level === 'critical') {
            self::write_to_file($message, $level, $context);
        }
        
        // Send email notification for critical errors
        if ($level === 'critical') {
            self::send_error_notification($message, $context);
        }
        
        // PHP error_log for WP_DEBUG
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log("[Eternal Auto Blogger] [{$level}] {$message}");
        }
    }
    
    /**
     * Shorthand methods for different log levels
     */
    public static function debug($message, $context = array()) {
        if (get_option('aiab_debug_mode', false)) {
            self::log($message, self::LEVEL_DEBUG, $context);
        }
    }
    
    public static function info($message, $context = array()) {
        self::log($message, self::LEVEL_INFO, $context);
    }
    
    public static function warning($message, $context = array()) {
        self::log($message, self::LEVEL_WARNING, $context);
    }
    
    public static function error($message, $context = array()) {
        self::log($message, self::LEVEL_ERROR, $context);
    }
    
    public static function critical($message, $context = array()) {
        self::log($message, self::LEVEL_CRITICAL, $context);
    }
    
    /**
     * Log an exception with full details
     */
    public static function exception(Exception $e, $context = array()) {
        $context['exception'] = array(
            'class' => get_class($e),
            'message' => $e->getMessage(),
            'code' => $e->getCode(),
            'file' => $e->getFile(),
            'line' => $e->getLine(),
            'trace' => self::format_trace($e->getTrace()),
        );
        
        self::log('Exception: ' . $e->getMessage(), self::LEVEL_ERROR, $context);
    }
    
    /**
     * Log API call with request/response details
     */
    public static function api_call($service, $endpoint, $request, $response, $duration, $success = true) {
        $level = $success ? self::LEVEL_DEBUG : self::LEVEL_ERROR;
        
        $context = array(
            'api_call' => array(
                'service' => $service,
                'endpoint' => $endpoint,
                'duration' => round($duration, 4) . 's',
                'success' => $success,
                'request' => self::sanitize_api_data($request),
                'response' => self::sanitize_api_data($response),
            )
        );
        
        $message = $success 
            ? "API Call: {$service} - {$endpoint} ({$context['api_call']['duration']})"
            : "API Call FAILED: {$service} - {$endpoint}";
        
        self::log($message, $level, $context);
    }
    
    /**
     * Log database query for debugging
     */
    public static function query($sql, $duration = null, $result_count = null) {
        if (!get_option('aiab_debug_mode', false)) {
            return;
        }
        
        $context = array(
            'query' => array(
                'sql' => $sql,
                'duration' => $duration ? round($duration, 4) . 's' : 'N/A',
                'result_count' => $result_count,
            )
        );
        
        self::log('DB Query', self::LEVEL_DEBUG, $context);
    }
    
    /**
     * Set context that will be included in all subsequent logs
     */
    public static function set_context($key, $value) {
        self::$current_context[$key] = $value;
    }
    
    /**
     * Clear context
     */
    public static function clear_context() {
        self::$current_context = array();
    }
    
    /**
     * Start a timed operation
     */
    public static function start_timer($operation) {
        self::$current_context['_timers'][$operation] = microtime(true);
        self::debug("Started: {$operation}");
    }
    
    /**
     * End a timed operation and log duration
     */
    public static function end_timer($operation, $context = array()) {
        if (isset(self::$current_context['_timers'][$operation])) {
            $duration = microtime(true) - self::$current_context['_timers'][$operation];
            $context['duration'] = round($duration, 4) . 's';
            unset(self::$current_context['_timers'][$operation]);
            self::debug("Completed: {$operation}", $context);
            return $duration;
        }
        return 0;
    }
    
    /**
     * Write to log file
     */
    private static function write_to_file($message, $level, $context) {
        $log_dir = WP_CONTENT_DIR . '/aiab-logs';
        
        // Create directory if needed
        if (!file_exists($log_dir)) {
            wp_mkdir_p($log_dir);
            // Add .htaccess to protect logs
            file_put_contents($log_dir . '/.htaccess', 'Deny from all');
        }
        
        $log_file = $log_dir . '/debug-' . date('Y-m-d') . '.log';
        
        $timestamp = date('Y-m-d H:i:s');
        $level_upper = strtoupper($level);
        
        $log_entry = "[{$timestamp}] [{$level_upper}] {$message}\n";
        
        if (!empty($context)) {
            $log_entry .= "Context: " . json_encode($context, JSON_PRETTY_PRINT) . "\n";
        }
        
        $log_entry .= str_repeat('-', 80) . "\n";
        
        file_put_contents($log_file, $log_entry, FILE_APPEND | LOCK_EX);
    }
    
    /**
     * Send email notification for critical errors
     */
    private static function send_error_notification($message, $context) {
        $notify_email = get_option('aiab_error_notification_email', '');
        
        if (empty($notify_email)) {
            $notify_email = get_option('admin_email');
        }
        
        if (!get_option('aiab_error_notifications', false)) {
            return;
        }
        
        $subject = '[Eternal Auto Blogger] Critical Error on ' . get_bloginfo('name');
        
        $body = "A critical error occurred in Eternal Auto Blogger:\n\n";
        $body .= "Message: {$message}\n\n";
        $body .= "Time: " . date('Y-m-d H:i:s') . "\n";
        $body .= "Site: " . get_bloginfo('url') . "\n\n";
        $body .= "Context:\n" . print_r($context, true);
        
        wp_mail($notify_email, $subject, $body);
    }
    
    /**
     * Format stack trace for readability
     */
    private static function format_trace($trace) {
        $formatted = array();
        
        foreach (array_slice($trace, 0, 10) as $i => $frame) {
            $formatted[] = sprintf(
                '#%d %s%s%s() at %s:%d',
                $i,
                isset($frame['class']) ? $frame['class'] : '',
                isset($frame['type']) ? $frame['type'] : '',
                isset($frame['function']) ? $frame['function'] : 'unknown',
                isset($frame['file']) ? basename($frame['file']) : 'unknown',
                isset($frame['line']) ? $frame['line'] : 0
            );
        }
        
        return $formatted;
    }
    
    /**
     * Sanitize API data (remove sensitive info)
     */
    private static function sanitize_api_data($data) {
        if (is_string($data)) {
            // Truncate long strings
            if (strlen($data) > 2000) {
                return substr($data, 0, 2000) . '... [TRUNCATED]';
            }
            return $data;
        }
        
        if (is_array($data)) {
            // Remove sensitive keys
            $sensitive_keys = array('api_key', 'x-api-key', 'authorization', 'password', 'secret');
            
            foreach ($data as $key => $value) {
                if (in_array(strtolower($key), $sensitive_keys)) {
                    $data[$key] = '[REDACTED]';
                } elseif (is_array($value)) {
                    $data[$key] = self::sanitize_api_data($value);
                } elseif (is_string($value) && strlen($value) > 1000) {
                    $data[$key] = substr($value, 0, 1000) . '... [TRUNCATED]';
                }
            }
        }
        
        return $data;
    }
    
    /**
     * Format bytes to human readable
     */
    private static function format_bytes($bytes) {
        $units = array('B', 'KB', 'MB', 'GB');
        $i = 0;
        while ($bytes >= 1024 && $i < count($units) - 1) {
            $bytes /= 1024;
            $i++;
        }
        return round($bytes, 2) . ' ' . $units[$i];
    }
    
    /**
     * Get recent logs
     */
    public static function get_logs($limit = 100, $level = null, $persona_id = null, $sphere_id = null, $search = null) {
        global $wpdb;
        $table = AIAB_Database::get_table('log');
        
        $sql = "SELECT * FROM $table WHERE 1=1";
        $params = array();
        
        if ($level && $level !== 'all') {
            $sql .= " AND level = %s";
            $params[] = $level;
        }
        
        if ($persona_id) {
            $sql .= " AND persona_id = %d";
            $params[] = $persona_id;
        }
        
        if ($sphere_id) {
            $sql .= " AND sphere_id = %d";
            $params[] = $sphere_id;
        }
        
        if ($search) {
            $sql .= " AND (message LIKE %s OR context LIKE %s)";
            $params[] = '%' . $wpdb->esc_like($search) . '%';
            $params[] = '%' . $wpdb->esc_like($search) . '%';
        }
        
        $sql .= " ORDER BY created_at DESC LIMIT %d";
        $params[] = $limit;
        
        if (!empty($params)) {
            $sql = $wpdb->prepare($sql, $params);
        }
        
        return $wpdb->get_results($sql);
    }
    
    /**
     * Get logs for a specific sphere
     */
    public static function get_sphere_logs($sphere_id) {
        global $wpdb;
        $table = AIAB_Database::get_table('log');
        
        return $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM $table WHERE sphere_id = %d ORDER BY created_at ASC",
            $sphere_id
        ));
    }
    
    /**
     * Get log statistics
     */
    public static function get_stats($days = 7) {
        global $wpdb;
        $table = AIAB_Database::get_table('log');
        
        $stats = array(
            'total' => $wpdb->get_var("SELECT COUNT(*) FROM $table"),
            'today' => $wpdb->get_var("SELECT COUNT(*) FROM $table WHERE DATE(created_at) = CURDATE()"),
            'debug' => $wpdb->get_var("SELECT COUNT(*) FROM $table WHERE level = 'debug' AND created_at >= DATE_SUB(NOW(), INTERVAL {$days} DAY)"),
            'info' => $wpdb->get_var("SELECT COUNT(*) FROM $table WHERE level = 'info' AND created_at >= DATE_SUB(NOW(), INTERVAL {$days} DAY)"),
            'warnings' => $wpdb->get_var("SELECT COUNT(*) FROM $table WHERE level = 'warning' AND created_at >= DATE_SUB(NOW(), INTERVAL {$days} DAY)"),
            'errors' => $wpdb->get_var("SELECT COUNT(*) FROM $table WHERE level = 'error' AND created_at >= DATE_SUB(NOW(), INTERVAL {$days} DAY)"),
            'critical' => $wpdb->get_var("SELECT COUNT(*) FROM $table WHERE level = 'critical' AND created_at >= DATE_SUB(NOW(), INTERVAL {$days} DAY)"),
            'by_day' => $wpdb->get_results(
                "SELECT DATE(created_at) as date, level, COUNT(*) as count 
                 FROM $table 
                 WHERE created_at >= DATE_SUB(NOW(), INTERVAL {$days} DAY) 
                 GROUP BY DATE(created_at), level 
                 ORDER BY date DESC"
            ),
            'top_errors' => $wpdb->get_results(
                "SELECT message, COUNT(*) as count 
                 FROM $table 
                 WHERE level IN ('error', 'critical') 
                 AND created_at >= DATE_SUB(NOW(), INTERVAL {$days} DAY) 
                 GROUP BY message 
                 ORDER BY count DESC 
                 LIMIT 10"
            ),
        );
        
        return $stats;
    }
    
    /**
     * Clear old logs
     */
    public static function clear_old_logs($days = 30) {
        global $wpdb;
        $table = AIAB_Database::get_table('log');
        
        $deleted = $wpdb->query($wpdb->prepare(
            "DELETE FROM $table WHERE created_at < DATE_SUB(NOW(), INTERVAL %d DAY)",
            $days
        ));
        
        self::info("Cleared {$deleted} logs older than {$days} days");
        
        return $deleted;
    }
    
    /**
     * Clear all logs
     */
    public static function clear_all_logs() {
        global $wpdb;
        $table = AIAB_Database::get_table('log');
        return $wpdb->query("TRUNCATE TABLE $table");
    }
    
    /**
     * Export logs
     */
    public static function export($format = 'json', $level = null, $days = 7) {
        $logs = self::get_logs(10000, $level);
        
        if ($format === 'json') {
            return json_encode($logs, JSON_PRETTY_PRINT);
        }
        
        // CSV format
        $csv = "ID,Timestamp,Level,Message,Persona ID,Sphere ID,Context\n";
        foreach ($logs as $log) {
            $csv .= sprintf(
                "%d,\"%s\",%s,\"%s\",%s,%s,\"%s\"\n",
                $log->id,
                $log->created_at,
                $log->level,
                str_replace('"', '""', $log->message),
                $log->persona_id ?: '',
                $log->sphere_id ?: '',
                str_replace('"', '""', $log->context ?: '')
            );
        }
        
        return $csv;
    }
    
    /**
     * Get log file list
     */
    public static function get_log_files() {
        $log_dir = WP_CONTENT_DIR . '/aiab-logs';
        $files = array();
        
        if (is_dir($log_dir)) {
            foreach (glob($log_dir . '/debug-*.log') as $file) {
                $files[] = array(
                    'name' => basename($file),
                    'path' => $file,
                    'size' => self::format_bytes(filesize($file)),
                    'modified' => date('Y-m-d H:i:s', filemtime($file)),
                );
            }
        }
        
        usort($files, function($a, $b) {
            return strcmp($b['name'], $a['name']);
        });
        
        return $files;
    }
    
    /**
     * Get log file contents
     */
    public static function get_log_file_contents($filename, $lines = 500) {
        $log_dir = WP_CONTENT_DIR . '/aiab-logs';
        $file = $log_dir . '/' . basename($filename);
        
        if (!file_exists($file)) {
            return '';
        }
        
        // Read last N lines efficiently
        $file_handle = fopen($file, 'r');
        $all_lines = array();
        
        while (!feof($file_handle)) {
            $line = fgets($file_handle);
            if ($line !== false) {
                $all_lines[] = $line;
                if (count($all_lines) > $lines) {
                    array_shift($all_lines);
                }
            }
        }
        
        fclose($file_handle);
        
        return implode('', $all_lines);
    }
    
    /**
     * System health check
     */
    public static function health_check() {
        $health = array(
            'php_version' => array(
                'label' => 'PHP Version',
                'value' => PHP_VERSION,
                'status' => version_compare(PHP_VERSION, '7.4', '>=') ? 'good' : 'warning',
            ),
            'memory_limit' => array(
                'label' => 'Memory Limit',
                'value' => ini_get('memory_limit'),
                'status' => intval(ini_get('memory_limit')) >= 256 ? 'good' : 'warning',
            ),
            'max_execution_time' => array(
                'label' => 'Max Execution Time',
                'value' => ini_get('max_execution_time') . 's',
                'status' => intval(ini_get('max_execution_time')) >= 120 ? 'good' : 'warning',
            ),
            'wp_cron' => array(
                'label' => 'WP-Cron',
                'value' => defined('DISABLE_WP_CRON') && DISABLE_WP_CRON ? 'Disabled' : 'Enabled',
                'status' => defined('DISABLE_WP_CRON') && DISABLE_WP_CRON ? 'warning' : 'good',
            ),
            'curl' => array(
                'label' => 'cURL',
                'value' => function_exists('curl_version') ? curl_version()['version'] : 'Not available',
                'status' => function_exists('curl_version') ? 'good' : 'error',
            ),
            'ssl' => array(
                'label' => 'SSL Support',
                'value' => extension_loaded('openssl') ? 'Enabled' : 'Disabled',
                'status' => extension_loaded('openssl') ? 'good' : 'error',
            ),
            'json' => array(
                'label' => 'JSON Support',
                'value' => function_exists('json_encode') ? 'Enabled' : 'Disabled',
                'status' => function_exists('json_encode') ? 'good' : 'error',
            ),
            'log_dir_writable' => array(
                'label' => 'Log Directory',
                'value' => is_writable(WP_CONTENT_DIR) ? 'Writable' : 'Not Writable',
                'status' => is_writable(WP_CONTENT_DIR) ? 'good' : 'warning',
            ),
        );
        
        // Check API keys
        $health['anthropic_key'] = array(
            'label' => 'Anthropic API Key',
            'value' => get_option('aiab_anthropic_api_key') ? 'Configured' : 'Not Set',
            'status' => get_option('aiab_anthropic_api_key') ? 'good' : 'warning',
        );
        
        $health['openai_key'] = array(
            'label' => 'OpenAI API Key',
            'value' => get_option('aiab_openai_api_key') ? 'Configured' : 'Not Set',
            'status' => get_option('aiab_openai_api_key') ? 'good' : 'warning',
        );
        
        $health['stability_key'] = array(
            'label' => 'Stability AI Key',
            'value' => get_option('aiab_stability_api_key') ? 'Configured' : 'Not Set',
            'status' => get_option('aiab_stability_api_key') ? 'good' : 'warning',
        );
        
        // Check image provider configuration
        $image_provider = get_option('aiab_image_provider', 'dalle');
        $image_key_ok = false;
        if ($image_provider === 'dalle' && get_option('aiab_openai_api_key')) {
            $image_key_ok = true;
        } elseif ($image_provider === 'stability' && get_option('aiab_stability_api_key')) {
            $image_key_ok = true;
        } elseif ($image_provider === 'none') {
            $image_key_ok = true; // Disabled is OK
        }
        
        $health['image_config'] = array(
            'label' => 'Image Generation',
            'value' => $image_provider === 'none' ? 'Disabled' : ($image_key_ok ? 'Ready' : 'Missing API Key'),
            'status' => $image_key_ok ? 'good' : 'warning',
        );
        
        // Check next scheduled run
        $next_run = wp_next_scheduled('aiab_run_autoblogger');
        $health['cron_scheduled'] = array(
            'label' => 'Next Scheduled Run',
            'value' => $next_run ? date('Y-m-d H:i:s', $next_run) : 'Not Scheduled',
            'status' => $next_run ? 'good' : 'warning',
        );
        
        return $health;
    }
}
