<?php
/**
 * Orchestrator
 * THE ETERNAL CYCLE - Coordinates the entire auto-blogging process
 */

if (!defined('ABSPATH')) {
    exit;
}

class AIAB_Orchestrator {
    
    private $persona;
    private $sphere;
    private $max_execution_time;
    private $start_time;
    private static $lock_key = 'aiab_orchestrator_lock';
    private $lock_id; // Unique ID for this instance
    
    // Static timeout for research engine to check
    private static $cycle_start_time = null;
    private static $cycle_max_time = 300;
    
    public function __construct() {
        $this->max_execution_time = get_option('aiab_max_execution_time', 300); // 5 minutes default
        
        // For Walter (local Ollama), use much longer timeout since CPU inference is slow
        if (get_option('aiab_ai_provider') === 'walter') {
            $this->max_execution_time = max($this->max_execution_time, 900); // At least 15 minutes for Walter
        }
        
        $this->start_time = time();
        $this->lock_id = uniqid('lock_', true) . '_' . getmypid(); // Unique per process
        
        // Set static values for other classes to check
        self::$cycle_start_time = $this->start_time;
        self::$cycle_max_time = $this->max_execution_time;
    }
    
    /**
     * Check if cycle should abort (for use by other classes like research engine)
     */
    public static function should_abort() {
        if (self::$cycle_start_time === null) {
            return false;
        }
        $elapsed = time() - self::$cycle_start_time;
        return $elapsed > (self::$cycle_max_time - 60); // 60 second buffer
    }
    
    /**
     * Acquire process lock to prevent concurrent execution
     * Uses WordPress native add_option which fails if option exists
     */
    private function acquire_lock() {
        $option_name = self::$lock_key;
        
        // Step 1: Clear all caches to get fresh data
        wp_cache_delete($option_name, 'options');
        wp_cache_delete('alloptions', 'options');
        
        // Step 2: Check for existing lock
        $existing_lock = get_option($option_name, false);
        
        if ($existing_lock !== false && !empty($existing_lock)) {
            $lock_data = json_decode($existing_lock, true);
            $lock_time = isset($lock_data['time']) ? intval($lock_data['time']) : 0;
            $lock_holder = isset($lock_data['id']) ? $lock_data['id'] : 'unknown';
            
            // Check if lock is stale (older than 30 minutes)
            if ($lock_time > 0 && (time() - $lock_time) > 1800) {
                AIAB_Logger::warning("Stale lock detected (from " . human_time_diff($lock_time) . " ago, holder: $lock_holder), clearing it");
                delete_option($option_name);
                wp_cache_delete($option_name, 'options');
                wp_cache_delete('alloptions', 'options');
            } else {
                AIAB_Logger::warning("Lock held by: $lock_holder (set " . ($lock_time > 0 ? human_time_diff($lock_time) . " ago" : "unknown time") . ")");
                return false;
            }
        }
        
        // Step 3: Try to acquire lock using add_option (fails if exists)
        $lock_value = json_encode(array(
            'id' => $this->lock_id,
            'time' => time(),
            'pid' => getmypid()
        ));
        
        // add_option returns FALSE if option already exists (atomic-ish)
        $added = add_option($option_name, $lock_value, '', 'no');
        
        if ($added) {
            AIAB_Logger::info("🔒 Lock acquired: " . $this->lock_id);
            return true;
        }
        
        // add_option failed - option might have been created by concurrent process
        // Small delay then verify
        usleep(50000); // 50ms
        
        wp_cache_delete($option_name, 'options');
        wp_cache_delete('alloptions', 'options');
        $current_lock = get_option($option_name, false);
        
        if ($current_lock === false || empty($current_lock)) {
            // Weird state - try update_option as fallback
            AIAB_Logger::warning("Lock state unclear after add_option failed, attempting update_option");
            update_option($option_name, $lock_value, 'no');
            
            usleep(50000);
            wp_cache_delete($option_name, 'options');
            $verify = get_option($option_name, false);
            $verify_data = $verify ? json_decode($verify, true) : null;
            
            if ($verify_data && isset($verify_data['id']) && $verify_data['id'] === $this->lock_id) {
                AIAB_Logger::info("🔒 Lock acquired via fallback: " . $this->lock_id);
                return true;
            }
            
            AIAB_Logger::error("Lock acquisition failed completely");
            return false;
        }
        
        $current_data = json_decode($current_lock, true);
        
        if ($current_data && isset($current_data['id']) && $current_data['id'] === $this->lock_id) {
            // We got it somehow
            AIAB_Logger::info("🔒 Lock acquired (verified): " . $this->lock_id);
            return true;
        }
        
        // Another process won
        $winner = isset($current_data['id']) ? $current_data['id'] : 'unknown';
        AIAB_Logger::warning("🔒 Lock race lost to: $winner");
        return false;
    }
    
    /**
     * Release process lock
     */
    private function release_lock() {
        // Only release if we hold the lock
        $current_lock = get_option(self::$lock_key);
        $current_data = json_decode($current_lock, true);
        
        if (isset($current_data['id']) && $current_data['id'] === $this->lock_id) {
            delete_option(self::$lock_key);
            AIAB_Logger::info("🔓 Lock released: " . $this->lock_id);
        } else {
            AIAB_Logger::warning("🔓 Not releasing lock - we don't own it");
        }
    }
    
    /**
     * Main entry point - THE ETERNAL CYCLE
     */
    public function run() {
        // Try to acquire lock
        if (!$this->acquire_lock()) {
            AIAB_Logger::warning('Another orchestrator cycle is already running. Skipping this execution.');
            return;
        }
        
        AIAB_Logger::info('=== ORCHESTRATOR STARTED ===');
        AIAB_Logger::start_timer('full_cycle');
        AIAB_Logger::set_context('cycle_id', uniqid('cycle_'));
        
        try {
            // Step 0: AUTO-HEAL - Fix any stuck articles before starting
            $this->auto_heal_stuck_articles();
            
            // Step 1: Select or continue with a persona
            AIAB_Logger::start_timer('persona_selection');
            $this->select_persona();
            AIAB_Logger::end_timer('persona_selection');
            
            if (!$this->persona) {
                AIAB_Logger::warning('No active personas found. Cycle skipped.');
                return;
            }
            
            AIAB_Logger::set_context('persona_id', $this->persona->get_id());
            AIAB_Logger::info("Selected persona: " . $this->persona->get_name(), array(
                'persona_id' => $this->persona->get_id(),
                'persona_name' => $this->persona->get_name()
            ));
            
            // Step 2: Check for incomplete sphere or create new one
            AIAB_Logger::start_timer('sphere_setup');
            $this->get_or_create_sphere();
            AIAB_Logger::end_timer('sphere_setup');
            
            if (!$this->sphere) {
                AIAB_Logger::error('Failed to create thought sphere');
                return;
            }
            
            AIAB_Logger::set_context('sphere_id', $this->sphere->get_id());
            
            // Step 3: Execute the current phase
            AIAB_Logger::start_timer('phase_execution');
            $this->execute_phase();
            AIAB_Logger::end_timer('phase_execution');
            
            // Step 4: Check if sphere is complete
            if ($this->sphere->get_status() === AIAB_Thought_Sphere::STATUS_COMPLETED) {
                AIAB_Logger::info('Thought sphere completed successfully!', array(
                    'sphere_id' => $this->sphere->get_id(),
                    'pillar' => $this->sphere->get_pillar_keyword(),
                    'total_articles' => $this->sphere->get_total_articles()
                ));
                
                // Update persona stats
                $this->persona->increment_spheres();
                $this->persona->increment_articles($this->sphere->get_total_articles());
            }
            
        } catch (Exception $e) {
            AIAB_Logger::exception($e, array(
                'phase' => $this->sphere ? $this->sphere->get_phase() : 'unknown',
                'persona_id' => $this->persona ? $this->persona->get_id() : null,
                'sphere_id' => $this->sphere ? $this->sphere->get_id() : null
            ));
            
            if ($this->sphere) {
                $this->sphere->log_error($e->getMessage());
                $this->sphere->update_status(AIAB_Thought_Sphere::STATUS_FAILED);
            }
            
            // Release lock on error
            $this->release_lock();
        }
        
        $total_time = AIAB_Logger::end_timer('full_cycle');
        AIAB_Logger::info('=== ORCHESTRATOR FINISHED ===', array(
            'total_duration' => $total_time . 's',
            'final_status' => $this->sphere ? $this->sphere->get_status() : 'no_sphere'
        ));
        AIAB_Logger::clear_context();
        
        // Release lock on normal completion
        $this->release_lock();
    }
    
    /**
     * Auto-heal stuck articles and spheres
     * This runs at the START of every cycle to ensure autonomous operation
     */
    private function auto_heal_stuck_articles() {
        global $wpdb;
        $articles_table = AIAB_Database::get_table('articles');
        $spheres_table = AIAB_Database::get_table('thought_spheres');
        
        // Check if updated_at column exists (for backwards compatibility)
        $has_updated_at_articles = $wpdb->get_var(
            "SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS 
             WHERE TABLE_SCHEMA = DATABASE() 
             AND TABLE_NAME = '$articles_table' 
             AND COLUMN_NAME = 'updated_at'"
        );
        
        $has_updated_at_spheres = $wpdb->get_var(
            "SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS 
             WHERE TABLE_SCHEMA = DATABASE() 
             AND TABLE_NAME = '$spheres_table' 
             AND COLUMN_NAME = 'updated_at'"
        );
        
        // 1. Fix articles stuck in 'writing' status with no content (10+ minutes old)
        if ($has_updated_at_articles) {
            $stuck_articles = $wpdb->get_results(
                "SELECT id, keyword FROM $articles_table 
                 WHERE status = 'writing' 
                 AND word_count < 100 
                 AND updated_at < DATE_SUB(NOW(), INTERVAL 10 MINUTE)"
            );
        } else {
            // Fallback: use created_at if updated_at doesn't exist
            $stuck_articles = $wpdb->get_results(
                "SELECT id, keyword FROM $articles_table 
                 WHERE status = 'writing' 
                 AND word_count < 100 
                 AND created_at < DATE_SUB(NOW(), INTERVAL 10 MINUTE)"
            );
        }
        
        if (!empty($stuck_articles)) {
            foreach ($stuck_articles as $article) {
                $wpdb->update(
                    $articles_table,
                    array('status' => 'planned', 'content' => '', 'word_count' => 0),
                    array('id' => $article->id)
                );
                AIAB_Logger::warning("🔧 Auto-healed stuck article: " . $article->keyword, array(
                    'article_id' => $article->id
                ));
            }
            AIAB_Logger::info("🔧 Auto-healed " . count($stuck_articles) . " stuck article(s)");
        }
        
        // 2. Fix spheres stuck in non-final status with no activity (30+ minutes)
        // These should be touched/updated every cycle, so 30 min inactive is a problem
        if ($has_updated_at_spheres) {
            $stuck_spheres = $wpdb->get_results(
                "SELECT id, pillar_keyword, phase, status FROM $spheres_table 
                 WHERE status NOT IN ('completed', 'failed', 'paused') 
                 AND updated_at < DATE_SUB(NOW(), INTERVAL 30 MINUTE)"
            );
            
            if (!empty($stuck_spheres)) {
                foreach ($stuck_spheres as $sphere) {
                    // Touch the sphere to update its timestamp
                    $wpdb->query($wpdb->prepare(
                        "UPDATE $spheres_table SET updated_at = NOW() WHERE id = %d",
                        $sphere->id
                    ));
                    AIAB_Logger::warning("🔧 Touched stuck sphere for continuation: " . $sphere->pillar_keyword, array(
                        'sphere_id' => $sphere->id,
                        'phase' => $sphere->phase,
                        'status' => $sphere->status
                    ));
                }
            }
        }
        
        // 3. Unpause any paused spheres (they should continue)
        $paused_spheres = $wpdb->get_var(
            "SELECT COUNT(*) FROM $spheres_table WHERE status = 'paused'"
        );
        
        if ($paused_spheres > 0) {
            // Set oldest paused sphere to 'writing' so it continues (not 'planned' which might restart research)
            $wpdb->query(
                "UPDATE $spheres_table 
                 SET status = 'writing' 
                 WHERE status = 'paused' 
                 ORDER BY created_at ASC 
                 LIMIT 1"
            );
            AIAB_Logger::info("🔧 Resumed 1 paused sphere for continuation");
        }
        
        // 4. Clear any stale locks that might have been orphaned
        $lock_option = get_option(self::$lock_key, false);
        if ($lock_option) {
            $lock_data = json_decode($lock_option, true);
            $lock_time = isset($lock_data['time']) ? intval($lock_data['time']) : 0;
            
            // If lock is older than 15 minutes (half of stale threshold), log it
            if ($lock_time > 0 && (time() - $lock_time) > 900 && (time() - $lock_time) < 1800) {
                AIAB_Logger::warning("⚠️ Lock is getting old (" . round((time() - $lock_time) / 60) . " minutes), will auto-clear in " . round((1800 - (time() - $lock_time)) / 60) . " minutes");
            }
        }
        
        // 5. AUTO-ADVANCE: Spheres stuck in writing phase that should move to linking
        // This catches cases where all articles are "written" or permanently failed
        $this->auto_advance_stuck_spheres();
    }
    
    /**
     * Auto-advance spheres stuck in writing phase to linking
     * This ensures autonomous operation when:
     * - All articles are written (even if some have low word counts)
     * - Or all remaining articles are permanently failed (auth_failed, budget_failed, max_retries)
     */
    private function auto_advance_stuck_spheres() {
        global $wpdb;
        $spheres_table = AIAB_Database::get_table('thought_spheres');
        $articles_table = AIAB_Database::get_table('articles');
        
        // Find spheres in writing phases
        $writing_spheres = $wpdb->get_results(
            "SELECT id, pillar_keyword, phase, status FROM $spheres_table 
             WHERE phase IN ('writing_pillar', 'writing_clusters') 
             AND status NOT IN ('completed', 'failed', 'paused')"
        );
        
        foreach ($writing_spheres as $sphere) {
            // Count articles that CAN still be worked on
            $workable = $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM $articles_table 
                 WHERE sphere_id = %d 
                 AND status IN ('planned', 'writing', 'failed')
                 AND (retry_count IS NULL OR retry_count < 3)",
                $sphere->id
            ));
            
            // Count written articles (regardless of word count)
            $written = $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM $articles_table 
                 WHERE sphere_id = %d 
                 AND status = 'written'",
                $sphere->id
            ));
            
            // Count permanently failed articles
            $perm_failed = $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM $articles_table 
                 WHERE sphere_id = %d 
                 AND status IN ('auth_failed', 'budget_failed', 'max_retries')",
                $sphere->id
            ));
            
            // If no workable articles AND we have at least some written ones, advance to linking
            if ($workable == 0 && $written > 0) {
                AIAB_Logger::warning("🔧 AUTO-ADVANCING sphere to linking phase", array(
                    'sphere_id' => $sphere->id,
                    'pillar' => $sphere->pillar_keyword,
                    'current_phase' => $sphere->phase,
                    'written_articles' => $written,
                    'permanently_failed' => $perm_failed,
                    'reason' => 'No more articles can be written - advancing to publish what we have'
                ));
                
                $wpdb->update(
                    $spheres_table,
                    array(
                        'phase' => 'linking',
                        'updated_at' => current_time('mysql')
                    ),
                    array('id' => $sphere->id)
                );
            }
        }
    }
    
    /**
     * Select a persona for this cycle
     */
    private function select_persona() {
        // First, check if there's an incomplete sphere - continue with that persona
        global $wpdb;
        $spheres_table = AIAB_Database::get_table('thought_spheres');
        
        $incomplete = $wpdb->get_row(
            "SELECT persona_id FROM $spheres_table 
             WHERE status NOT IN ('completed', 'failed') 
             ORDER BY created_at ASC LIMIT 1"
        );
        
        if ($incomplete) {
            $this->persona = AIAB_Persona::get($incomplete->persona_id);
            return;
        }
        
        // Otherwise, select the persona that wrote least recently (round-robin style)
        $this->persona = AIAB_Persona::get_least_recent();
        
        // Fallback: just get any active persona
        if (!$this->persona) {
            $this->persona = AIAB_Persona::get_random_active();
        }
    }
    
    /**
     * Get existing incomplete sphere or create new one
     */
    private function get_or_create_sphere() {
        // Check for incomplete sphere
        $this->sphere = AIAB_Thought_Sphere::get_incomplete_for_persona($this->persona->get_id());
        
        if ($this->sphere) {
            AIAB_Logger::log("Resuming incomplete sphere: " . $this->sphere->get_pillar_keyword(), 'info', array(
                'sphere_id' => $this->sphere->get_id(),
                'phase' => $this->sphere->get_phase()
            ));
            return;
        }
        
        // FREE VERSION LIMIT: Check daily sphere limit (1 per day)
        $daily_limit = 1; // Free version limit
        $spheres_today = $this->count_spheres_created_today();
        
        if ($spheres_today >= $daily_limit) {
            AIAB_Logger::warning("📊 FREE VERSION LIMIT: Daily sphere limit reached ({$spheres_today}/{$daily_limit})", array(
                'spheres_today' => $spheres_today,
                'daily_limit' => $daily_limit,
                'message' => 'Upgrade to Pro for unlimited spheres: https://eternalautoblogger.com'
            ));
            
            // Store notification for admin
            set_transient('aiab_daily_limit_reached', array(
                'date' => date('Y-m-d'),
                'spheres' => $spheres_today,
                'limit' => $daily_limit
            ), DAY_IN_SECONDS);
            
            return; // Stop - don't create new sphere
        }
        
        // Create new sphere - RESEARCH PHASE
        AIAB_Logger::log("Starting new thought sphere research");
        
        $research = new AIAB_Research_Engine($this->persona);
        $pillar_topic = $research->discover_pillar_topic();
        
        if (!$pillar_topic) {
            throw new Exception('Failed to discover pillar topic');
        }
        
        // Create sphere - FREE VERSION: Fixed at 4 articles
        $total_articles = 4; // Free version limit: 1 pillar + 3 clusters
        $this->sphere = AIAB_Thought_Sphere::create(
            $this->persona->get_id(),
            $pillar_topic['keyword'],
            $total_articles
        );
        
        // Check if sphere was created successfully
        if (!$this->sphere) {
            AIAB_Logger::log("Failed to create thought sphere", 'error', array(
                'pillar_keyword' => $pillar_topic['keyword'],
                'persona_id' => $this->persona->get_id()
            ));
            throw new Exception("Failed to create thought sphere for: " . $pillar_topic['keyword']);
        }
        
        // Store research data
        $this->sphere->set_research_data($pillar_topic);
        $this->sphere->save();
        
        AIAB_Logger::log("Created new sphere with pillar: " . $pillar_topic['keyword'], 'info', array(
            'sphere_id' => $this->sphere->get_id()
        ));
    }
    
    /**
     * Count spheres created today (for daily limit)
     * FREE VERSION LIMIT
     */
    private function count_spheres_created_today() {
        global $wpdb;
        $table = AIAB_Database::get_table('thought_spheres');
        
        $today_start = date('Y-m-d 00:00:00');
        
        $count = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM $table WHERE created_at >= %s",
            $today_start
        ));
        
        return (int) $count;
    }
    
    /**
     * Execute the current phase
     */
    private function execute_phase() {
        $phase = $this->sphere->get_phase();
        
        AIAB_Logger::log("Executing phase: $phase", 'info', array(
            'sphere_id' => $this->sphere->get_id()
        ));
        
        switch ($phase) {
            case AIAB_Thought_Sphere::PHASE_RESEARCH:
                $this->phase_research();
                break;
                
            case AIAB_Thought_Sphere::PHASE_PLANNING:
                $this->phase_planning();
                break;
                
            case AIAB_Thought_Sphere::PHASE_WRITING_PILLAR:
                $this->phase_write_pillar();
                break;
                
            case AIAB_Thought_Sphere::PHASE_WRITING_CLUSTERS:
                $this->phase_write_clusters();
                break;
                
            case AIAB_Thought_Sphere::PHASE_LINKING:
                $this->phase_linking();
                break;
                
            case AIAB_Thought_Sphere::PHASE_PUBLISHING:
                $this->phase_publishing();
                break;
                
            case AIAB_Thought_Sphere::PHASE_COMPLETE:
                $this->sphere->update_status(AIAB_Thought_Sphere::STATUS_COMPLETED);
                break;
        }
    }
    
    /**
     * Phase 1: Research supporting topics
     */
    private function phase_research() {
        $this->sphere->update_status(AIAB_Thought_Sphere::STATUS_RESEARCHING);
        
        $research = new AIAB_Research_Engine($this->persona);
        $cluster_count = $this->sphere->get_total_articles() - 1; // Minus the pillar
        
        $supporting = $research->generate_supporting_topics(
            $this->sphere->get_pillar_keyword(),
            $cluster_count
        );
        
        // Move to planning phase
        $this->sphere->plan_structure($this->persona, $supporting);
        $this->sphere->update_phase(AIAB_Thought_Sphere::PHASE_PLANNING);
        
        // Continue to next phase if time allows
        if ($this->has_time_remaining()) {
            $this->execute_phase();
        }
    }
    
    /**
     * Phase 2: Planning (create article records)
     */
    private function phase_planning() {
        $this->sphere->create_article_records();
        $this->sphere->update_phase(AIAB_Thought_Sphere::PHASE_WRITING_PILLAR);
        
        if ($this->has_time_remaining()) {
            $this->execute_phase();
        }
    }
    
    /**
     * Phase 3: Write pillar article
     */
    private function phase_write_pillar() {
        $this->sphere->update_status(AIAB_Thought_Sphere::STATUS_WRITING);
        
        $pillar = $this->sphere->get_pillar_article();
        
        if (!$pillar) {
            // No pillar record found, move on
            $this->sphere->update_phase(AIAB_Thought_Sphere::PHASE_WRITING_CLUSTERS);
            return;
        }
        
        // Check if pillar needs (re)writing
        $needs_writing = false;
        
        if ($pillar->status === 'planned') {
            $needs_writing = true;
            AIAB_Logger::debug("Pillar needs writing: planned status");
        } elseif ($pillar->status === 'failed') {
            // Check retry count before retrying
            $retry_count = isset($pillar->retry_count) ? (int) $pillar->retry_count : 0;
            if ($retry_count < 3) {
                $needs_writing = true;
                AIAB_Logger::debug("Pillar needs re-writing: previously failed (attempt " . ($retry_count + 1) . "/3)");
            } else {
                AIAB_Logger::error("Pillar permanently failed after max retries", array(
                    'keyword' => $pillar->keyword,
                    'retry_count' => $retry_count,
                    'last_error' => $pillar->last_error ?? 'unknown'
                ));
            }
        } elseif (in_array($pillar->status, array('auth_failed', 'max_retries', 'budget_failed'))) {
            // Skip permanently failed articles
            AIAB_Logger::error("Pillar permanently failed ({$pillar->status})", array(
                'keyword' => $pillar->keyword,
                'last_error' => $pillar->last_error ?? 'unknown'
            ));
        } elseif ($pillar->status === 'writing' && (int)$pillar->word_count < 100) {
            // Stuck in 'writing' status - reset and retry
            $needs_writing = true;
            AIAB_Logger::warning("Pillar stuck in 'writing' status with " . $pillar->word_count . " words - resetting");
            global $wpdb;
            $wpdb->update(
                AIAB_Database::get_table('articles'),
                array('status' => 'planned', 'content' => '', 'word_count' => 0),
                array('id' => $pillar->id)
            );
        } elseif ($pillar->status === 'written' && (int)$pillar->word_count < 450) {
            // Written but too short for a pillar
            $needs_writing = true;
            AIAB_Logger::warning("Pillar marked written but only " . $pillar->word_count . " words - re-writing");
            global $wpdb;
            $wpdb->update(
                AIAB_Database::get_table('articles'),
                array('status' => 'planned', 'content' => '', 'word_count' => 0),
                array('id' => $pillar->id)
            );
        }
        
        if ($needs_writing) {
            $this->write_single_article($pillar, true);
        } else {
            AIAB_Logger::debug("Pillar already written: " . $pillar->word_count . " words");
        }
        
        // Move to clusters phase
        $this->sphere->update_phase(AIAB_Thought_Sphere::PHASE_WRITING_CLUSTERS);
        
        if ($this->has_time_remaining()) {
            $this->execute_phase();
        } else {
            // Touch sphere when pausing to prevent being seen as stuck
            $this->touch_sphere();
            AIAB_Logger::info("Time limit reached after pillar, will continue clusters next cycle");
        }
    }
    
    /**
     * Phase 4: Write cluster articles
     */
    private function phase_write_clusters() {
        // First, ensure pillar is written - if not, go back to pillar phase
        $pillar = $this->sphere->get_pillar_article();
        $pillar_ok_statuses = array('written', 'published', 'linked', 'auth_failed', 'max_retries', 'budget_failed');
        
        if ($pillar && !in_array($pillar->status, $pillar_ok_statuses)) {
            AIAB_Logger::warning("Pillar not yet written (status: {$pillar->status}), returning to pillar phase");
            $this->sphere->update_phase(AIAB_Thought_Sphere::PHASE_WRITING_PILLAR);
            $this->execute_phase();
            return;
        }
        
        // If pillar is permanently failed, log warning but continue with clusters
        if ($pillar && in_array($pillar->status, array('auth_failed', 'max_retries', 'budget_failed'))) {
            AIAB_Logger::warning("⚠️ Pillar article permanently failed - continuing with cluster articles", array(
                'pillar_keyword' => $pillar->keyword,
                'pillar_status' => $pillar->status,
                'last_error' => $pillar->last_error ?? 'unknown'
            ));
        }
        
        $clusters = $this->sphere->get_cluster_articles();
        $sphere_data = $this->sphere->get_sphere_data();
        
        // Get pillar keyword for context
        $pillar_keyword = $this->sphere->get_pillar_keyword();
        
        // Get related topics for context
        $related_keywords = array_map(function($c) {
            return $c->keyword;
        }, $clusters);
        
        // Log the status of all cluster articles for debugging
        $status_summary = array();
        foreach ($clusters as $cluster) {
            $status_key = $cluster->status . '_' . ($cluster->word_count ?: 0) . 'w';
            if (!isset($status_summary[$status_key])) {
                $status_summary[$status_key] = 0;
            }
            $status_summary[$status_key]++;
        }
        AIAB_Logger::debug("Cluster articles status summary", array(
            'total' => count($clusters),
            'statuses' => $status_summary
        ));
        
        $articles_to_write = 0;
        
        foreach ($clusters as $cluster) {
            // Check if article needs (re)writing:
            // - status = 'planned' (never written)
            // - status = 'failed' (previous error)
            // - status = 'writing' with low word count (stuck mid-process)
            // - status = 'written' but word_count < 400 (failed silently)
            $needs_writing = false;
            
            if ($cluster->status === 'planned') {
                $needs_writing = true;
                AIAB_Logger::debug("Article needs writing: planned", array('keyword' => $cluster->keyword));
            } elseif ($cluster->status === 'failed') {
                // Check retry count before retrying
                $retry_count = isset($cluster->retry_count) ? (int) $cluster->retry_count : 0;
                if ($retry_count < 3) {
                    $needs_writing = true;
                    AIAB_Logger::debug("Article needs re-writing: previously failed (attempt " . ($retry_count + 1) . "/3)", array('keyword' => $cluster->keyword));
                } else {
                    AIAB_Logger::debug("Skipping article: max retries reached", array('keyword' => $cluster->keyword, 'retry_count' => $retry_count));
                }
            } elseif (in_array($cluster->status, array('auth_failed', 'max_retries', 'budget_failed'))) {
                // Skip permanently failed articles
                AIAB_Logger::debug("Skipping article: permanently failed ({$cluster->status})", array(
                    'keyword' => $cluster->keyword,
                    'last_error' => $cluster->last_error ?? 'unknown'
                ));
            } elseif ($cluster->status === 'writing' && (int)$cluster->word_count < 100) {
                // Stuck in 'writing' status - reset and retry
                $needs_writing = true;
                AIAB_Logger::warning("Article stuck in 'writing' status with " . $cluster->word_count . " words - resetting", array(
                    'keyword' => $cluster->keyword,
                    'word_count' => $cluster->word_count
                ));
                global $wpdb;
                $wpdb->update(
                    AIAB_Database::get_table('articles'),
                    array('status' => 'planned', 'content' => '', 'word_count' => 0),
                    array('id' => $cluster->id)
                );
            } elseif ($cluster->status === 'written' && (int)$cluster->word_count < 350) {
                $needs_writing = true;
                AIAB_Logger::warning("Article needs re-writing: marked written but only " . $cluster->word_count . " words", array(
                    'keyword' => $cluster->keyword,
                    'word_count' => $cluster->word_count
                ));
                // Reset status to planned so it gets rewritten
                global $wpdb;
                $wpdb->update(
                    AIAB_Database::get_table('articles'),
                    array('status' => 'planned', 'content' => '', 'word_count' => 0),
                    array('id' => $cluster->id)
                );
            }
            
            if (!$needs_writing) {
                continue; // Already successfully written
            }
            
            $articles_to_write++;
            
            if (!$this->has_time_remaining()) {
                // Touch the sphere to update its timestamp before returning
                $this->touch_sphere();
                
                AIAB_Logger::log("Time limit reached, pausing cluster writing", 'info', array(
                    'remaining_articles' => count(array_filter($clusters, function($c) {
                        return in_array($c->status, array('planned', 'failed')) || 
                               ($c->status === 'writing' && (int)$c->word_count < 100);
                    })),
                    'written_this_cycle' => $articles_to_write
                ));
                return; // Will continue next cycle
            }
            
            $context = array(
                'pillar_keyword' => $pillar_keyword,
                'related_topics' => array_diff($related_keywords, array($cluster->keyword))
            );
            
            $this->write_single_article($cluster, false, $context);
        }
        
        AIAB_Logger::info("Cluster writing phase complete", array(
            'articles_processed' => $articles_to_write
        ));
        
        // All clusters written, move to linking
        // This check handles both: just finished writing AND resuming with all already written
        if ($this->sphere->all_articles_written()) {
            AIAB_Logger::info("✅ All articles written - advancing to linking phase");
            $this->sphere->update_phase(AIAB_Thought_Sphere::PHASE_LINKING);
            
            // Also touch the sphere to update timestamp
            $this->touch_sphere();
            
            if ($this->has_time_remaining()) {
                $this->execute_phase();
            }
        } elseif ($articles_to_write === 0) {
            // Edge case: we're in writing_clusters phase but loop processed 0 articles
            // This can happen when:
            // 1. All remaining articles are permanently failed (auth_failed, budget_failed, max_retries)
            // 2. Articles are marked written but have low word counts
            
            // Check if we should force advance despite incomplete articles
            global $wpdb;
            $articles_table = AIAB_Database::get_table('articles');
            
            // Count articles that CAN potentially be worked on
            $workable = $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM $articles_table 
                 WHERE sphere_id = %d 
                 AND status IN ('planned', 'failed', 'writing')",
                $this->sphere->get_id()
            ));
            
            // Count written articles (even with low word counts)
            $written = $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM $articles_table 
                 WHERE sphere_id = %d 
                 AND status = 'written'",
                $this->sphere->get_id()
            ));
            
            if ($workable == 0 && $written > 0) {
                // No workable articles left, but we have written articles
                // Force advance to linking phase to publish what we have
                AIAB_Logger::warning("⚠️ No more articles can be written - force advancing to linking phase", array(
                    'sphere_id' => $this->sphere->get_id(),
                    'written_articles' => $written
                ));
                $this->sphere->update_phase(AIAB_Thought_Sphere::PHASE_LINKING);
                $this->touch_sphere();
                
                if ($this->has_time_remaining()) {
                    $this->execute_phase();
                }
            } else {
                AIAB_Logger::warning("⚠️ In writing_clusters phase but no articles processed", array(
                    'sphere_id' => $this->sphere->get_id(),
                    'workable_articles' => $workable,
                    'written_articles' => $written
                ));
            }
        }
    }
    
    /**
     * Write a single article
     */
    private function write_single_article($article, $is_pillar = false, $context = array()) {
        global $wpdb;
        $table = AIAB_Database::get_table('articles');
        
        AIAB_Logger::log("Writing article: " . $article->keyword, 'info', array(
            'article_id' => $article->id,
            'is_pillar' => $is_pillar,
            'article_type' => $article->article_type
        ));
        
        // Update status to writing
        $wpdb->update($table, array('status' => 'writing'), array('id' => $article->id));
        
        try {
            $writer = new AIAB_AI_Writer($this->persona);
            $article_data = $writer->write_article(
                $article->keyword,
                $article->article_type,
                $is_pillar,
                $context
            );
            
            // VALIDATION: Check if article content is sufficient
            // Note: OpenAI models limited to 4096 tokens produce ~500-600 words max
            // For longer articles, use Anthropic Claude which supports 8192 tokens
            // Be slightly forgiving - 450 for pillar, 350 for cluster (90% of target minimum)
            $min_words = $is_pillar ? 450 : 350;
            if ($article_data['word_count'] < $min_words) {
                AIAB_Logger::error("Article generation failed - content too short!", array(
                    'article_id' => $article->id,
                    'keyword' => $article->keyword,
                    'word_count' => $article_data['word_count'],
                    'min_required' => $min_words,
                    'content_preview' => substr(strip_tags($article_data['content']), 0, 500)
                ));
                
                // Mark as failed so it will be retried
                $wpdb->update($table, array(
                    'status' => 'failed',
                    'content' => 'FAILED: Only ' . $article_data['word_count'] . ' words generated. ' . substr($article_data['content'], 0, 1000),
                    'word_count' => $article_data['word_count']
                ), array('id' => $article->id));
                
                throw new Exception("Article content too short: " . $article_data['word_count'] . " words (minimum: $min_words)");
            }
            
            // Generate featured image
            $image_gen = new AIAB_Image_Generator();
            $image_id = $image_gen->generate(
                $article_data['title'],
                $article->keyword,
                $article_data['alt_text']
            );
            
            // Update article record
            $wpdb->update($table, array(
                'title' => $article_data['title'],
                'slug' => sanitize_title($article_data['title']),
                'content' => $article_data['content'],
                'excerpt' => $article_data['excerpt'],
                'meta_description' => $article_data['meta_description'],
                'featured_image_id' => $image_id,
                'featured_image_alt' => $article_data['alt_text'],
                'word_count' => $article_data['word_count'],
                'status' => 'written'
            ), array('id' => $article->id));
            
            AIAB_Logger::info("✅ Article written successfully: " . $article_data['title'], array(
                'word_count' => $article_data['word_count'],
                'focus_keyword' => $article_data['focus_keyword'] ?? 'N/A'
            ));
            
            // WALTER COOLDOWN: Give CPU a break to avoid VPS throttling
            $provider = get_option('aiab_ai_provider', 'anthropic');
            if ($provider === 'walter') {
                $cooldown_seconds = (int) get_option('aiab_walter_cooldown', 60); // Default 60 seconds
                if ($cooldown_seconds > 0) {
                    AIAB_Logger::info("🧊 Custom Server cooldown: Pausing {$cooldown_seconds}s to prevent CPU throttling...");
                    sleep($cooldown_seconds);
                    AIAB_Logger::debug("🧊 Cooldown complete, continuing...");
                }
            }
            
            // Reset retry count on success
            $wpdb->update($table, array('retry_count' => 0, 'last_error' => null), array('id' => $article->id));
            
        } catch (Exception $e) {
            $error_message = $e->getMessage();
            $max_retries = 3;
            
            // Get current retry count
            $current_retry = (int) $wpdb->get_var($wpdb->prepare(
                "SELECT retry_count FROM $table WHERE id = %d",
                $article->id
            ));
            $new_retry_count = $current_retry + 1;
            
            // Check if this is an authentication error (should not retry)
            $is_auth_error = strpos($error_message, 'Authentication failed') !== false 
                          || strpos($error_message, 'API key') !== false
                          || strpos($error_message, '401') !== false;
            
            // Check if this is a budget/quota error (should not retry - needs payment)
            $is_budget_error = strpos($error_message, 'Budget') !== false
                            || strpos($error_message, 'Quota') !== false
                            || strpos($error_message, 'Payment required') !== false
                            || strpos($error_message, '402') !== false
                            || strpos($error_message, 'exceeded') !== false
                            || strpos($error_message, 'insufficient') !== false;
            
            if ($is_auth_error) {
                // Don't retry auth errors - mark as permanently failed
                AIAB_Logger::error("🔴 API Authentication failed - article marked as permanently failed", array(
                    'article_id' => $article->id,
                    'keyword' => $article->keyword,
                    'error' => $error_message
                ));
                $wpdb->update($table, array(
                    'status' => 'auth_failed',
                    'last_error' => substr($error_message, 0, 500),
                    'retry_count' => $new_retry_count
                ), array('id' => $article->id));
            } elseif ($is_budget_error) {
                // Don't retry budget errors - needs payment/credits
                AIAB_Logger::error("🔴 API Budget/Quota exceeded - article marked as budget_failed", array(
                    'article_id' => $article->id,
                    'keyword' => $article->keyword,
                    'error' => $error_message
                ));
                $wpdb->update($table, array(
                    'status' => 'budget_failed',
                    'last_error' => substr($error_message, 0, 500),
                    'retry_count' => $new_retry_count
                ), array('id' => $article->id));
            } elseif ($new_retry_count >= $max_retries) {
                // Max retries reached
                AIAB_Logger::error("❌ Article failed after $max_retries attempts - marked as permanently failed", array(
                    'article_id' => $article->id,
                    'keyword' => $article->keyword,
                    'retry_count' => $new_retry_count,
                    'error' => $error_message
                ));
                $wpdb->update($table, array(
                    'status' => 'max_retries',
                    'last_error' => substr($error_message, 0, 500),
                    'retry_count' => $new_retry_count
                ), array('id' => $article->id));
            } else {
                // Retry later
                AIAB_Logger::warning("⚠️ Article failed (attempt $new_retry_count/$max_retries) - will retry", array(
                    'article_id' => $article->id,
                    'keyword' => $article->keyword,
                    'error' => $error_message
                ));
                $wpdb->update($table, array(
                    'status' => 'failed',
                    'last_error' => substr($error_message, 0, 500),
                    'retry_count' => $new_retry_count
                ), array('id' => $article->id));
            }
            // Don't re-throw - allow other articles to continue
        }
    }
    
    /**
     * Phase 5: Internal and external linking
     */
    private function phase_linking() {
        // First, publish all articles (without links yet)
        $publisher = new AIAB_Article_Publisher($this->persona);
        $articles = $this->sphere->get_articles('written');
        
        foreach ($articles as $article) {
            if (!$article->wp_post_id) {
                $article_data = array(
                    'title' => $article->title,
                    'content' => $article->content,
                    'excerpt' => $article->excerpt,
                    'meta_description' => $article->meta_description,
                    'keyword' => $article->keyword,
                    'focus_keyword' => $this->extract_focus_keyword($article->keyword), // For RankMath
                    'article_type' => $article->article_type,
                    'tags' => array(),
                    'is_pillar' => (bool) $article->is_pillar,
                    'featured_image_id' => $article->featured_image_id,
                    'word_count' => $article->word_count
                );
                
                $publisher->publish($article_data, $article->id);
                $this->sphere->increment_published();
            }
        }
        
        // Now inject links
        $link_manager = new AIAB_Link_Manager();
        $link_manager->process_sphere_links($this->sphere);
        
        $this->sphere->update_phase(AIAB_Thought_Sphere::PHASE_PUBLISHING);
        
        if ($this->has_time_remaining()) {
            $this->execute_phase();
        }
    }
    
    /**
     * Phase 6: Final publishing verification
     */
    private function phase_publishing() {
        $this->sphere->update_status(AIAB_Thought_Sphere::STATUS_PUBLISHING);
        
        // Get retry count from sphere meta
        global $wpdb;
        $table = AIAB_Database::get_table('thought_spheres');
        $retry_count = (int) $wpdb->get_var($wpdb->prepare(
            "SELECT JSON_EXTRACT(research_data, '$.publish_retries') FROM $table WHERE id = %d",
            $this->sphere->get_id()
        ));
        
        // Verify all articles are published
        if ($this->sphere->all_articles_published()) {
            $this->sphere->update_phase(AIAB_Thought_Sphere::PHASE_COMPLETE);
            $this->sphere->update_status(AIAB_Thought_Sphere::STATUS_COMPLETED);
            
            AIAB_Logger::log("SPHERE COMPLETE: " . $this->sphere->get_pillar_keyword(), 'info', array(
                'sphere_id' => $this->sphere->get_id(),
                'total_articles' => $this->sphere->get_total_articles(),
                'published' => $this->sphere->get_published_articles()
            ));
        } else {
            // Increment retry counter
            $retry_count++;
            $wpdb->query($wpdb->prepare(
                "UPDATE $table SET research_data = JSON_SET(COALESCE(research_data, '{}'), '$.publish_retries', %d) WHERE id = %d",
                $retry_count,
                $this->sphere->get_id()
            ));
            
            // Check if we've exceeded max retries (prevent infinite loop)
            $max_retries = 5;
            if ($retry_count >= $max_retries) {
                AIAB_Logger::log("Max publish retries reached ($max_retries), forcing sphere to complete", 'warning', array(
                    'sphere_id' => $this->sphere->get_id(),
                    'retry_count' => $retry_count
                ));
                
                // Force complete the sphere even with unpublished articles
                $this->sphere->update_phase(AIAB_Thought_Sphere::PHASE_COMPLETE);
                $this->sphere->update_status(AIAB_Thought_Sphere::STATUS_COMPLETED);
                
                // Log which articles weren't published
                $unpublished = $this->sphere->get_unpublished_articles();
                if (!empty($unpublished)) {
                    AIAB_Logger::log("Unpublished articles skipped", 'warning', array(
                        'count' => count($unpublished),
                        'articles' => array_map(function($a) { return $a->keyword; }, $unpublished)
                    ));
                }
            } else {
                // Try to recover - go back to linking phase
                AIAB_Logger::log("Not all articles published, retrying... (attempt $retry_count/$max_retries)", 'warning');
                $this->sphere->update_phase(AIAB_Thought_Sphere::PHASE_LINKING);
            }
        }
    }
    
    /**
     * Check if we have time remaining in this execution cycle
     */
    private function has_time_remaining() {
        $elapsed = time() - $this->start_time;
        $remaining = $this->max_execution_time - $elapsed;
        
        // Keep 30 second buffer
        return $remaining > 30;
    }
    
    /**
     * Touch the sphere to update its timestamp
     * This prevents it from being flagged as "stuck" by auto-heal
     */
    private function touch_sphere() {
        if (!$this->sphere) {
            return;
        }
        
        global $wpdb;
        $table = AIAB_Database::get_table('thought_spheres');
        
        // Check if updated_at column exists
        $has_updated_at = $wpdb->get_var(
            "SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS 
             WHERE TABLE_SCHEMA = DATABASE() 
             AND TABLE_NAME = '$table' 
             AND COLUMN_NAME = 'updated_at'"
        );
        
        if ($has_updated_at) {
            $wpdb->query($wpdb->prepare(
                "UPDATE $table SET updated_at = NOW() WHERE id = %d",
                $this->sphere->get_id()
            ));
        }
        
        // Also save sphere to trigger any ON UPDATE behavior
        $this->sphere->save();
    }
    
    /**
     * Manual run with specific persona
     */
    public function run_for_persona($persona_id) {
        $this->persona = AIAB_Persona::get($persona_id);
        
        if (!$this->persona) {
            throw new Exception("Persona not found: $persona_id");
        }
        
        $this->run();
    }
    
    /**
     * Extract optimized focus keyword from full keyword phrase
     * Used for RankMath SEO optimization
     */
    private function extract_focus_keyword($keyword) {
        $focus = strtolower(trim($keyword));
        
        // Remove common question patterns
        $patterns = array(
            '/^what (is|are|does|do|can|should|would|will|distinguishes?|makes?|causes?) /i',
            '/^why (is|are|does|do|did|would|should|can) /i',
            '/^how (to|do|does|can|much|many|long|often|is|are) /i',
            '/^when (to|should|do|does|is|are|can|will) /i',
            '/^where (to|can|do|does|is|are|should) /i',
            '/^which (is|are|one|type|kind) /i',
            '/^who (is|are|can|should|does) /i',
            '/^can (you|i|we|they|it|this) /i',
            '/^should (you|i|we|they|it) /i',
            '/^is (it|there|this|that) /i',
            '/^are (there|these|those) /i',
            '/^does (it|this|that) /i',
            '/^do (i|you|we|they) /i',
        );
        
        foreach ($patterns as $pattern) {
            $focus = preg_replace($pattern, '', $focus);
        }
        
        // Remove trailing question mark and filler words
        $focus = rtrim($focus, '?');
        $focus = preg_replace('/\s+(work|be|happen|change|differ|vary|mean|matter)\s*$/i', '', $focus);
        
        // Clean up
        $focus = trim($focus);
        $focus = preg_replace('/[^\w\s\'-]/u', '', $focus);
        $focus = preg_replace('/\s+/', ' ', $focus);
        
        // If still very long, take last 4 words (usually the core topic)
        $words = explode(' ', $focus);
        if (count($words) > 5) {
            $focus = implode(' ', array_slice($words, -4));
        }
        
        return $focus;
    }
    
    /**
     * Get current status for admin display
     */
    public function get_status() {
        return array(
            'persona' => $this->persona ? $this->persona->get_name() : null,
            'sphere' => $this->sphere ? array(
                'id' => $this->sphere->get_id(),
                'pillar' => $this->sphere->get_pillar_keyword(),
                'status' => $this->sphere->get_status(),
                'phase' => $this->sphere->get_phase(),
                'progress' => $this->sphere->get_published_articles() . '/' . $this->sphere->get_total_articles()
            ) : null,
            'elapsed' => time() - $this->start_time,
            'time_remaining' => $this->max_execution_time - (time() - $this->start_time)
        );
    }
}
