<?php
/**
 * Persona Model
 * Represents an AI Author with their interests, passions, and writing style
 */

if (!defined('ABSPATH')) {
    exit;
}

class AIAB_Persona {
    
    private $id;
    private $name;
    private $slug;
    private $wp_author_id;
    private $categories = array();
    private $interests = array();
    private $passions = array();
    private $expertise_areas = array();
    private $localities = array();
    private $voice_style;
    private $tone;
    private $knowledge_sources = array();
    private $writing_guidelines;
    private $avoid_topics = array();
    private $personal_story;
    private $target_audience;
    private $language;
    private $is_active;
    private $total_articles;
    private $total_spheres;
    
    /**
     * Constructor
     */
    public function __construct($data = null) {
        if ($data) {
            $this->hydrate($data);
        }
    }
    
    /**
     * Hydrate from database row
     */
    private function hydrate($data) {
        $this->id = isset($data->id) ? intval($data->id) : null;
        $this->name = isset($data->name) ? $data->name : '';
        $this->slug = isset($data->slug) ? $data->slug : '';
        $this->wp_author_id = isset($data->wp_author_id) ? intval($data->wp_author_id) : null;
        $this->categories = isset($data->categories) ? json_decode($data->categories, true) : array();
        $this->interests = isset($data->interests) ? json_decode($data->interests, true) : array();
        $this->passions = isset($data->passions) ? json_decode($data->passions, true) : array();
        $this->expertise_areas = isset($data->expertise_areas) ? json_decode($data->expertise_areas, true) : array();
        $this->localities = isset($data->localities) ? json_decode($data->localities, true) : array();
        $this->voice_style = isset($data->voice_style) ? $data->voice_style : 'professional';
        $this->tone = isset($data->tone) ? $data->tone : 'informative';
        $this->knowledge_sources = isset($data->knowledge_sources) ? json_decode($data->knowledge_sources, true) : array();
        $this->writing_guidelines = isset($data->writing_guidelines) ? $data->writing_guidelines : '';
        $this->avoid_topics = isset($data->avoid_topics) ? json_decode($data->avoid_topics, true) : array();
        $this->personal_story = isset($data->personal_story) ? $data->personal_story : '';
        $this->target_audience = isset($data->target_audience) ? $data->target_audience : '';
        $this->language = isset($data->language) ? $data->language : 'en';
        $this->is_active = isset($data->is_active) ? (bool) $data->is_active : true;
        $this->total_articles = isset($data->total_articles) ? intval($data->total_articles) : 0;
        $this->total_spheres = isset($data->total_spheres) ? intval($data->total_spheres) : 0;
    }
    
    /**
     * Get by ID
     */
    public static function get($id) {
        global $wpdb;
        $table = AIAB_Database::get_table('personas');
        
        $data = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM $table WHERE id = %d",
            $id
        ));
        
        return $data ? new self($data) : null;
    }
    
    /**
     * Get all personas
     */
    public static function get_all($active_only = false) {
        global $wpdb;
        $table = AIAB_Database::get_table('personas');
        
        $sql = "SELECT * FROM $table";
        if ($active_only) {
            $sql .= " WHERE is_active = 1";
        }
        $sql .= " ORDER BY name ASC";
        
        $results = $wpdb->get_results($sql);
        $personas = array();
        
        foreach ($results as $data) {
            $personas[] = new self($data);
        }
        
        return $personas;
    }
    
    /**
     * Get random active persona for the cycle
     */
    public static function get_random_active() {
        global $wpdb;
        $table = AIAB_Database::get_table('personas');
        
        $data = $wpdb->get_row(
            "SELECT * FROM $table WHERE is_active = 1 ORDER BY RAND() LIMIT 1"
        );
        
        return $data ? new self($data) : null;
    }
    
    /**
     * Get the persona that has written the least recently
     */
    public static function get_least_recent() {
        global $wpdb;
        $personas_table = AIAB_Database::get_table('personas');
        $spheres_table = AIAB_Database::get_table('thought_spheres');
        
        $data = $wpdb->get_row(
            "SELECT p.* FROM $personas_table p 
             LEFT JOIN $spheres_table s ON p.id = s.persona_id 
             WHERE p.is_active = 1 
             GROUP BY p.id 
             ORDER BY MAX(s.created_at) ASC, p.id ASC 
             LIMIT 1"
        );
        
        return $data ? new self($data) : null;
    }
    
    /**
     * Save persona
     */
    public function save() {
        global $wpdb;
        $table = AIAB_Database::get_table('personas');
        
        // Check if table exists
        $table_exists = $wpdb->get_var("SHOW TABLES LIKE '$table'");
        if (!$table_exists) {
            AIAB_Database::create_tables();
        }
        
        // Prepare all data with proper defaults
        $name = $this->name ?: 'Unnamed Persona';
        $slug = $this->slug ?: sanitize_title($name);
        
        // Ensure slug doesn't exceed 191 chars (MySQL 5.5 index limit)
        $slug = substr($slug, 0, 191);
        
        // For new persona, ensure unique slug
        if (!$this->id) {
            $base_slug = $slug;
            $counter = 1;
            while ($wpdb->get_var($wpdb->prepare("SELECT id FROM $table WHERE slug = %s", $slug))) {
                $slug = substr($base_slug, 0, 180) . '-' . $counter;
                $counter++;
            }
        }
        
        // Current timestamp for MySQL 5.5 compatibility
        $now = current_time('mysql');
        
        // Sanitize all string values and ensure proper lengths
        $target_audience = is_string($this->target_audience) ? substr($this->target_audience, 0, 255) : '';
        $voice_style = is_string($this->voice_style) ? substr($this->voice_style, 0, 100) : 'professional';
        $tone = is_string($this->tone) ? substr($this->tone, 0, 100) : 'informative';
        $language = is_string($this->language) ? substr($this->language, 0, 50) : 'English';
        $writing_guidelines = is_string($this->writing_guidelines) ? $this->writing_guidelines : '';
        $personal_story = is_string($this->personal_story) ? $this->personal_story : '';
        
        // Build data array with explicit types
        $data = array(
            'name' => substr($name, 0, 255),
            'slug' => $slug,
            'categories' => json_encode($this->categories ?: array()),
            'interests' => json_encode($this->interests ?: array()),
            'passions' => json_encode($this->passions ?: array()),
            'expertise_areas' => json_encode($this->expertise_areas ?: array()),
            'localities' => json_encode($this->localities ?: array()),
            'voice_style' => $voice_style,
            'tone' => $tone,
            'knowledge_sources' => json_encode($this->knowledge_sources ?: array()),
            'writing_guidelines' => $writing_guidelines,
            'avoid_topics' => json_encode($this->avoid_topics ?: array()),
            'personal_story' => $personal_story,
            'target_audience' => $target_audience,
            'language' => $language,
            'is_active' => $this->is_active ? 1 : 0,
            'updated_at' => $now,
        );
        
        // Format specifiers for wpdb
        $format = array(
            '%s', // name
            '%s', // slug
            '%s', // categories
            '%s', // interests
            '%s', // passions
            '%s', // expertise_areas
            '%s', // localities
            '%s', // voice_style
            '%s', // tone
            '%s', // knowledge_sources
            '%s', // writing_guidelines
            '%s', // avoid_topics
            '%s', // personal_story
            '%s', // target_audience
            '%s', // language
            '%d', // is_active
            '%s', // updated_at
        );
        
        // Add wp_author_id if set
        if ($this->wp_author_id && $this->wp_author_id > 0) {
            $data['wp_author_id'] = intval($this->wp_author_id);
            $format[] = '%d';
        }
        
        if ($this->id) {
            // For update, handle NULL wp_author_id explicitly
            if (!$this->wp_author_id || $this->wp_author_id <= 0) {
                $wpdb->query($wpdb->prepare("UPDATE $table SET wp_author_id = NULL WHERE id = %d", $this->id));
            }
            
            // Update existing
            $result = $wpdb->update($table, $data, array('id' => $this->id), $format, array('%d'));
            
            if ($result === false) {
                error_log("AIAB Persona Update Failed. Error: " . $wpdb->last_error);
                return false;
            }
            return $this->id;
        } else {
            // Add created_at for new records
            $data['created_at'] = $now;
            $format[] = '%s';
            
            // Insert new
            $result = $wpdb->insert($table, $data, $format);
            
            if ($result === false) {
                error_log("AIAB Persona Insert Failed. Error: " . $wpdb->last_error);
                return false;
            }
            
            $this->id = $wpdb->insert_id;
            return $this->id;
        }
    }
    
    /**
     * Delete persona
     */
    public function delete() {
        global $wpdb;
        $table = AIAB_Database::get_table('personas');
        return $wpdb->delete($table, array('id' => $this->id));
    }
    
    /**
     * Increment article count
     */
    public function increment_articles($count = 1) {
        global $wpdb;
        $table = AIAB_Database::get_table('personas');
        
        $wpdb->query($wpdb->prepare(
            "UPDATE $table SET total_articles = total_articles + %d WHERE id = %d",
            $count,
            $this->id
        ));
        
        $this->total_articles += $count;
    }
    
    /**
     * Increment sphere count
     */
    public function increment_spheres() {
        global $wpdb;
        $table = AIAB_Database::get_table('personas');
        
        $wpdb->query($wpdb->prepare(
            "UPDATE $table SET total_spheres = total_spheres + 1 WHERE id = %d",
            $this->id
        ));
        
        $this->total_spheres++;
    }
    
    /**
     * Generate system prompt for AI
     */
    public function generate_system_prompt() {
        $prompt = "You are {$this->name}, a knowledgeable writer and content creator.\n\n";
        
        // Personal story - shapes voice and perspective
        if (!empty($this->personal_story)) {
            $prompt .= "YOUR BACKGROUND & PERSONAL STORY:\n";
            $prompt .= "{$this->personal_story}\n\n";
            $prompt .= "Draw from this personal experience to make your writing authentic and relatable. ";
            $prompt .= "Reference your journey and lessons learned when relevant. ";
            $prompt .= "Let your unique perspective shine through in your content.\n\n";
        }
        
        $prompt .= "YOUR EXPERTISE:\n";
        foreach ($this->expertise_areas as $area) {
            $prompt .= "- {$area}\n";
        }
        
        $prompt .= "\nYOUR INTERESTS:\n";
        foreach ($this->interests as $interest) {
            $prompt .= "- {$interest}\n";
        }
        
        $prompt .= "\nYOUR PASSIONS:\n";
        foreach ($this->passions as $passion) {
            $prompt .= "- {$passion}\n";
        }
        
        // Localities - Local SEO targeting with EXPLICIT localization
        if (!empty($this->localities)) {
            $prompt .= "\n🌍 TARGET LOCALITIES (LOCAL SEO):\n";
            $prompt .= "You are writing specifically for audiences in these locations:\n";
            
            $primary_locality = $this->localities[0];
            $localization = $this->get_localization_data($primary_locality);
            
            foreach ($this->localities as $locality) {
                $loc_name = $this->get_locality_display_name($locality);
                $prompt .= "- {$loc_name}\n";
            }
            
            $prompt .= "\n⚠️ MANDATORY LOCALIZATION RULES - YOU MUST FOLLOW THESE:\n";
            $prompt .= "Currency: ALWAYS use {$localization['currency_symbol']} ({$localization['currency_name']})\n";
            $prompt .= "  - Format prices as: {$localization['currency_format']}\n";
            $prompt .= "  - NEVER use \$, €, £, or any other currency unless specifically about that country\n";
            $prompt .= "Measurements: Use {$localization['measurement']} system\n";
            if ($localization['measurement'] === 'metric') {
                $prompt .= "  - Distances in kilometers (km), meters (m)\n";
                $prompt .= "  - Weights in kilograms (kg), grams (g)\n";
                $prompt .= "  - Volume in liters (L), milliliters (ml)\n";
            } else {
                $prompt .= "  - Distances in miles, feet, inches\n";
                $prompt .= "  - Weights in pounds (lbs), ounces (oz)\n";
                $prompt .= "  - Volume in gallons, quarts, cups\n";
            }
            $prompt .= "Temperature: Use {$localization['temperature']}\n";
            $prompt .= "Spelling: Use {$localization['spelling']}\n";
            if (strpos($localization['spelling'], 'British') !== false) {
                $prompt .= "  - colour (not color), favour (not favor), organise (not organize)\n";
                $prompt .= "  - centre (not center), metre (not meter), litre (not liter)\n";
                $prompt .= "  - realise (not realize), analyse (not analyze)\n";
            } elseif (strpos($localization['spelling'], 'American') !== false) {
                $prompt .= "  - color (not colour), favor (not favour), organize (not organise)\n";
                $prompt .= "  - center (not centre), meter (not metre), liter (not litre)\n";
            }
            $prompt .= "Dates: Format as {$localization['date_format']}\n";
            $prompt .= "Paper sizes: Reference {$localization['paper_size']} (not Letter/A4 from other regions)\n";
            
            $prompt .= "\nLOCAL CONTENT GUIDELINES:\n";
            $prompt .= "- Write content specifically relevant to these geographic areas\n";
            $prompt .= "- Reference local landmarks, neighborhoods, or regions when relevant\n";
            $prompt .= "- Consider local climate, regulations, customs, and market conditions\n";
            $prompt .= "- Address concerns and questions specific to these local audiences\n";
            $prompt .= "- Use local terminology and conventions throughout\n";
            $prompt .= "- When mentioning costs, salaries, or prices, use realistic local figures in {$localization['currency_symbol']}\n\n";
        }
        
        $prompt .= "\nWRITING STYLE:\n";
        $prompt .= "- Voice: {$this->voice_style}\n";
        $prompt .= "- Tone: {$this->tone}\n";
        $prompt .= "- Target Audience: {$this->target_audience}\n";
        
        if (!empty($this->writing_guidelines)) {
            $prompt .= "\nWRITING GUIDELINES:\n{$this->writing_guidelines}\n";
        }
        
        if (!empty($this->avoid_topics)) {
            $prompt .= "\nTOPICS TO AVOID:\n";
            foreach ($this->avoid_topics as $topic) {
                $prompt .= "- {$topic}\n";
            }
        }
        
        if (!empty($this->knowledge_sources)) {
            $prompt .= "\nKNOWLEDGE SOURCES (cite when relevant):\n";
            foreach ($this->knowledge_sources as $source) {
                $prompt .= "- {$source}\n";
            }
        }
        
        return $prompt;
    }
    
    /**
     * Get display name for locality code
     */
    private function get_locality_display_name($code) {
        $localities = array(
            'ae' => 'United Arab Emirates',
            'ae-dubai' => 'Dubai, UAE',
            'ae-abudhabi' => 'Abu Dhabi, UAE',
            'ae-sharjah' => 'Sharjah, UAE',
            'ae-ajman' => 'Ajman, UAE',
            'sa' => 'Saudi Arabia',
            'sa-riyadh' => 'Riyadh, Saudi Arabia',
            'sa-jeddah' => 'Jeddah, Saudi Arabia',
            'qa' => 'Qatar',
            'qa-doha' => 'Doha, Qatar',
            'kw' => 'Kuwait',
            'bh' => 'Bahrain',
            'om' => 'Oman',
            'us' => 'United States',
            'us-newyork' => 'New York, USA',
            'us-california' => 'California, USA',
            'us-texas' => 'Texas, USA',
            'us-florida' => 'Florida, USA',
            'ca' => 'Canada',
            'ca-ontario' => 'Ontario, Canada',
            'mx' => 'Mexico',
            'gb' => 'United Kingdom',
            'gb-london' => 'London, UK',
            'de' => 'Germany',
            'fr' => 'France',
            'es' => 'Spain',
            'it' => 'Italy',
            'nl' => 'Netherlands',
            'ch' => 'Switzerland',
            'au' => 'Australia',
            'au-sydney' => 'Sydney, Australia',
            'au-melbourne' => 'Melbourne, Australia',
            'sg' => 'Singapore',
            'hk' => 'Hong Kong',
            'jp' => 'Japan',
            'in' => 'India',
            'in-mumbai' => 'Mumbai, India',
            'in-delhi' => 'Delhi, India',
            'ph' => 'Philippines',
            'my' => 'Malaysia',
            'za' => 'South Africa',
            'eg' => 'Egypt',
            'ng' => 'Nigeria',
            'ke' => 'Kenya',
        );
        
        return isset($localities[$code]) ? $localities[$code] : $code;
    }
    
    /**
     * Get comprehensive localization data for a locality
     * Accepts both old codes (ae, gb) and new city strings (Dubai, United Arab Emirates)
     * Returns currency, measurements, spelling, date format, etc.
     */
    public function get_localization_data($locality) {
        // Map country names to codes
        $country_to_code = array(
            'united arab emirates' => 'ae',
            'uae' => 'ae',
            'saudi arabia' => 'sa',
            'ksa' => 'sa',
            'qatar' => 'qa',
            'kuwait' => 'kw',
            'bahrain' => 'bh',
            'oman' => 'om',
            'united states' => 'us',
            'usa' => 'us',
            'united kingdom' => 'gb',
            'uk' => 'gb',
            'canada' => 'ca',
            'mexico' => 'mx',
            'germany' => 'de',
            'france' => 'fr',
            'spain' => 'es',
            'italy' => 'it',
            'netherlands' => 'nl',
            'switzerland' => 'ch',
            'australia' => 'au',
            'singapore' => 'sg',
            'hong kong' => 'hk',
            'japan' => 'jp',
            'india' => 'in',
            'philippines' => 'ph',
            'malaysia' => 'my',
            'south africa' => 'za',
            'egypt' => 'eg',
            'nigeria' => 'ng',
            'kenya' => 'ke',
            'brazil' => 'br',
            'argentina' => 'ar',
            'china' => 'cn',
            'south korea' => 'kr',
            'taiwan' => 'tw',
            'thailand' => 'th',
            'vietnam' => 'vn',
            'indonesia' => 'id',
            'new zealand' => 'nz',
            'ireland' => 'ie',
            'portugal' => 'pt',
            'sweden' => 'se',
            'norway' => 'no',
            'denmark' => 'dk',
            'finland' => 'fi',
            'poland' => 'pl',
            'czech republic' => 'cz',
            'hungary' => 'hu',
            'austria' => 'at',
            'belgium' => 'be',
            'greece' => 'gr',
            'turkey' => 'tr',
            'russia' => 'ru',
            'israel' => 'il',
            'jordan' => 'jo',
            'lebanon' => 'lb',
            'morocco' => 'ma',
            'tunisia' => 'tn',
            'ethiopia' => 'et',
            'ghana' => 'gh',
            'peru' => 'pe',
            'colombia' => 'co',
            'chile' => 'cl',
            'venezuela' => 've',
        );
        
        // Determine country code
        $country_code = null;
        
        // Check if it's already a 2-letter code
        if (strlen($locality) === 2) {
            $country_code = strtolower($locality);
        }
        // Check if it's an old format code like "ae-dubai"
        elseif (preg_match('/^([a-z]{2})-/', strtolower($locality), $matches)) {
            $country_code = $matches[1];
        }
        // It's a city string like "Dubai, United Arab Emirates"
        else {
            $locality_lower = strtolower($locality);
            
            // Try to find country in the string
            foreach ($country_to_code as $country_name => $code) {
                if (strpos($locality_lower, $country_name) !== false) {
                    $country_code = $code;
                    break;
                }
            }
            
            // Special case for city-states and ambiguous names
            if (!$country_code) {
                if (stripos($locality, 'Singapore') !== false) {
                    $country_code = 'sg';
                } elseif (stripos($locality, 'Hong Kong') !== false) {
                    $country_code = 'hk';
                }
            }
        }
        
        $localization = array(
            // Gulf Countries
            'ae' => array(
                'currency_symbol' => 'AED',
                'currency_name' => 'UAE Dirham',
                'currency_format' => 'AED X,XXX or X,XXX AED',
                'measurement' => 'metric',
                'spelling' => 'British English',
                'date_format' => 'DD/MM/YYYY',
                'temperature' => 'Celsius',
                'driving_side' => 'right',
                'paper_size' => 'A4',
                'week_start' => 'Sunday',
                'decimal_separator' => '.',
                'thousands_separator' => ',',
            ),
            'sa' => array(
                'currency_symbol' => 'SAR',
                'currency_name' => 'Saudi Riyal',
                'currency_format' => 'SAR X,XXX or X,XXX SAR',
                'measurement' => 'metric',
                'spelling' => 'British English',
                'date_format' => 'DD/MM/YYYY',
                'temperature' => 'Celsius',
                'driving_side' => 'right',
                'paper_size' => 'A4',
                'week_start' => 'Sunday',
                'decimal_separator' => '.',
                'thousands_separator' => ',',
            ),
            'qa' => array(
                'currency_symbol' => 'QAR',
                'currency_name' => 'Qatari Riyal',
                'currency_format' => 'QAR X,XXX',
                'measurement' => 'metric',
                'spelling' => 'British English',
                'date_format' => 'DD/MM/YYYY',
                'temperature' => 'Celsius',
                'driving_side' => 'right',
                'paper_size' => 'A4',
                'week_start' => 'Sunday',
                'decimal_separator' => '.',
                'thousands_separator' => ',',
            ),
            'kw' => array(
                'currency_symbol' => 'KWD',
                'currency_name' => 'Kuwaiti Dinar',
                'currency_format' => 'KWD X.XXX',
                'measurement' => 'metric',
                'spelling' => 'British English',
                'date_format' => 'DD/MM/YYYY',
                'temperature' => 'Celsius',
                'driving_side' => 'right',
                'paper_size' => 'A4',
                'week_start' => 'Sunday',
                'decimal_separator' => '.',
                'thousands_separator' => ',',
            ),
            'bh' => array(
                'currency_symbol' => 'BHD',
                'currency_name' => 'Bahraini Dinar',
                'currency_format' => 'BHD X.XXX',
                'measurement' => 'metric',
                'spelling' => 'British English',
                'date_format' => 'DD/MM/YYYY',
                'temperature' => 'Celsius',
                'driving_side' => 'right',
                'paper_size' => 'A4',
                'week_start' => 'Sunday',
                'decimal_separator' => '.',
                'thousands_separator' => ',',
            ),
            'om' => array(
                'currency_symbol' => 'OMR',
                'currency_name' => 'Omani Rial',
                'currency_format' => 'OMR X.XXX',
                'measurement' => 'metric',
                'spelling' => 'British English',
                'date_format' => 'DD/MM/YYYY',
                'temperature' => 'Celsius',
                'driving_side' => 'right',
                'paper_size' => 'A4',
                'week_start' => 'Sunday',
                'decimal_separator' => '.',
                'thousands_separator' => ',',
            ),
            // Americas
            'us' => array(
                'currency_symbol' => '$',
                'currency_name' => 'US Dollar',
                'currency_format' => '$X,XXX.XX',
                'measurement' => 'imperial',
                'spelling' => 'American English',
                'date_format' => 'MM/DD/YYYY',
                'temperature' => 'Fahrenheit',
                'driving_side' => 'right',
                'paper_size' => 'Letter',
                'week_start' => 'Sunday',
                'decimal_separator' => '.',
                'thousands_separator' => ',',
            ),
            'ca' => array(
                'currency_symbol' => 'C$',
                'currency_name' => 'Canadian Dollar',
                'currency_format' => '$X,XXX.XX CAD',
                'measurement' => 'metric',
                'spelling' => 'Canadian English (British-based)',
                'date_format' => 'YYYY-MM-DD or DD/MM/YYYY',
                'temperature' => 'Celsius',
                'driving_side' => 'right',
                'paper_size' => 'Letter',
                'week_start' => 'Sunday',
                'decimal_separator' => '.',
                'thousands_separator' => ',',
            ),
            'mx' => array(
                'currency_symbol' => 'MX$',
                'currency_name' => 'Mexican Peso',
                'currency_format' => '$X,XXX.XX MXN',
                'measurement' => 'metric',
                'spelling' => 'Spanish',
                'date_format' => 'DD/MM/YYYY',
                'temperature' => 'Celsius',
                'driving_side' => 'right',
                'paper_size' => 'Letter',
                'week_start' => 'Monday',
                'decimal_separator' => '.',
                'thousands_separator' => ',',
            ),
            'br' => array(
                'currency_symbol' => 'R$',
                'currency_name' => 'Brazilian Real',
                'currency_format' => 'R$ X.XXX,XX',
                'measurement' => 'metric',
                'spelling' => 'Portuguese',
                'date_format' => 'DD/MM/YYYY',
                'temperature' => 'Celsius',
                'driving_side' => 'right',
                'paper_size' => 'A4',
                'week_start' => 'Sunday',
                'decimal_separator' => ',',
                'thousands_separator' => '.',
            ),
            'ar' => array(
                'currency_symbol' => 'ARS',
                'currency_name' => 'Argentine Peso',
                'currency_format' => '$ X.XXX,XX',
                'measurement' => 'metric',
                'spelling' => 'Spanish',
                'date_format' => 'DD/MM/YYYY',
                'temperature' => 'Celsius',
                'driving_side' => 'right',
                'paper_size' => 'A4',
                'week_start' => 'Monday',
                'decimal_separator' => ',',
                'thousands_separator' => '.',
            ),
            // Europe
            'gb' => array(
                'currency_symbol' => '£',
                'currency_name' => 'British Pound Sterling',
                'currency_format' => '£X,XXX.XX',
                'measurement' => 'metric (with imperial for some uses)',
                'spelling' => 'British English',
                'date_format' => 'DD/MM/YYYY',
                'temperature' => 'Celsius',
                'driving_side' => 'left',
                'paper_size' => 'A4',
                'week_start' => 'Monday',
                'decimal_separator' => '.',
                'thousands_separator' => ',',
            ),
            'ie' => array(
                'currency_symbol' => '€',
                'currency_name' => 'Euro',
                'currency_format' => '€X,XXX.XX',
                'measurement' => 'metric',
                'spelling' => 'Irish English (British-based)',
                'date_format' => 'DD/MM/YYYY',
                'temperature' => 'Celsius',
                'driving_side' => 'left',
                'paper_size' => 'A4',
                'week_start' => 'Monday',
                'decimal_separator' => '.',
                'thousands_separator' => ',',
            ),
            'de' => array(
                'currency_symbol' => '€',
                'currency_name' => 'Euro',
                'currency_format' => 'X.XXX,XX € or €X,XXX.XX',
                'measurement' => 'metric',
                'spelling' => 'German (or British English)',
                'date_format' => 'DD.MM.YYYY',
                'temperature' => 'Celsius',
                'driving_side' => 'right',
                'paper_size' => 'A4',
                'week_start' => 'Monday',
                'decimal_separator' => ',',
                'thousands_separator' => '.',
            ),
            'fr' => array(
                'currency_symbol' => '€',
                'currency_name' => 'Euro',
                'currency_format' => 'X XXX,XX € or €X,XXX.XX',
                'measurement' => 'metric',
                'spelling' => 'French (or British English)',
                'date_format' => 'DD/MM/YYYY',
                'temperature' => 'Celsius',
                'driving_side' => 'right',
                'paper_size' => 'A4',
                'week_start' => 'Monday',
                'decimal_separator' => ',',
                'thousands_separator' => ' ',
            ),
            'es' => array(
                'currency_symbol' => '€',
                'currency_name' => 'Euro',
                'currency_format' => 'X.XXX,XX € or €X,XXX.XX',
                'measurement' => 'metric',
                'spelling' => 'Spanish (or British English)',
                'date_format' => 'DD/MM/YYYY',
                'temperature' => 'Celsius',
                'driving_side' => 'right',
                'paper_size' => 'A4',
                'week_start' => 'Monday',
                'decimal_separator' => ',',
                'thousands_separator' => '.',
            ),
            'it' => array(
                'currency_symbol' => '€',
                'currency_name' => 'Euro',
                'currency_format' => '€ X.XXX,XX or €X,XXX.XX',
                'measurement' => 'metric',
                'spelling' => 'Italian (or British English)',
                'date_format' => 'DD/MM/YYYY',
                'temperature' => 'Celsius',
                'driving_side' => 'right',
                'paper_size' => 'A4',
                'week_start' => 'Monday',
                'decimal_separator' => ',',
                'thousands_separator' => '.',
            ),
            'nl' => array(
                'currency_symbol' => '€',
                'currency_name' => 'Euro',
                'currency_format' => '€ X.XXX,XX or €X,XXX.XX',
                'measurement' => 'metric',
                'spelling' => 'Dutch (or British English)',
                'date_format' => 'DD-MM-YYYY',
                'temperature' => 'Celsius',
                'driving_side' => 'right',
                'paper_size' => 'A4',
                'week_start' => 'Monday',
                'decimal_separator' => ',',
                'thousands_separator' => '.',
            ),
            'ch' => array(
                'currency_symbol' => 'CHF',
                'currency_name' => 'Swiss Franc',
                'currency_format' => 'CHF X\'XXX.XX',
                'measurement' => 'metric',
                'spelling' => 'Swiss (German/French/Italian) or British English',
                'date_format' => 'DD.MM.YYYY',
                'temperature' => 'Celsius',
                'driving_side' => 'right',
                'paper_size' => 'A4',
                'week_start' => 'Monday',
                'decimal_separator' => '.',
                'thousands_separator' => '\'',
            ),
            'at' => array(
                'currency_symbol' => '€',
                'currency_name' => 'Euro',
                'currency_format' => '€ X.XXX,XX',
                'measurement' => 'metric',
                'spelling' => 'Austrian German (or British English)',
                'date_format' => 'DD.MM.YYYY',
                'temperature' => 'Celsius',
                'driving_side' => 'right',
                'paper_size' => 'A4',
                'week_start' => 'Monday',
                'decimal_separator' => ',',
                'thousands_separator' => '.',
            ),
            'be' => array(
                'currency_symbol' => '€',
                'currency_name' => 'Euro',
                'currency_format' => '€ X.XXX,XX',
                'measurement' => 'metric',
                'spelling' => 'Belgian (Dutch/French) or British English',
                'date_format' => 'DD/MM/YYYY',
                'temperature' => 'Celsius',
                'driving_side' => 'right',
                'paper_size' => 'A4',
                'week_start' => 'Monday',
                'decimal_separator' => ',',
                'thousands_separator' => '.',
            ),
            'pt' => array(
                'currency_symbol' => '€',
                'currency_name' => 'Euro',
                'currency_format' => '€ X.XXX,XX',
                'measurement' => 'metric',
                'spelling' => 'Portuguese (or British English)',
                'date_format' => 'DD/MM/YYYY',
                'temperature' => 'Celsius',
                'driving_side' => 'right',
                'paper_size' => 'A4',
                'week_start' => 'Monday',
                'decimal_separator' => ',',
                'thousands_separator' => ' ',
            ),
            'se' => array(
                'currency_symbol' => 'SEK',
                'currency_name' => 'Swedish Krona',
                'currency_format' => 'X XXX kr',
                'measurement' => 'metric',
                'spelling' => 'Swedish (or British English)',
                'date_format' => 'YYYY-MM-DD',
                'temperature' => 'Celsius',
                'driving_side' => 'right',
                'paper_size' => 'A4',
                'week_start' => 'Monday',
                'decimal_separator' => ',',
                'thousands_separator' => ' ',
            ),
            'no' => array(
                'currency_symbol' => 'NOK',
                'currency_name' => 'Norwegian Krone',
                'currency_format' => 'kr X XXX,XX',
                'measurement' => 'metric',
                'spelling' => 'Norwegian (or British English)',
                'date_format' => 'DD.MM.YYYY',
                'temperature' => 'Celsius',
                'driving_side' => 'right',
                'paper_size' => 'A4',
                'week_start' => 'Monday',
                'decimal_separator' => ',',
                'thousands_separator' => ' ',
            ),
            'dk' => array(
                'currency_symbol' => 'DKK',
                'currency_name' => 'Danish Krone',
                'currency_format' => 'kr X.XXX,XX',
                'measurement' => 'metric',
                'spelling' => 'Danish (or British English)',
                'date_format' => 'DD-MM-YYYY',
                'temperature' => 'Celsius',
                'driving_side' => 'right',
                'paper_size' => 'A4',
                'week_start' => 'Monday',
                'decimal_separator' => ',',
                'thousands_separator' => '.',
            ),
            'pl' => array(
                'currency_symbol' => 'PLN',
                'currency_name' => 'Polish Zloty',
                'currency_format' => 'X XXX,XX zł',
                'measurement' => 'metric',
                'spelling' => 'Polish (or British English)',
                'date_format' => 'DD.MM.YYYY',
                'temperature' => 'Celsius',
                'driving_side' => 'right',
                'paper_size' => 'A4',
                'week_start' => 'Monday',
                'decimal_separator' => ',',
                'thousands_separator' => ' ',
            ),
            'tr' => array(
                'currency_symbol' => '₺',
                'currency_name' => 'Turkish Lira',
                'currency_format' => '₺X.XXX,XX',
                'measurement' => 'metric',
                'spelling' => 'Turkish (or British English)',
                'date_format' => 'DD.MM.YYYY',
                'temperature' => 'Celsius',
                'driving_side' => 'right',
                'paper_size' => 'A4',
                'week_start' => 'Monday',
                'decimal_separator' => ',',
                'thousands_separator' => '.',
            ),
            'ru' => array(
                'currency_symbol' => '₽',
                'currency_name' => 'Russian Ruble',
                'currency_format' => 'X XXX ₽',
                'measurement' => 'metric',
                'spelling' => 'Russian (or British English)',
                'date_format' => 'DD.MM.YYYY',
                'temperature' => 'Celsius',
                'driving_side' => 'right',
                'paper_size' => 'A4',
                'week_start' => 'Monday',
                'decimal_separator' => ',',
                'thousands_separator' => ' ',
            ),
            // Asia-Pacific
            'au' => array(
                'currency_symbol' => 'A$',
                'currency_name' => 'Australian Dollar',
                'currency_format' => '$X,XXX.XX AUD',
                'measurement' => 'metric',
                'spelling' => 'Australian English (British-based)',
                'date_format' => 'DD/MM/YYYY',
                'temperature' => 'Celsius',
                'driving_side' => 'left',
                'paper_size' => 'A4',
                'week_start' => 'Monday',
                'decimal_separator' => '.',
                'thousands_separator' => ',',
            ),
            'nz' => array(
                'currency_symbol' => 'NZ$',
                'currency_name' => 'New Zealand Dollar',
                'currency_format' => '$X,XXX.XX NZD',
                'measurement' => 'metric',
                'spelling' => 'New Zealand English (British-based)',
                'date_format' => 'DD/MM/YYYY',
                'temperature' => 'Celsius',
                'driving_side' => 'left',
                'paper_size' => 'A4',
                'week_start' => 'Monday',
                'decimal_separator' => '.',
                'thousands_separator' => ',',
            ),
            'sg' => array(
                'currency_symbol' => 'S$',
                'currency_name' => 'Singapore Dollar',
                'currency_format' => '$X,XXX.XX SGD',
                'measurement' => 'metric',
                'spelling' => 'British English',
                'date_format' => 'DD/MM/YYYY',
                'temperature' => 'Celsius',
                'driving_side' => 'left',
                'paper_size' => 'A4',
                'week_start' => 'Monday',
                'decimal_separator' => '.',
                'thousands_separator' => ',',
            ),
            'hk' => array(
                'currency_symbol' => 'HK$',
                'currency_name' => 'Hong Kong Dollar',
                'currency_format' => 'HK$X,XXX.XX',
                'measurement' => 'metric',
                'spelling' => 'British English',
                'date_format' => 'DD/MM/YYYY',
                'temperature' => 'Celsius',
                'driving_side' => 'left',
                'paper_size' => 'A4',
                'week_start' => 'Monday',
                'decimal_separator' => '.',
                'thousands_separator' => ',',
            ),
            'jp' => array(
                'currency_symbol' => '¥',
                'currency_name' => 'Japanese Yen',
                'currency_format' => '¥X,XXX',
                'measurement' => 'metric',
                'spelling' => 'Japanese (or British English)',
                'date_format' => 'YYYY/MM/DD',
                'temperature' => 'Celsius',
                'driving_side' => 'left',
                'paper_size' => 'A4',
                'week_start' => 'Sunday',
                'decimal_separator' => '.',
                'thousands_separator' => ',',
            ),
            'cn' => array(
                'currency_symbol' => '¥',
                'currency_name' => 'Chinese Yuan',
                'currency_format' => '¥X,XXX.XX',
                'measurement' => 'metric',
                'spelling' => 'Chinese (or British English)',
                'date_format' => 'YYYY-MM-DD',
                'temperature' => 'Celsius',
                'driving_side' => 'right',
                'paper_size' => 'A4',
                'week_start' => 'Monday',
                'decimal_separator' => '.',
                'thousands_separator' => ',',
            ),
            'kr' => array(
                'currency_symbol' => '₩',
                'currency_name' => 'South Korean Won',
                'currency_format' => '₩X,XXX',
                'measurement' => 'metric',
                'spelling' => 'Korean (or British English)',
                'date_format' => 'YYYY.MM.DD',
                'temperature' => 'Celsius',
                'driving_side' => 'right',
                'paper_size' => 'A4',
                'week_start' => 'Sunday',
                'decimal_separator' => '.',
                'thousands_separator' => ',',
            ),
            'in' => array(
                'currency_symbol' => '₹',
                'currency_name' => 'Indian Rupee',
                'currency_format' => '₹X,XX,XXX.XX (lakhs/crores system)',
                'measurement' => 'metric',
                'spelling' => 'Indian English (British-based)',
                'date_format' => 'DD/MM/YYYY',
                'temperature' => 'Celsius',
                'driving_side' => 'left',
                'paper_size' => 'A4',
                'week_start' => 'Monday',
                'decimal_separator' => '.',
                'thousands_separator' => ',',
            ),
            'th' => array(
                'currency_symbol' => '฿',
                'currency_name' => 'Thai Baht',
                'currency_format' => '฿X,XXX.XX',
                'measurement' => 'metric',
                'spelling' => 'Thai (or British English)',
                'date_format' => 'DD/MM/YYYY',
                'temperature' => 'Celsius',
                'driving_side' => 'left',
                'paper_size' => 'A4',
                'week_start' => 'Sunday',
                'decimal_separator' => '.',
                'thousands_separator' => ',',
            ),
            'my' => array(
                'currency_symbol' => 'RM',
                'currency_name' => 'Malaysian Ringgit',
                'currency_format' => 'RM X,XXX.XX',
                'measurement' => 'metric',
                'spelling' => 'Malaysian English (British-based)',
                'date_format' => 'DD/MM/YYYY',
                'temperature' => 'Celsius',
                'driving_side' => 'left',
                'paper_size' => 'A4',
                'week_start' => 'Monday',
                'decimal_separator' => '.',
                'thousands_separator' => ',',
            ),
            'id' => array(
                'currency_symbol' => 'Rp',
                'currency_name' => 'Indonesian Rupiah',
                'currency_format' => 'Rp X.XXX.XXX',
                'measurement' => 'metric',
                'spelling' => 'Indonesian (or British English)',
                'date_format' => 'DD/MM/YYYY',
                'temperature' => 'Celsius',
                'driving_side' => 'left',
                'paper_size' => 'A4',
                'week_start' => 'Monday',
                'decimal_separator' => ',',
                'thousands_separator' => '.',
            ),
            'ph' => array(
                'currency_symbol' => '₱',
                'currency_name' => 'Philippine Peso',
                'currency_format' => '₱X,XXX.XX',
                'measurement' => 'metric',
                'spelling' => 'Philippine English (American-based)',
                'date_format' => 'MM/DD/YYYY',
                'temperature' => 'Celsius',
                'driving_side' => 'right',
                'paper_size' => 'Letter',
                'week_start' => 'Sunday',
                'decimal_separator' => '.',
                'thousands_separator' => ',',
            ),
            'vn' => array(
                'currency_symbol' => '₫',
                'currency_name' => 'Vietnamese Dong',
                'currency_format' => 'X.XXX.XXX ₫',
                'measurement' => 'metric',
                'spelling' => 'Vietnamese (or British English)',
                'date_format' => 'DD/MM/YYYY',
                'temperature' => 'Celsius',
                'driving_side' => 'right',
                'paper_size' => 'A4',
                'week_start' => 'Monday',
                'decimal_separator' => ',',
                'thousands_separator' => '.',
            ),
            // Africa
            'za' => array(
                'currency_symbol' => 'R',
                'currency_name' => 'South African Rand',
                'currency_format' => 'R X XXX,XX or R X,XXX.XX',
                'measurement' => 'metric',
                'spelling' => 'South African English (British-based)',
                'date_format' => 'YYYY/MM/DD or DD/MM/YYYY',
                'temperature' => 'Celsius',
                'driving_side' => 'left',
                'paper_size' => 'A4',
                'week_start' => 'Sunday',
                'decimal_separator' => ',',
                'thousands_separator' => ' ',
            ),
            'eg' => array(
                'currency_symbol' => 'E£',
                'currency_name' => 'Egyptian Pound',
                'currency_format' => 'E£ X,XXX.XX or X,XXX.XX EGP',
                'measurement' => 'metric',
                'spelling' => 'British English',
                'date_format' => 'DD/MM/YYYY',
                'temperature' => 'Celsius',
                'driving_side' => 'right',
                'paper_size' => 'A4',
                'week_start' => 'Sunday',
                'decimal_separator' => '.',
                'thousands_separator' => ',',
            ),
            'ng' => array(
                'currency_symbol' => '₦',
                'currency_name' => 'Nigerian Naira',
                'currency_format' => '₦X,XXX.XX',
                'measurement' => 'metric',
                'spelling' => 'Nigerian English (British-based)',
                'date_format' => 'DD/MM/YYYY',
                'temperature' => 'Celsius',
                'driving_side' => 'right',
                'paper_size' => 'A4',
                'week_start' => 'Monday',
                'decimal_separator' => '.',
                'thousands_separator' => ',',
            ),
            'ke' => array(
                'currency_symbol' => 'KSh',
                'currency_name' => 'Kenyan Shilling',
                'currency_format' => 'KSh X,XXX.XX',
                'measurement' => 'metric',
                'spelling' => 'Kenyan English (British-based)',
                'date_format' => 'DD/MM/YYYY',
                'temperature' => 'Celsius',
                'driving_side' => 'left',
                'paper_size' => 'A4',
                'week_start' => 'Monday',
                'decimal_separator' => '.',
                'thousands_separator' => ',',
            ),
            'ma' => array(
                'currency_symbol' => 'MAD',
                'currency_name' => 'Moroccan Dirham',
                'currency_format' => 'X.XXX,XX MAD',
                'measurement' => 'metric',
                'spelling' => 'French/Arabic (or British English)',
                'date_format' => 'DD/MM/YYYY',
                'temperature' => 'Celsius',
                'driving_side' => 'right',
                'paper_size' => 'A4',
                'week_start' => 'Monday',
                'decimal_separator' => ',',
                'thousands_separator' => '.',
            ),
            // Middle East additions
            'il' => array(
                'currency_symbol' => '₪',
                'currency_name' => 'Israeli Shekel',
                'currency_format' => '₪X,XXX.XX',
                'measurement' => 'metric',
                'spelling' => 'Hebrew (or British English)',
                'date_format' => 'DD/MM/YYYY',
                'temperature' => 'Celsius',
                'driving_side' => 'right',
                'paper_size' => 'A4',
                'week_start' => 'Sunday',
                'decimal_separator' => '.',
                'thousands_separator' => ',',
            ),
            'jo' => array(
                'currency_symbol' => 'JOD',
                'currency_name' => 'Jordanian Dinar',
                'currency_format' => 'JOD X.XXX',
                'measurement' => 'metric',
                'spelling' => 'British English',
                'date_format' => 'DD/MM/YYYY',
                'temperature' => 'Celsius',
                'driving_side' => 'right',
                'paper_size' => 'A4',
                'week_start' => 'Sunday',
                'decimal_separator' => '.',
                'thousands_separator' => ',',
            ),
            'lb' => array(
                'currency_symbol' => 'LBP',
                'currency_name' => 'Lebanese Pound',
                'currency_format' => 'LBP X,XXX',
                'measurement' => 'metric',
                'spelling' => 'British English',
                'date_format' => 'DD/MM/YYYY',
                'temperature' => 'Celsius',
                'driving_side' => 'right',
                'paper_size' => 'A4',
                'week_start' => 'Monday',
                'decimal_separator' => '.',
                'thousands_separator' => ',',
            ),
        );
        
        // Return localization for the country code, or default to British English/metric
        if ($country_code && isset($localization[$country_code])) {
            return $localization[$country_code];
        }
        
        // Default fallback
        return array(
            'currency_symbol' => '$',
            'currency_name' => 'US Dollar (default)',
            'currency_format' => '$X,XXX.XX',
            'measurement' => 'metric',
            'spelling' => 'British English',
            'date_format' => 'DD/MM/YYYY',
            'temperature' => 'Celsius',
            'driving_side' => 'right',
            'paper_size' => 'A4',
            'week_start' => 'Monday',
            'decimal_separator' => '.',
            'thousands_separator' => ',',
        );
    }
    
    /**
     * Get topics this persona should research
     */
    public function get_research_topics() {
        // Only use expertise areas for topic research
        // Interests and passions are used for writing style/voice, not topic discovery
        // This conserves API quota significantly
        return array_unique($this->expertise_areas);
    }
    
    /**
     * Get writing context (interests + passions) for AI prompts
     * These influence HOW we write, not WHAT we write about
     */
    public function get_writing_context() {
        return array(
            'interests' => $this->interests,
            'passions' => $this->passions
        );
    }
    
    // Getters
    public function get_id() { return $this->id; }
    public function get_name() { return $this->name; }
    public function get_slug() { return $this->slug; }
    public function get_wp_author_id() { return $this->wp_author_id; }
    public function get_categories() { return $this->categories ?: array(); }
    public function get_interests() { return $this->interests; }
    public function get_passions() { return $this->passions; }
    public function get_expertise_areas() { return $this->expertise_areas; }
    public function get_localities() { return $this->localities ?: array(); }
    public function get_voice_style() { return $this->voice_style; }
    public function get_tone() { return $this->tone; }
    public function get_knowledge_sources() { return $this->knowledge_sources; }
    public function get_writing_guidelines() { return $this->writing_guidelines; }
    public function get_avoid_topics() { return $this->avoid_topics; }
    public function get_personal_story() { return $this->personal_story; }
    public function get_target_audience() { return $this->target_audience; }
    public function get_language() { return $this->language; }
    public function is_active() { return $this->is_active; }
    public function get_total_articles() { return $this->total_articles; }
    public function get_total_spheres() { return $this->total_spheres; }
    
    /**
     * Check if persona has locality restrictions
     */
    public function has_localities() {
        return !empty($this->localities);
    }
    
    /**
     * Get primary locality (first one) for search geo-targeting
     */
    public function get_primary_locality() {
        if (!empty($this->localities)) {
            return $this->localities[0];
        }
        return null;
    }
    
    /**
     * Get locality names as readable string
     */
    public function get_localities_display() {
        if (empty($this->localities)) {
            return 'Global (No restriction)';
        }
        return implode(', ', array_map(function($loc) {
            // Extract display name from locality data
            if (is_array($loc)) {
                return isset($loc['name']) ? $loc['name'] : $loc['code'];
            }
            return $loc;
        }, $this->localities));
    }
    
    // Setters
    public function set_name($name) { $this->name = $name; }
    public function set_slug($slug) { $this->slug = $slug; }
    public function set_wp_author_id($id) { $this->wp_author_id = $id ? intval($id) : null; }
    public function set_categories($categories) { $this->categories = (array) $categories; }
    public function set_interests($interests) { $this->interests = (array) $interests; }
    public function set_passions($passions) { $this->passions = (array) $passions; }
    public function set_expertise_areas($areas) { $this->expertise_areas = (array) $areas; }
    public function set_localities($localities) { $this->localities = (array) $localities; }
    public function set_voice_style($style) { $this->voice_style = $style; }
    public function set_tone($tone) { $this->tone = $tone; }
    public function set_knowledge_sources($sources) { $this->knowledge_sources = (array) $sources; }
    public function set_writing_guidelines($guidelines) { $this->writing_guidelines = $guidelines; }
    public function set_avoid_topics($topics) { $this->avoid_topics = (array) $topics; }
    public function set_personal_story($story) { $this->personal_story = $story; }
    public function set_target_audience($audience) { $this->target_audience = $audience; }
    public function set_language($language) { $this->language = $language; }
    public function set_active($active) { $this->is_active = (bool) $active; }
}
