<?php
/**
 * Research Engine
 * Discovers trending topics, questions, and content gaps using REAL search data
 * Supports multiple search providers: Free (Google + DuckDuckGo), ValueSERP, SerpAPI
 */

if (!defined('ABSPATH')) {
    exit;
}

class AIAB_Research_Engine {
    
    private $persona;
    private $search_provider;
    private $serpapi_key;
    private $valueserp_key;
    private $google_cse_key;
    private $google_cse_cx;
    private $localities;
    private $primary_locality;
    private $google_cse_quota_exceeded = false;
    
    // Locality to Google geo parameters mapping
    private static $locality_map = array(
        // Middle East
        'ae' => array('gl' => 'ae', 'hl' => 'en', 'cr' => 'countryAE', 'name' => 'UAE'),
        'ae-dubai' => array('gl' => 'ae', 'hl' => 'en', 'cr' => 'countryAE', 'name' => 'Dubai, UAE', 'near' => 'Dubai'),
        'ae-abudhabi' => array('gl' => 'ae', 'hl' => 'en', 'cr' => 'countryAE', 'name' => 'Abu Dhabi, UAE', 'near' => 'Abu Dhabi'),
        'ae-sharjah' => array('gl' => 'ae', 'hl' => 'en', 'cr' => 'countryAE', 'name' => 'Sharjah, UAE', 'near' => 'Sharjah'),
        'ae-ajman' => array('gl' => 'ae', 'hl' => 'en', 'cr' => 'countryAE', 'name' => 'Ajman, UAE', 'near' => 'Ajman'),
        'sa' => array('gl' => 'sa', 'hl' => 'ar', 'cr' => 'countrySA', 'name' => 'Saudi Arabia'),
        'sa-riyadh' => array('gl' => 'sa', 'hl' => 'ar', 'cr' => 'countrySA', 'name' => 'Riyadh, KSA', 'near' => 'Riyadh'),
        'sa-jeddah' => array('gl' => 'sa', 'hl' => 'ar', 'cr' => 'countrySA', 'name' => 'Jeddah, KSA', 'near' => 'Jeddah'),
        'qa' => array('gl' => 'qa', 'hl' => 'en', 'cr' => 'countryQA', 'name' => 'Qatar'),
        'qa-doha' => array('gl' => 'qa', 'hl' => 'en', 'cr' => 'countryQA', 'name' => 'Doha, Qatar', 'near' => 'Doha'),
        'kw' => array('gl' => 'kw', 'hl' => 'ar', 'cr' => 'countryKW', 'name' => 'Kuwait'),
        'bh' => array('gl' => 'bh', 'hl' => 'ar', 'cr' => 'countryBH', 'name' => 'Bahrain'),
        'om' => array('gl' => 'om', 'hl' => 'ar', 'cr' => 'countryOM', 'name' => 'Oman'),
        // North America
        'us' => array('gl' => 'us', 'hl' => 'en', 'cr' => 'countryUS', 'name' => 'United States'),
        'us-newyork' => array('gl' => 'us', 'hl' => 'en', 'cr' => 'countryUS', 'name' => 'New York, USA', 'near' => 'New York'),
        'us-california' => array('gl' => 'us', 'hl' => 'en', 'cr' => 'countryUS', 'name' => 'California, USA', 'near' => 'California'),
        'us-texas' => array('gl' => 'us', 'hl' => 'en', 'cr' => 'countryUS', 'name' => 'Texas, USA', 'near' => 'Texas'),
        'us-florida' => array('gl' => 'us', 'hl' => 'en', 'cr' => 'countryUS', 'name' => 'Florida, USA', 'near' => 'Florida'),
        'ca' => array('gl' => 'ca', 'hl' => 'en', 'cr' => 'countryCA', 'name' => 'Canada'),
        'ca-ontario' => array('gl' => 'ca', 'hl' => 'en', 'cr' => 'countryCA', 'name' => 'Ontario, Canada', 'near' => 'Ontario'),
        'mx' => array('gl' => 'mx', 'hl' => 'es', 'cr' => 'countryMX', 'name' => 'Mexico'),
        // Europe
        'gb' => array('gl' => 'uk', 'hl' => 'en', 'cr' => 'countryUK', 'name' => 'United Kingdom'),
        'gb-london' => array('gl' => 'uk', 'hl' => 'en', 'cr' => 'countryUK', 'name' => 'London, UK', 'near' => 'London'),
        'de' => array('gl' => 'de', 'hl' => 'de', 'cr' => 'countryDE', 'name' => 'Germany'),
        'fr' => array('gl' => 'fr', 'hl' => 'fr', 'cr' => 'countryFR', 'name' => 'France'),
        'es' => array('gl' => 'es', 'hl' => 'es', 'cr' => 'countryES', 'name' => 'Spain'),
        'it' => array('gl' => 'it', 'hl' => 'it', 'cr' => 'countryIT', 'name' => 'Italy'),
        'nl' => array('gl' => 'nl', 'hl' => 'nl', 'cr' => 'countryNL', 'name' => 'Netherlands'),
        'ch' => array('gl' => 'ch', 'hl' => 'de', 'cr' => 'countryCH', 'name' => 'Switzerland'),
        // Asia Pacific
        'au' => array('gl' => 'au', 'hl' => 'en', 'cr' => 'countryAU', 'name' => 'Australia'),
        'au-sydney' => array('gl' => 'au', 'hl' => 'en', 'cr' => 'countryAU', 'name' => 'Sydney, Australia', 'near' => 'Sydney'),
        'au-melbourne' => array('gl' => 'au', 'hl' => 'en', 'cr' => 'countryAU', 'name' => 'Melbourne, Australia', 'near' => 'Melbourne'),
        'sg' => array('gl' => 'sg', 'hl' => 'en', 'cr' => 'countrySG', 'name' => 'Singapore'),
        'hk' => array('gl' => 'hk', 'hl' => 'en', 'cr' => 'countryHK', 'name' => 'Hong Kong'),
        'jp' => array('gl' => 'jp', 'hl' => 'ja', 'cr' => 'countryJP', 'name' => 'Japan'),
        'in' => array('gl' => 'in', 'hl' => 'en', 'cr' => 'countryIN', 'name' => 'India'),
        'in-mumbai' => array('gl' => 'in', 'hl' => 'en', 'cr' => 'countryIN', 'name' => 'Mumbai, India', 'near' => 'Mumbai'),
        'in-delhi' => array('gl' => 'in', 'hl' => 'en', 'cr' => 'countryIN', 'name' => 'Delhi, India', 'near' => 'Delhi'),
        'ph' => array('gl' => 'ph', 'hl' => 'en', 'cr' => 'countryPH', 'name' => 'Philippines'),
        'my' => array('gl' => 'my', 'hl' => 'en', 'cr' => 'countryMY', 'name' => 'Malaysia'),
        // Africa
        'za' => array('gl' => 'za', 'hl' => 'en', 'cr' => 'countryZA', 'name' => 'South Africa'),
        'eg' => array('gl' => 'eg', 'hl' => 'ar', 'cr' => 'countryEG', 'name' => 'Egypt'),
        'ng' => array('gl' => 'ng', 'hl' => 'en', 'cr' => 'countryNG', 'name' => 'Nigeria'),
        'ke' => array('gl' => 'ke', 'hl' => 'en', 'cr' => 'countryKE', 'name' => 'Kenya'),
    );
    
    public function __construct(AIAB_Persona $persona) {
        $this->persona = $persona;
        $this->search_provider = get_option('aiab_search_provider', 'free');
        $this->serpapi_key = get_option('aiab_serpapi_key', '');
        $this->valueserp_key = get_option('aiab_valueserp_key', '');
        $this->google_cse_key = get_option('aiab_google_cse_key', '');
        $this->google_cse_cx = get_option('aiab_google_cse_cx', '');
        
        // Check if Google CSE quota was recently exceeded
        if (get_transient('aiab_google_cse_quota_exceeded')) {
            $this->google_cse_quota_exceeded = true;
            AIAB_Logger::debug("Google CSE quota exceeded (from cache) - using fallback provider");
        }
        
        // Load localities from persona
        $this->localities = $persona->get_localities();
        $this->primary_locality = !empty($this->localities) ? $this->localities[0] : null;
        
        AIAB_Logger::info("Research Engine initialized with provider: " . $this->search_provider);
        
        if ($this->primary_locality) {
            $loc_name = $this->get_locality_name($this->primary_locality);
            AIAB_Logger::info("🌍 Geo-targeting enabled: " . $loc_name);
        }
    }
    
    /**
     * Get Google geo parameters for a locality code
     */
    private function get_geo_params($locality_code = null) {
        $code = $locality_code ?: $this->primary_locality;
        
        if (!$code || !isset(self::$locality_map[$code])) {
            return array(); // No geo targeting
        }
        
        return self::$locality_map[$code];
    }
    
    /**
     * Get locality display name
     */
    private function get_locality_name($locality_code) {
        if (isset(self::$locality_map[$locality_code])) {
            return self::$locality_map[$locality_code]['name'];
        }
        return $locality_code;
    }
    
    /**
     * Get all locality names for display
     */
    public function get_localities_display() {
        if (empty($this->localities)) {
            return 'Global';
        }
        
        $names = array();
        foreach ($this->localities as $code) {
            $names[] = $this->get_locality_name($code);
        }
        return implode(', ', $names);
    }
    
    /**
     * Check if geo-targeting is enabled
     */
    public function has_geo_targeting() {
        return !empty($this->primary_locality);
    }
    
    /**
     * Get location suffix for search queries (e.g., "in Dubai")
     */
    private function get_location_suffix() {
        if (!$this->primary_locality) {
            return '';
        }
        
        $geo = $this->get_geo_params();
        if (isset($geo['near'])) {
            return ' in ' . $geo['near'];
        }
        if (isset($geo['name'])) {
            return ' in ' . $geo['name'];
        }
        return '';
    }
    
    /**
     * Main research method - finds the best topic to write about
     * Prioritizes REAL questions people are searching for
     */
    public function discover_pillar_topic() {
        AIAB_Logger::log("🔍 Starting topic discovery for persona: " . $this->persona->get_name());
        
        // Log locality targeting
        if ($this->has_geo_targeting()) {
            AIAB_Logger::info("🌍 GEO-TARGETING ENABLED: " . $this->get_localities_display());
            AIAB_Logger::info("   Searches will be localized to find region-specific topics");
        } else {
            AIAB_Logger::info("🌐 Global search (no locality restrictions)");
        }
        
        $all_topics = array();
        $research_areas = $this->persona->get_research_topics();
        
        AIAB_Logger::info("Research areas: " . implode(', ', $research_areas));
        AIAB_Logger::info("Using search provider: " . $this->search_provider);
        
        foreach ($research_areas as $area) {
            AIAB_Logger::debug("Researching area: $area");
            
            // Use the configured search provider
            switch ($this->search_provider) {
                case 'serpapi':
                    if (!empty($this->serpapi_key)) {
                        $topics = $this->research_with_serpapi($area);
                    } else {
                        AIAB_Logger::warning("SerpAPI key not configured, falling back to free");
                        $topics = $this->research_with_free($area);
                    }
                    break;
                    
                case 'valueserp':
                    if (!empty($this->valueserp_key)) {
                        $topics = $this->research_with_valueserp($area);
                    } else {
                        AIAB_Logger::warning("ValueSERP key not configured, falling back to free");
                        $topics = $this->research_with_free($area);
                    }
                    break;
                
                case 'google_cse':
                    if (!empty($this->google_cse_key) && !empty($this->google_cse_cx)) {
                        $topics = $this->research_with_google_cse($area);
                    } else {
                        AIAB_Logger::warning("Google CSE not configured, falling back to free");
                        $topics = $this->research_with_free($area);
                    }
                    break;
                    
                case 'free':
                default:
                    $topics = $this->research_with_free($area);
                    break;
            }
            
            $all_topics = array_merge($all_topics, $topics);
        }
        
        AIAB_Logger::info("Total raw topics discovered: " . count($all_topics));
        
        // Sanitize all topics (remove Reddit references, clean up formatting)
        $all_topics = $this->sanitize_all_topics($all_topics);
        
        // Filter out already-written topics
        $fresh_topics = $this->filter_written_topics($all_topics);
        AIAB_Logger::info("Fresh topics after filtering: " . count($fresh_topics));
        
        // Score and rank topics
        $scored_topics = $this->score_topics($fresh_topics);
        
        // 🧠 PERSONA FILTER: Only keep topics that match the persona's niche
        // This is the "Would this person write about this?" test
        $scored_topics = $this->filter_by_niche_relevance($scored_topics);
        AIAB_Logger::info("Topics after persona relevance filter: " . count($scored_topics));
        
        // 🌍 LOCATION FILTER: Remove topics with mismatched locations
        // If persona targets Dubai, filter out "Berkeley County", "Texas", etc.
        $scored_topics = $this->filter_by_location_relevance($scored_topics);
        AIAB_Logger::info("Topics after location filter: " . count($scored_topics));
        
        if (empty($scored_topics)) {
            AIAB_Logger::warning("No topics found from research - falling back to AI generation");
            return $this->generate_topic_from_ai();
        }
        
        usort($scored_topics, function($a, $b) {
            return $b['score'] - $a['score'];
        });
        
        // Log top 5 candidates
        AIAB_Logger::info("🎯 Top 5 topic candidates (filtered through persona):");
        foreach (array_slice($scored_topics, 0, 5) as $i => $t) {
            AIAB_Logger::info("  " . ($i+1) . ". [{$t['score']}] {$t['keyword']} (source: {$t['source']})");
        }
        
        $best_topic = $scored_topics[0];
        
        AIAB_Logger::log("✅ SELECTED PILLAR TOPIC: " . $best_topic['keyword'], 'info', array(
            'score' => $best_topic['score'],
            'source' => $best_topic['source'],
            'search_volume' => $best_topic['search_volume'] ?? 'unknown'
        ));
        
        return $best_topic;
    }
    
    // ========================================
    // FREE PROVIDER (Super-powered free research!)
    // ========================================
    
    /**
     * Research using completely FREE methods
     * Now with PAA scraping, AnswerThePublic-style, Wikipedia, Reddit, and Google Trends!
     */
    private function research_with_free($area) {
        $topics = array();
        
        AIAB_Logger::info("🆓 Using SUPER-POWERED FREE search provider for: $area");
        
        // Source 1: Enhanced Google Autocomplete
        $autocomplete = $this->get_google_autocomplete_enhanced($area);
        AIAB_Logger::info("✅ Google Autocomplete: " . count($autocomplete) . " suggestions");
        $topics = array_merge($topics, $autocomplete);
        
        // Check for timeout
        if (class_exists('AIAB_Orchestrator') && AIAB_Orchestrator::should_abort()) {
            AIAB_Logger::warning("⏱️ Timeout approaching - returning early with " . count($topics) . " topics");
            return $topics;
        }
        
        // Source 2: DuckDuckGo Instant Answers
        $ddg = $this->get_duckduckgo_topics($area);
        AIAB_Logger::info("✅ DuckDuckGo: " . count($ddg) . " topics");
        $topics = array_merge($topics, $ddg);
        
        // Source 3: Alphabet Soup (a-z variations)
        $alphabet = $this->get_alphabet_soup($area);
        AIAB_Logger::info("✅ Alphabet Soup: " . count($alphabet) . " variations");
        $topics = array_merge($topics, $alphabet);
        
        // Check for timeout
        if (class_exists('AIAB_Orchestrator') && AIAB_Orchestrator::should_abort()) {
            AIAB_Logger::warning("⏱️ Timeout approaching - returning early with " . count($topics) . " topics");
            return $topics;
        }
        
        // Source 4: Question modifiers (who, what, when, where, why, how)
        $questions = $this->get_question_variations($area);
        AIAB_Logger::info("✅ Question Variations: " . count($questions) . " questions");
        $topics = array_merge($topics, $questions);
        
        // Source 5: 🆕 Google "People Also Ask" Scraper (THE KILLER FEATURE!)
        $paa = $this->scrape_google_paa($area);
        AIAB_Logger::info("✅ Google PAA Scraper: " . count($paa) . " questions");
        $topics = array_merge($topics, $paa);
        
        // Check for timeout
        if (class_exists('AIAB_Orchestrator') && AIAB_Orchestrator::should_abort()) {
            AIAB_Logger::warning("⏱️ Timeout approaching - returning early with " . count($topics) . " topics");
            return $topics;
        }
        
        // Source 6: 🆕 AnswerThePublic-style comprehensive questions
        $atp = $this->get_answer_the_public_style($area);
        AIAB_Logger::info("✅ AnswerThePublic-style: " . count($atp) . " questions");
        $topics = array_merge($topics, $atp);
        
        // Source 7: 🆕 Wikipedia related topics
        $wiki = $this->get_wikipedia_topics($area);
        AIAB_Logger::info("✅ Wikipedia: " . count($wiki) . " related topics");
        $topics = array_merge($topics, $wiki);
        
        // Check for timeout
        if (class_exists('AIAB_Orchestrator') && AIAB_Orchestrator::should_abort()) {
            AIAB_Logger::warning("⏱️ Timeout approaching - returning early with " . count($topics) . " topics");
            return $topics;
        }
        
        // Source 8: 🆕 Reddit topic mining
        $reddit = $this->get_reddit_topics($area);
        AIAB_Logger::info("✅ Reddit: " . count($reddit) . " discussion topics");
        $topics = array_merge($topics, $reddit);
        
        // Source 9: 🆕 Google Trends
        $trends = $this->get_google_trends($area);
        AIAB_Logger::info("✅ Google Trends: " . count($trends) . " trending topics");
        $topics = array_merge($topics, $trends);
        
        AIAB_Logger::info("🎯 TOTAL FREE TOPICS DISCOVERED: " . count($topics));
        
        return $topics;
    }
    
    /**
     * Enhanced Google Autocomplete with multiple prefixes/suffixes
     */
    private function get_google_autocomplete_enhanced($query) {
        $cached = AIAB_Database::get_research_cache($query, 'autocomplete_enhanced');
        if ($cached) {
            return $cached;
        }
        
        // Check if we're currently rate-limited (global flag)
        $rate_limited_until = get_transient('aiab_google_rate_limited');
        if ($rate_limited_until) {
            AIAB_Logger::debug("Skipping Google Autocomplete - rate limited until " . date('H:i:s', $rate_limited_until));
            return array();
        }
        
        $topics = array();
        $consecutive_failures = 0;
        $max_failures = 3; // Stop after 3 consecutive 403s
        
        // Various modifiers to discover different intents
        $modifiers = array(
            '',                           // Base query
            'how to ',                    // How-to intent
            'what is ',                   // Informational
            'why ',                       // Explanation
            'best ',                      // Listicle
            'top ',                       // Ranking
            $query . ' vs ',              // Comparison
            $query . ' or ',              // Comparison
            $query . ' cost ',            // Price intent
            $query . ' price ',           // Price intent
            $query . ' problems ',        // Problem-solution
            $query . ' tips ',            // Advice
            $query . ' benefits ',        // Benefit-focused
            $query . ' for beginners ',   // Beginner content
            $query . ' near me ',         // Local SEO
            $query . ' in dubai ',        // Local (customizable)
            'is ' . $query . ' ',         // Questions
            'can ' . $query . ' ',        // Questions
            'does ' . $query . ' ',       // Questions
            'should i ' . $query . ' ',   // Decision
        );
        
        foreach ($modifiers as $modifier) {
            // FAIL FAST: Stop if we've hit too many consecutive failures
            if ($consecutive_failures >= $max_failures) {
                AIAB_Logger::warning("Google rate-limited - stopping autocomplete after {$consecutive_failures} consecutive 403s");
                // Set global rate limit flag for 10 minutes
                set_transient('aiab_google_rate_limited', time() + 600, 600);
                break;
            }
            
            $search_query = trim($modifier . ($modifier ? '' : $query));
            if (empty($search_query)) $search_query = $query;
            
            $result = $this->fetch_google_autocomplete($search_query);
            
            // Track failures
            if ($result === false) {
                $consecutive_failures++;
                continue; // Skip to next modifier
            }
            
            // Reset failure count on success
            $consecutive_failures = 0;
            
            foreach ($result as $suggestion) {
                $topics[] = array(
                    'keyword' => $suggestion,
                    'source' => 'google_autocomplete',
                    'base_query' => $query,
                    'search_volume' => 'medium'
                );
            }
            
            // Rate limiting to avoid blocks
            usleep(300000); // 300ms delay (increased from 200)
        }
        
        // Remove duplicates
        $topics = $this->deduplicate_topics($topics);
        
        // Only cache if we got results
        if (!empty($topics)) {
            AIAB_Database::set_research_cache($query, 'autocomplete_enhanced', $topics, 6);
        }
        
        return $topics;
    }
    
    /**
     * Fetch Google Autocomplete suggestions
     * Returns array of suggestions on success, FALSE on rate-limit/error
     */
    private function fetch_google_autocomplete($query) {
        $url = 'https://suggestqueries.google.com/complete/search?' . http_build_query(array(
            'client' => 'firefox',
            'q' => $query
        ));
        
        $response = wp_remote_get($url, array(
            'timeout' => 5, // Reduced timeout
            'headers' => array(
                'User-Agent' => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36',
                'Accept' => 'application/json, text/javascript, */*; q=0.01',
                'Accept-Language' => 'en-US,en;q=0.9',
            )
        ));
        
        if (is_wp_error($response)) {
            AIAB_Logger::debug("Autocomplete error for '$query': " . $response->get_error_message());
            return false; // Return FALSE to indicate failure
        }
        
        $status_code = wp_remote_retrieve_response_code($response);
        if ($status_code === 403 || $status_code === 429) {
            AIAB_Logger::debug("Google Autocomplete HTTP {$status_code} - rate limited");
            return false; // Return FALSE to trigger fail-fast
        }
        
        if ($status_code !== 200) {
            AIAB_Logger::debug("Google Autocomplete HTTP {$status_code}");
            return false;
        }
        
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);
        
        if (is_array($data) && isset($data[1]) && is_array($data[1])) {
            return $data[1];
        }
        
        return array(); // Empty array = success but no results
    }
    
    /**
     * DuckDuckGo Instant Answers API (completely free, no API key)
     */
    private function get_duckduckgo_topics($query) {
        $cached = AIAB_Database::get_research_cache($query, 'duckduckgo');
        if ($cached) {
            return $cached;
        }
        
        $topics = array();
        
        $url = 'https://api.duckduckgo.com/?' . http_build_query(array(
            'q' => $query,
            'format' => 'json',
            'no_html' => 1,
            'skip_disambig' => 1
        ));
        
        $response = wp_remote_get($url, array(
            'timeout' => 15,
            'headers' => array(
                'User-Agent' => 'Mozilla/5.0 (compatible; WordPressBot/1.0)'
            )
        ));
        
        if (is_wp_error($response)) {
            AIAB_Logger::debug("DuckDuckGo error: " . $response->get_error_message());
            return $topics;
        }
        
        $data = json_decode(wp_remote_retrieve_body($response), true);
        
        // Extract related topics
        if (!empty($data['RelatedTopics'])) {
            foreach ($data['RelatedTopics'] as $topic) {
                if (isset($topic['Text'])) {
                    // Extract the first sentence as a potential topic
                    $text = $topic['Text'];
                    $first_sentence = strtok($text, '.');
                    
                    if (strlen($first_sentence) > 10 && strlen($first_sentence) < 100) {
                        $topics[] = array(
                            'keyword' => $first_sentence,
                            'source' => 'duckduckgo_related',
                            'snippet' => $text,
                            'base_query' => $query,
                            'search_volume' => 'medium'
                        );
                    }
                }
                
                // Handle nested topics (categories)
                if (isset($topic['Topics']) && is_array($topic['Topics'])) {
                    foreach ($topic['Topics'] as $subtopic) {
                        if (isset($subtopic['Text'])) {
                            $text = $subtopic['Text'];
                            $first_sentence = strtok($text, '.');
                            
                            if (strlen($first_sentence) > 10 && strlen($first_sentence) < 100) {
                                $topics[] = array(
                                    'keyword' => $first_sentence,
                                    'source' => 'duckduckgo_subtopic',
                                    'snippet' => $text,
                                    'base_query' => $query,
                                    'search_volume' => 'low'
                                );
                            }
                        }
                    }
                }
            }
        }
        
        // Extract from abstract
        if (!empty($data['AbstractText']) && strlen($data['AbstractText']) > 20) {
            $topics[] = array(
                'keyword' => 'What is ' . $query,
                'source' => 'duckduckgo_abstract',
                'snippet' => $data['AbstractText'],
                'base_query' => $query,
                'search_volume' => 'high'
            );
        }
        
        AIAB_Database::set_research_cache($query, 'duckduckgo', $topics, 12);
        
        return $topics;
    }
    
    /**
     * Alphabet Soup technique - A to Z variations
     */
    private function get_alphabet_soup($query) {
        $cached = AIAB_Database::get_research_cache($query, 'alphabet_soup');
        if ($cached) {
            return $cached;
        }
        
        // Check if rate limited
        if (get_transient('aiab_google_rate_limited')) {
            return array();
        }
        
        $topics = array();
        $letters = range('a', 'z');
        $consecutive_failures = 0;
        
        // Only sample a few letters to avoid rate limiting
        $sample_letters = array_rand(array_flip($letters), 8);
        
        foreach ($sample_letters as $letter) {
            // Stop on rate limiting
            if ($consecutive_failures >= 3) {
                break;
            }
            
            // Query + letter prefix
            $suggestions = $this->fetch_google_autocomplete($query . ' ' . $letter);
            
            if ($suggestions === false) {
                $consecutive_failures++;
                continue;
            }
            $consecutive_failures = 0;
            
            foreach ($suggestions as $suggestion) {
                $topics[] = array(
                    'keyword' => $suggestion,
                    'source' => 'alphabet_soup',
                    'base_query' => $query,
                    'search_volume' => 'medium'
                );
            }
            
            usleep(300000); // 300ms delay
        }
        
        if (!empty($topics)) {
            AIAB_Database::set_research_cache($query, 'alphabet_soup', $topics, 12);
        }
        
        return $topics;
    }
    
    /**
     * Question variations - who, what, when, where, why, how
     */
    private function get_question_variations($query) {
        $cached = AIAB_Database::get_research_cache($query, 'questions');
        if ($cached) {
            return $cached;
        }
        
        // Check if rate limited
        if (get_transient('aiab_google_rate_limited')) {
            return array();
        }
        
        $topics = array();
        $consecutive_failures = 0;
        
        $question_words = array(
            'what', 'why', 'how', 'when', 'where', 'which', 
            'who', 'can', 'does', 'is', 'are', 'should'
        );
        
        foreach ($question_words as $qword) {
            // Stop on rate limiting
            if ($consecutive_failures >= 3) {
                break;
            }
            
            $suggestions = $this->fetch_google_autocomplete($qword . ' ' . $query);
            
            if ($suggestions === false) {
                $consecutive_failures++;
                continue;
            }
            $consecutive_failures = 0;
            
            foreach ($suggestions as $suggestion) {
                // Check if it ends with a question mark or is a question structure
                $topics[] = array(
                    'keyword' => $suggestion,
                    'source' => 'question_variation',
                    'base_query' => $query,
                    'search_volume' => 'high' // Questions are high value
                );
            }
            
            usleep(300000); // 300ms delay
        }
        
        if (!empty($topics)) {
            AIAB_Database::set_research_cache($query, 'questions', $topics, 6);
        }
        
        return $topics;
    }
    
    // ========================================
    // 🆕 GOOGLE PEOPLE ALSO ASK SCRAPER
    // ========================================
    
    /**
     * Scrape Google's "People Also Ask" boxes directly
     * This is the KILLER feature that makes free nearly as good as paid!
     */
    private function scrape_google_paa($query) {
        $cached = AIAB_Database::get_research_cache($query, 'google_paa_scrape');
        if ($cached) {
            return $cached;
        }
        
        $topics = array();
        
        // Google search URL
        $url = 'https://www.google.com/search?' . http_build_query(array(
            'q' => $query,
            'hl' => 'en',
            'gl' => 'us'
        ));
        
        $response = wp_remote_get($url, array(
            'timeout' => 15,
            'headers' => array(
                'User-Agent' => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36',
                'Accept' => 'text/html,application/xhtml+xml,application/xml;q=0.9,image/webp,*/*;q=0.8',
                'Accept-Language' => 'en-US,en;q=0.5',
                'Accept-Encoding' => 'gzip, deflate',
                'Connection' => 'keep-alive',
                'Upgrade-Insecure-Requests' => '1',
            )
        ));
        
        if (is_wp_error($response)) {
            AIAB_Logger::debug("Google PAA scrape error: " . $response->get_error_message());
            return $topics;
        }
        
        $html = wp_remote_retrieve_body($response);
        
        // Method 1: Look for data-q attribute (PAA questions)
        if (preg_match_all('/data-q="([^"]+)"/i', $html, $matches)) {
            foreach ($matches[1] as $question) {
                $question = html_entity_decode($question, ENT_QUOTES, 'UTF-8');
                if (strlen($question) > 10 && strlen($question) < 200) {
                    $topics[] = array(
                        'keyword' => $question,
                        'source' => 'google_paa',
                        'base_query' => $query,
                        'search_volume' => 'high' // PAA = proven search demand
                    );
                }
            }
        }
        
        // Method 2: Look for aria-label patterns with questions
        if (preg_match_all('/aria-label="([^"]*\?[^"]*)"/i', $html, $matches)) {
            foreach ($matches[1] as $question) {
                $question = html_entity_decode($question, ENT_QUOTES, 'UTF-8');
                $question = trim($question);
                // Filter out navigation elements
                if (strlen($question) > 15 && strlen($question) < 150 && !preg_match('/^(Search|Close|More|Menu)/i', $question)) {
                    $topics[] = array(
                        'keyword' => $question,
                        'source' => 'google_paa',
                        'base_query' => $query,
                        'search_volume' => 'high'
                    );
                }
            }
        }
        
        // Method 3: Look for jsname patterns (newer Google format)
        if (preg_match_all('/<div[^>]*jsname="[^"]*"[^>]*>([^<]*\?)<\/div>/i', $html, $matches)) {
            foreach ($matches[1] as $question) {
                $question = html_entity_decode($question, ENT_QUOTES, 'UTF-8');
                $question = trim($question);
                if (strlen($question) > 15 && strlen($question) < 150) {
                    $topics[] = array(
                        'keyword' => $question,
                        'source' => 'google_paa',
                        'base_query' => $query,
                        'search_volume' => 'high'
                    );
                }
            }
        }
        
        // Method 4: Extract "Related searches" from Google
        if (preg_match_all('/<a[^>]*>((?:How|What|Why|When|Where|Which|Who|Can|Does|Is|Are|Should)[^<]{10,80})<\/a>/i', $html, $matches)) {
            foreach ($matches[1] as $related) {
                $related = html_entity_decode(strip_tags($related), ENT_QUOTES, 'UTF-8');
                $related = trim($related);
                if (strlen($related) > 10) {
                    $topics[] = array(
                        'keyword' => $related,
                        'source' => 'google_related',
                        'base_query' => $query,
                        'search_volume' => 'medium'
                    );
                }
            }
        }
        
        // Deduplicate
        $topics = $this->deduplicate_topics($topics);
        
        AIAB_Logger::debug("Scraped " . count($topics) . " PAA questions for: $query");
        
        // Cache for 4 hours (shorter since it's scraped data)
        AIAB_Database::set_research_cache($query, 'google_paa_scrape', $topics, 4);
        
        // Rate limit - be respectful
        usleep(500000); // 500ms delay
        
        return $topics;
    }
    
    // ========================================
    // 🆕 ANSWERTHEPUBLIC-STYLE QUESTIONS
    // ========================================
    
    /**
     * Generate comprehensive questions like AnswerThePublic
     * Creates a matrix of question patterns for any topic
     */
    private function get_answer_the_public_style($query) {
        $cached = AIAB_Database::get_research_cache($query, 'atp_style');
        if ($cached) {
            return $cached;
        }
        
        // Check if rate limited - skip entirely if Google is blocking us
        if (get_transient('aiab_google_rate_limited')) {
            AIAB_Logger::debug("Skipping AnswerThePublic-style - Google rate limited");
            return array();
        }
        
        $topics = array();
        $consecutive_failures = 0;
        $max_failures = 3;
        
        // Question words with specific patterns - REDUCED set to avoid rate limiting
        $question_patterns = array(
            // Core HOW questions
            'how to ' . $query,
            'how does ' . $query . ' work',
            'how much does ' . $query . ' cost',
            'how to fix ' . $query,
            'how to prevent ' . $query,
            
            // Core WHAT questions
            'what is ' . $query,
            'what causes ' . $query,
            
            // Core WHY questions  
            'why is ' . $query . ' important',
            
            // Core IS questions
            'is ' . $query . ' dangerous',
            'is ' . $query . ' worth it',
            
            // Comparison patterns
            'best ' . $query,
            $query . ' vs ',
            $query . ' alternatives',
            $query . ' pros and cons',
            
            // Problem/Solution patterns
            $query . ' problems',
            $query . ' tips',
            $query . ' benefits',
        );
        
        // Get autocomplete for each pattern
        foreach ($question_patterns as $pattern) {
            // Stop if rate limited
            if ($consecutive_failures >= $max_failures) {
                AIAB_Logger::warning("ATP stopping early - rate limited after {$consecutive_failures} failures");
                break;
            }
            
            $suggestions = $this->fetch_google_autocomplete($pattern);
            
            if ($suggestions === false) {
                $consecutive_failures++;
                continue;
            }
            $consecutive_failures = 0;
            
            foreach ($suggestions as $suggestion) {
                $topics[] = array(
                    'keyword' => $suggestion,
                    'source' => 'answer_the_public',
                    'pattern' => $pattern,
                    'base_query' => $query,
                    'search_volume' => 'medium'
                );
            }
            
            // Rate limiting
            usleep(300000); // 300ms delay
        }
        
        // Deduplicate
        $topics = $this->deduplicate_topics($topics);
        
        if (!empty($topics)) {
            AIAB_Database::set_research_cache($query, 'atp_style', $topics, 12);
        }
        
        return $topics;
    }
    
    // ========================================
    // 🆕 WIKIPEDIA TOPIC MINING
    // ========================================
    
    /**
     * Mine related topics from Wikipedia
     * Great for finding semantic relationships and subtopics
     */
    private function get_wikipedia_topics($query) {
        $cached = AIAB_Database::get_research_cache($query, 'wikipedia');
        if ($cached) {
            return $cached;
        }
        
        $topics = array();
        
        // Step 1: Search Wikipedia for the topic
        $search_url = 'https://en.wikipedia.org/w/api.php?' . http_build_query(array(
            'action' => 'query',
            'list' => 'search',
            'srsearch' => $query,
            'srlimit' => 5,
            'format' => 'json',
            'origin' => '*'
        ));
        
        $response = wp_remote_get($search_url, array('timeout' => 10));
        
        if (is_wp_error($response)) {
            return $topics;
        }
        
        $data = json_decode(wp_remote_retrieve_body($response), true);
        
        // Get search results as potential topics
        if (!empty($data['query']['search'])) {
            foreach ($data['query']['search'] as $result) {
                $title = $result['title'];
                $snippet = strip_tags($result['snippet']);
                
                $topics[] = array(
                    'keyword' => 'What is ' . $title,
                    'source' => 'wikipedia_search',
                    'snippet' => $snippet,
                    'base_query' => $query,
                    'search_volume' => 'medium'
                );
            }
            
            // Get the first result's page for more data
            if (!empty($data['query']['search'][0]['title'])) {
                $page_title = $data['query']['search'][0]['title'];
                
                // Step 2: Get links from the Wikipedia page (related topics)
                $links_url = 'https://en.wikipedia.org/w/api.php?' . http_build_query(array(
                    'action' => 'query',
                    'titles' => $page_title,
                    'prop' => 'links|categories',
                    'pllimit' => 30,
                    'cllimit' => 20,
                    'format' => 'json',
                    'origin' => '*'
                ));
                
                $links_response = wp_remote_get($links_url, array('timeout' => 10));
                
                if (!is_wp_error($links_response)) {
                    $links_data = json_decode(wp_remote_retrieve_body($links_response), true);
                    
                    if (!empty($links_data['query']['pages'])) {
                        foreach ($links_data['query']['pages'] as $page) {
                            // Extract internal links as related topics
                            if (!empty($page['links'])) {
                                foreach ($page['links'] as $link) {
                                    $link_title = $link['title'];
                                    // Filter out Wikipedia system pages
                                    if (!preg_match('/^(Wikipedia:|Template:|Category:|Help:|File:|Portal:)/i', $link_title)) {
                                        // IMPORTANT: Only include links that are relevant to the base query
                                        // This prevents random unrelated topics like "How to Get Away with Murder"
                                        if ($this->is_topic_relevant($link_title, $query)) {
                                            $topics[] = array(
                                                'keyword' => $link_title,
                                                'source' => 'wikipedia_link',
                                                'base_query' => $query,
                                                'search_volume' => 'low'
                                            );
                                        }
                                    }
                                }
                            }
                            
                            // Extract categories as topic clusters
                            if (!empty($page['categories'])) {
                                foreach ($page['categories'] as $cat) {
                                    $cat_name = str_replace('Category:', '', $cat['title']);
                                    // Filter out meta categories
                                    if (!preg_match('/(articles|pages|Wikipedia|stubs|errors|accuracy)/i', $cat_name)) {
                                        $topics[] = array(
                                            'keyword' => $cat_name,
                                            'source' => 'wikipedia_category',
                                            'base_query' => $query,
                                            'search_volume' => 'low'
                                        );
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
        
        // Step 3: Get Wikipedia "See Also" style suggestions
        $related_url = 'https://en.wikipedia.org/w/api.php?' . http_build_query(array(
            'action' => 'opensearch',
            'search' => $query,
            'limit' => 10,
            'format' => 'json',
            'origin' => '*'
        ));
        
        $related_response = wp_remote_get($related_url, array('timeout' => 10));
        
        if (!is_wp_error($related_response)) {
            $related_data = json_decode(wp_remote_retrieve_body($related_response), true);
            
            if (!empty($related_data[1])) {
                foreach ($related_data[1] as $suggestion) {
                    $topics[] = array(
                        'keyword' => $suggestion,
                        'source' => 'wikipedia_suggest',
                        'base_query' => $query,
                        'search_volume' => 'medium'
                    );
                }
            }
        }
        
        // Deduplicate
        $topics = $this->deduplicate_topics($topics);
        
        AIAB_Database::set_research_cache($query, 'wikipedia', $topics, 24);
        
        return $topics;
    }
    
    // ========================================
    // 🆕 REDDIT TOPIC MINING
    // ========================================
    
    /**
     * Mine discussion topics from Reddit
     * Great for finding real questions people are asking
     */
    private function get_reddit_topics($query) {
        $cached = AIAB_Database::get_research_cache($query, 'reddit');
        if ($cached) {
            return $cached;
        }
        
        $topics = array();
        
        // Search Reddit (public JSON API - no auth required)
        $url = 'https://www.reddit.com/search.json?' . http_build_query(array(
            'q' => $query,
            'sort' => 'relevance',
            'limit' => 25,
            't' => 'year' // Last year
        ));
        
        $response = wp_remote_get($url, array(
            'timeout' => 15,
            'headers' => array(
                'User-Agent' => 'Mozilla/5.0 (compatible; WordPressBot/1.0; +http://wordpress.org/)'
            )
        ));
        
        if (is_wp_error($response)) {
            AIAB_Logger::debug("Reddit API error: " . $response->get_error_message());
            return $topics;
        }
        
        $status = wp_remote_retrieve_response_code($response);
        if ($status !== 200) {
            AIAB_Logger::debug("Reddit API returned status: $status");
            return $topics;
        }
        
        $data = json_decode(wp_remote_retrieve_body($response), true);
        
        if (!empty($data['data']['children'])) {
            foreach ($data['data']['children'] as $post) {
                $post_data = $post['data'];
                $title = $post_data['title'] ?? '';
                $subreddit = $post_data['subreddit'] ?? '';
                $score = $post_data['score'] ?? 0;
                $num_comments = $post_data['num_comments'] ?? 0;
                
                // Skip low-quality posts
                if ($score < 5 || strlen($title) < 15) {
                    continue;
                }
                
                // Clean the title
                $title = html_entity_decode($title, ENT_QUOTES, 'UTF-8');
                $title = preg_replace('/\[.*?\]/', '', $title); // Remove [tags]
                
                // Remove Reddit-specific patterns
                $title = preg_replace('/\s*[:\-|]\s*r\/\w+\s*$/i', '', $title); // Remove : r/subreddit
                $title = preg_replace('/\s*\(r\/\w+\)\s*$/i', '', $title); // Remove (r/subreddit)
                $title = preg_replace('/\s*from\s+r\/\w+\s*$/i', '', $title); // Remove "from r/subreddit"
                $title = preg_replace('/\s*on\s+r\/\w+\s*$/i', '', $title); // Remove "on r/subreddit"
                $title = preg_replace('/\s*via\s+r\/\w+\s*$/i', '', $title); // Remove "via r/subreddit"
                $title = preg_replace('/r\/\w+/i', '', $title); // Remove any remaining r/subreddit references
                
                $title = trim($title);
                
                if (strlen($title) > 10 && strlen($title) < 200) {
                    // Determine search volume based on engagement
                    $volume = 'low';
                    if ($score > 100 || $num_comments > 50) {
                        $volume = 'high';
                    } elseif ($score > 20 || $num_comments > 10) {
                        $volume = 'medium';
                    }
                    
                    $topics[] = array(
                        'keyword' => $title,
                        'source' => 'reddit',
                        'subreddit' => $subreddit,
                        'score' => $score,
                        'comments' => $num_comments,
                        'base_query' => $query,
                        'search_volume' => $volume
                    );
                }
            }
        }
        
        // Also try to get questions specifically
        $question_url = 'https://www.reddit.com/search.json?' . http_build_query(array(
            'q' => $query . ' (how OR what OR why OR when OR where)',
            'sort' => 'relevance',
            'limit' => 15,
            't' => 'year'
        ));
        
        $question_response = wp_remote_get($question_url, array(
            'timeout' => 15,
            'headers' => array(
                'User-Agent' => 'Mozilla/5.0 (compatible; WordPressBot/1.0)'
            )
        ));
        
        if (!is_wp_error($question_response)) {
            $question_data = json_decode(wp_remote_retrieve_body($question_response), true);
            
            if (!empty($question_data['data']['children'])) {
                foreach ($question_data['data']['children'] as $post) {
                    $post_data = $post['data'];
                    $title = $post_data['title'] ?? '';
                    
                    // Only include if it's actually a question
                    if (preg_match('/^(how|what|why|when|where|which|who|can|does|is|are|should|will|would)/i', $title)) {
                        $title = html_entity_decode($title, ENT_QUOTES, 'UTF-8');
                        $title = preg_replace('/\[.*?\]/', '', $title);
                        $title = trim($title);
                        
                        if (strlen($title) > 15 && strlen($title) < 150) {
                            $topics[] = array(
                                'keyword' => $title,
                                'source' => 'reddit_question',
                                'base_query' => $query,
                                'search_volume' => 'high' // Questions are valuable
                            );
                        }
                    }
                }
            }
        }
        
        // Deduplicate
        $topics = $this->deduplicate_topics($topics);
        
        // Rate limit
        usleep(500000); // 500ms delay - be nice to Reddit
        
        AIAB_Database::set_research_cache($query, 'reddit', $topics, 6);
        
        return $topics;
    }
    
    // ========================================
    // 🆕 GOOGLE TRENDS INTEGRATION
    // ========================================
    
    /**
     * Get trending and related topics from Google Trends
     * Uses the unofficial Google Trends API
     */
    private function get_google_trends($query) {
        $cached = AIAB_Database::get_research_cache($query, 'google_trends');
        if ($cached) {
            return $cached;
        }
        
        $topics = array();
        
        // Method 1: Google Trends Autocomplete
        $autocomplete_url = 'https://trends.google.com/trends/api/autocomplete/' . urlencode($query) . '?hl=en-US';
        
        $response = wp_remote_get($autocomplete_url, array(
            'timeout' => 15,
            'headers' => array(
                'User-Agent' => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36'
            )
        ));
        
        if (!is_wp_error($response)) {
            $body = wp_remote_retrieve_body($response);
            // Google Trends returns JSON with a prefix that needs to be removed
            $body = preg_replace('/^\)\]\}\',\n/', '', $body);
            $data = json_decode($body, true);
            
            if (!empty($data['default']['topics'])) {
                foreach ($data['default']['topics'] as $topic) {
                    $title = $topic['title'] ?? '';
                    $type = $topic['type'] ?? '';
                    
                    if (!empty($title)) {
                        $topics[] = array(
                            'keyword' => $title,
                            'source' => 'google_trends_topic',
                            'type' => $type,
                            'base_query' => $query,
                            'search_volume' => 'high' // Trending = high interest
                        );
                    }
                }
            }
        }
        
        // Method 2: Related Queries via explore endpoint
        // First, we need to get a token
        $explore_url = 'https://trends.google.com/trends/api/explore?' . http_build_query(array(
            'hl' => 'en-US',
            'tz' => '-240',
            'req' => json_encode(array(
                'comparisonItem' => array(
                    array(
                        'keyword' => $query,
                        'geo' => 'US',
                        'time' => 'today 12-m'
                    )
                ),
                'category' => 0,
                'property' => ''
            ))
        ));
        
        $explore_response = wp_remote_get($explore_url, array(
            'timeout' => 15,
            'headers' => array(
                'User-Agent' => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36'
            )
        ));
        
        if (!is_wp_error($explore_response)) {
            $body = wp_remote_retrieve_body($explore_response);
            $body = preg_replace('/^\)\]\}\',\n/', '', $body);
            $data = json_decode($body, true);
            
            // Extract related queries token
            if (!empty($data['widgets'])) {
                foreach ($data['widgets'] as $widget) {
                    if (isset($widget['id']) && $widget['id'] === 'RELATED_QUERIES') {
                        $token = $widget['token'] ?? '';
                        $request = $widget['request'] ?? array();
                        
                        if (!empty($token)) {
                            // Fetch related queries
                            $related_url = 'https://trends.google.com/trends/api/widgetdata/relatedsearches?' . http_build_query(array(
                                'hl' => 'en-US',
                                'tz' => '-240',
                                'req' => json_encode($request),
                                'token' => $token
                            ));
                            
                            $related_response = wp_remote_get($related_url, array(
                                'timeout' => 15,
                                'headers' => array(
                                    'User-Agent' => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36'
                                )
                            ));
                            
                            if (!is_wp_error($related_response)) {
                                $related_body = wp_remote_retrieve_body($related_response);
                                $related_body = preg_replace('/^\)\]\}\',\n/', '', $related_body);
                                $related_data = json_decode($related_body, true);
                                
                                // Rising queries (trending up!)
                                if (!empty($related_data['default']['rankedList'][0]['rankedKeyword'])) {
                                    foreach ($related_data['default']['rankedList'][0]['rankedKeyword'] as $kw) {
                                        $topics[] = array(
                                            'keyword' => $kw['query'] ?? '',
                                            'source' => 'google_trends_rising',
                                            'base_query' => $query,
                                            'search_volume' => 'high' // Rising = opportunity!
                                        );
                                    }
                                }
                                
                                // Top queries (established)
                                if (!empty($related_data['default']['rankedList'][1]['rankedKeyword'])) {
                                    foreach ($related_data['default']['rankedList'][1]['rankedKeyword'] as $kw) {
                                        $topics[] = array(
                                            'keyword' => $kw['query'] ?? '',
                                            'source' => 'google_trends_top',
                                            'base_query' => $query,
                                            'search_volume' => 'high'
                                        );
                                    }
                                }
                            }
                        }
                        break;
                    }
                }
            }
        }
        
        // Method 3: Daily trends (what's hot today)
        $daily_url = 'https://trends.google.com/trends/api/dailytrends?' . http_build_query(array(
            'hl' => 'en-US',
            'tz' => '-240',
            'geo' => 'US',
            'ns' => 15
        ));
        
        $daily_response = wp_remote_get($daily_url, array(
            'timeout' => 15,
            'headers' => array(
                'User-Agent' => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36'
            )
        ));
        
        if (!is_wp_error($daily_response)) {
            $body = wp_remote_retrieve_body($daily_response);
            $body = preg_replace('/^\)\]\}\',\n/', '', $body);
            $data = json_decode($body, true);
            
            if (!empty($data['default']['trendingSearchesDays'])) {
                foreach ($data['default']['trendingSearchesDays'] as $day) {
                    if (!empty($day['trendingSearches'])) {
                        foreach (array_slice($day['trendingSearches'], 0, 10) as $trend) {
                            $title = $trend['title']['query'] ?? '';
                            
                            // Only include if somewhat related to our query
                            if (!empty($title) && (
                                stripos($title, $query) !== false ||
                                stripos($query, $title) !== false ||
                                similar_text(strtolower($title), strtolower($query)) > 30
                            )) {
                                $topics[] = array(
                                    'keyword' => $title,
                                    'source' => 'google_trends_daily',
                                    'traffic' => $trend['formattedTraffic'] ?? '',
                                    'base_query' => $query,
                                    'search_volume' => 'high'
                                );
                            }
                        }
                    }
                }
            }
        }
        
        // Filter and deduplicate
        $topics = array_filter($topics, function($t) {
            return !empty($t['keyword']) && strlen($t['keyword']) > 2;
        });
        $topics = $this->deduplicate_topics($topics);
        
        // Rate limit
        usleep(300000); // 300ms delay
        
        AIAB_Database::set_research_cache($query, 'google_trends', $topics, 4);
        
        return $topics;
    }
    
    // ========================================
    // GOOGLE CUSTOM SEARCH API PROVIDER (Recommended!)
    // ========================================
    
    /**
     * Research using Google Custom Search API
     * Official API - no rate limiting, reliable results
     * 100 free queries/day, then $5/1000
     */
    private function research_with_google_cse($area) {
        $topics = array();
        
        AIAB_Logger::info("🔵 Using Google Custom Search API for: $area");
        
        // Check cache first
        $cached = AIAB_Database::get_research_cache($area, 'google_cse');
        if ($cached) {
            AIAB_Logger::info("Using cached Google CSE results for: $area");
            return $cached;
        }
        
        // Build variations - OPTIMIZED: Only 3 queries to conserve API quota
        $queries = array(
            $area,                              // Base query
            'best ' . $area,                    // Commercial/comparison intent
            $area . ' guide',                   // Informational intent
        );
        
        foreach ($queries as $query) {
            $results = $this->google_cse_search($query);
            
            if ($results === false) {
                AIAB_Logger::warning("Google CSE API error for query: $query");
                continue;
            }
            
            // Extract topics from organic results
            if (!empty($results['items'])) {
                foreach ($results['items'] as $item) {
                    // Extract title as potential topic
                    $title = $this->clean_search_title($item['title']);
                    if (!empty($title) && strlen($title) > 10) {
                        $topics[] = array(
                            'keyword' => $title,
                            'source' => 'google_cse_organic',
                            'base_query' => $area,
                            'url' => $item['link'],
                            'snippet' => $item['snippet'] ?? '',
                            'search_volume' => 'high'
                        );
                    }
                    
                    // Extract questions from snippets
                    if (!empty($item['snippet'])) {
                        $questions = $this->extract_questions_from_text($item['snippet']);
                        foreach ($questions as $q) {
                            $topics[] = array(
                                'keyword' => $q,
                                'source' => 'google_cse_snippet',
                                'base_query' => $area,
                                'search_volume' => 'medium'
                            );
                        }
                    }
                }
            }
            
            // Extract related searches (if available)
            if (!empty($results['queries']['relatedSearches'])) {
                foreach ($results['queries']['relatedSearches'] as $related) {
                    $topics[] = array(
                        'keyword' => $related['title'],
                        'source' => 'google_cse_related',
                        'base_query' => $area,
                        'search_volume' => 'high'
                    );
                }
            }
            
            // Extract "request" suggestions (next page queries)
            if (!empty($results['queries']['request'])) {
                foreach ($results['queries']['request'] as $req) {
                    if (!empty($req['searchTerms']) && $req['searchTerms'] !== $query) {
                        $topics[] = array(
                            'keyword' => $req['searchTerms'],
                            'source' => 'google_cse_suggestion',
                            'base_query' => $area,
                            'search_volume' => 'medium'
                        );
                    }
                }
            }
            
            // Small delay between queries (be nice to API)
            usleep(100000); // 100ms
        }
        
        // Deduplicate
        $topics = $this->deduplicate_topics($topics);
        
        // Note: Using Google CSE only - no free supplements mixed in
        // Google CSE provides high-quality structured data
        
        AIAB_Logger::info("🎯 Google CSE total topics: " . count($topics));
        
        // Cache results for 24 hours (conserve API quota)
        AIAB_Database::set_research_cache($area, 'google_cse', $topics, 24);
        
        return $topics;
    }
    
    /**
     * Make a Google Custom Search API request
     */
    private function google_cse_search($query) {
        // Skip if quota already exceeded this session
        if ($this->google_cse_quota_exceeded) {
            return false;
        }
        
        $geo = $this->get_geo_params();
        
        // Build params with geo targeting
        $params = array(
            'key' => $this->google_cse_key,
            'cx' => $this->google_cse_cx,
            'q' => $query,
            'num' => 10,  // Max results per query
        );
        
        // Add geo-targeting parameters if locality is set
        if (!empty($geo)) {
            if (isset($geo['gl'])) {
                $params['gl'] = $geo['gl']; // Geolocation (country code)
            }
            if (isset($geo['hl'])) {
                $params['hl'] = $geo['hl']; // Interface language
            }
            if (isset($geo['cr'])) {
                $params['cr'] = $geo['cr']; // Country restriction
            }
            
            AIAB_Logger::debug("🌍 Google CSE geo-targeting: " . ($geo['name'] ?? 'Unknown'));
        }
        
        $url = 'https://www.googleapis.com/customsearch/v1?' . http_build_query($params);
        
        $response = wp_remote_get($url, array(
            'timeout' => 15,
            'headers' => array(
                'Accept' => 'application/json',
            )
        ));
        
        if (is_wp_error($response)) {
            AIAB_Logger::error("Google CSE request failed: " . $response->get_error_message());
            return false;
        }
        
        $status_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);
        
        if ($status_code !== 200) {
            $error_msg = isset($data['error']['message']) ? $data['error']['message'] : 'Unknown error';
            AIAB_Logger::error("Google CSE API error ({$status_code}): {$error_msg}");
            
            // Check for quota exceeded - set flag to prevent further calls this session
            if ($status_code === 429 || strpos($error_msg, 'quota') !== false || strpos($error_msg, 'Quota') !== false) {
                $this->google_cse_quota_exceeded = true;
                AIAB_Logger::warning("Google CSE quota exceeded - stopping all CSE calls for this session. Will use fallback provider.");
                
                // Store in transient to remember across requests for 1 hour
                set_transient('aiab_google_cse_quota_exceeded', true, HOUR_IN_SECONDS);
            }
            
            return false;
        }
        
        return $data;
    }
    
    /**
     * Clean search result title to extract topic
     */
    private function clean_search_title($title) {
        // Remove common site name patterns
        $title = preg_replace('/\s*[\|\-–—]\s*[^|–—]+$/', '', $title);
        // Remove years in parentheses
        $title = preg_replace('/\s*\(\d{4}\)\s*/', ' ', $title);
        // Remove "..." at the end
        $title = preg_replace('/\.{3}$/', '', $title);
        // Clean up whitespace
        $title = trim(preg_replace('/\s+/', ' ', $title));
        
        return $title;
    }
    
    /**
     * Extract questions from text
     */
    private function extract_questions_from_text($text) {
        $questions = array();
        
        // Match sentences that look like questions
        if (preg_match_all('/([A-Z][^.!?]*\?)/u', $text, $matches)) {
            foreach ($matches[1] as $q) {
                $q = trim($q);
                if (strlen($q) > 15 && strlen($q) < 150) {
                    $questions[] = $q;
                }
            }
        }
        
        // Match question patterns without question marks
        $patterns = array(
            '/\b(how to [^.!?]{10,60})/i',
            '/\b(what is [^.!?]{10,60})/i',
            '/\b(why do [^.!?]{10,60})/i',
            '/\b(when to [^.!?]{10,60})/i',
        );
        
        foreach ($patterns as $pattern) {
            if (preg_match_all($pattern, $text, $matches)) {
                foreach ($matches[1] as $q) {
                    $questions[] = trim($q);
                }
            }
        }
        
        return array_unique($questions);
    }
    
    // ========================================
    // VALUESERP PROVIDER (Affordable - 100 free/month)
    // ========================================
    
    /**
     * Research using ValueSERP API
     */
    private function research_with_valueserp($area) {
        $topics = array();
        
        AIAB_Logger::info("💰 Using ValueSERP provider for: $area");
        
        // Source 1: People Also Ask
        $paa = $this->get_valueserp_questions($area);
        AIAB_Logger::info("ValueSERP PAA returned " . count($paa) . " questions");
        $topics = array_merge($topics, $paa);
        
        // Source 2: Related Searches
        $related = $this->get_valueserp_related($area);
        AIAB_Logger::info("ValueSERP Related returned " . count($related) . " topics");
        $topics = array_merge($topics, $related);
        
        // Source 3: Autocomplete (still use free Google for this)
        $autocomplete = $this->get_google_autocomplete_enhanced($area);
        AIAB_Logger::info("Google Autocomplete returned " . count($autocomplete) . " suggestions");
        $topics = array_merge($topics, $autocomplete);
        
        return $topics;
    }
    
    /**
     * ValueSERP - Get People Also Ask and organic questions
     */
    private function get_valueserp_questions($query) {
        $cached = AIAB_Database::get_research_cache($query, 'valueserp_paa');
        if ($cached) {
            return $cached;
        }
        
        $topics = array();
        
        $url = 'https://api.valueserp.com/search?' . http_build_query(array(
            'api_key' => $this->valueserp_key,
            'q' => $query,
            'location' => 'United States',
            'google_domain' => 'google.com',
            'gl' => 'us',
            'hl' => 'en',
            'include_answer_box' => 'true'
        ));
        
        AIAB_Logger::debug("ValueSERP request: $query");
        
        $response = wp_remote_get($url, array('timeout' => 20));
        
        if (is_wp_error($response)) {
            AIAB_Logger::error("ValueSERP error: " . $response->get_error_message());
            return $topics;
        }
        
        $data = json_decode(wp_remote_retrieve_body($response), true);
        
        // Extract People Also Ask
        if (!empty($data['related_questions'])) {
            foreach ($data['related_questions'] as $question) {
                $q_text = $question['question'] ?? '';
                if (!empty($q_text)) {
                    $topics[] = array(
                        'keyword' => $q_text,
                        'source' => 'valueserp_paa',
                        'snippet' => $question['answer'] ?? '',
                        'base_query' => $query,
                        'search_volume' => 'high'
                    );
                }
            }
        }
        
        // Extract questions from organic titles
        if (!empty($data['organic_results'])) {
            foreach ($data['organic_results'] as $result) {
                $title = $result['title'] ?? '';
                if (preg_match('/^(what|how|why|when|where|which|who|can|does|is|are|should)\b/i', $title)) {
                    $topics[] = array(
                        'keyword' => $this->clean_title_to_question($title),
                        'source' => 'valueserp_organic',
                        'snippet' => $result['snippet'] ?? '',
                        'base_query' => $query,
                        'search_volume' => 'medium'
                    );
                }
            }
        }
        
        // Extract from Answer Box
        if (!empty($data['answer_box']['answer'])) {
            $topics[] = array(
                'keyword' => 'What is ' . $query,
                'source' => 'valueserp_answer_box',
                'snippet' => $data['answer_box']['answer'],
                'base_query' => $query,
                'search_volume' => 'high'
            );
        }
        
        AIAB_Database::set_research_cache($query, 'valueserp_paa', $topics, 24);
        
        return $topics;
    }
    
    /**
     * ValueSERP - Get Related Searches
     */
    private function get_valueserp_related($query) {
        $cached = AIAB_Database::get_research_cache($query, 'valueserp_related');
        if ($cached) {
            return $cached;
        }
        
        $topics = array();
        
        $url = 'https://api.valueserp.com/search?' . http_build_query(array(
            'api_key' => $this->valueserp_key,
            'q' => $query,
            'location' => 'United States',
            'google_domain' => 'google.com'
        ));
        
        $response = wp_remote_get($url, array('timeout' => 20));
        
        if (is_wp_error($response)) {
            return $topics;
        }
        
        $data = json_decode(wp_remote_retrieve_body($response), true);
        
        // Related searches
        if (!empty($data['related_searches'])) {
            foreach ($data['related_searches'] as $related) {
                $topics[] = array(
                    'keyword' => $related['query'] ?? $related,
                    'source' => 'valueserp_related',
                    'base_query' => $query,
                    'search_volume' => 'medium'
                );
            }
        }
        
        AIAB_Database::set_research_cache($query, 'valueserp_related', $topics, 24);
        
        return $topics;
    }
    
    // ========================================
    // SERPAPI PROVIDER (Premium)
    // ========================================
    
    /**
     * Research using SerpAPI
     */
    private function research_with_serpapi($area) {
        $topics = array();
        
        AIAB_Logger::info("💎 Using SerpAPI provider for: $area");
        
        // Source 1: People Also Ask
        $paa = $this->get_serpapi_questions($area);
        AIAB_Logger::info("SerpAPI PAA returned " . count($paa) . " questions");
        $topics = array_merge($topics, $paa);
        
        // Source 2: Related Searches
        $related = $this->get_serpapi_related_searches($area);
        AIAB_Logger::info("SerpAPI Related returned " . count($related) . " topics");
        $topics = array_merge($topics, $related);
        
        // Source 3: Autocomplete
        $autocomplete = $this->get_serpapi_autocomplete($area);
        AIAB_Logger::info("SerpAPI Autocomplete returned " . count($autocomplete) . " suggestions");
        $topics = array_merge($topics, $autocomplete);
        
        return $topics;
    }
    
    /**
     * SerpAPI - Get People Also Ask questions
     */
    private function get_serpapi_questions($query) {
        $geo = $this->get_geo_params();
        $cache_key = $query . ($this->primary_locality ? '_' . $this->primary_locality : '');
        
        $cached = AIAB_Database::get_research_cache($cache_key, 'serpapi_paa');
        if ($cached) {
            return $cached;
        }
        
        $topics = array();
        
        // Build query params with geo targeting
        $params = array(
            'q' => $query . $this->get_location_suffix(),
            'api_key' => $this->serpapi_key,
            'engine' => 'google',
            'gl' => isset($geo['gl']) ? $geo['gl'] : 'us',
            'hl' => isset($geo['hl']) ? $geo['hl'] : 'en',
            'num' => 20
        );
        
        if ($this->primary_locality) {
            AIAB_Logger::debug("🌍 SerpAPI PAA geo-targeted to: " . $this->get_locality_name($this->primary_locality));
        }
        
        $url = 'https://serpapi.com/search.json?' . http_build_query($params);
        
        $response = wp_remote_get($url, array('timeout' => 20));
        
        if (is_wp_error($response)) {
            AIAB_Logger::error("SerpAPI PAA error: " . $response->get_error_message());
            return $topics;
        }
        
        $data = json_decode(wp_remote_retrieve_body($response), true);
        
        if (!empty($data['related_questions'])) {
            foreach ($data['related_questions'] as $question) {
                $topics[] = array(
                    'keyword' => $question['question'],
                    'source' => 'serpapi_paa',
                    'snippet' => $question['snippet'] ?? '',
                    'base_query' => $query,
                    'search_volume' => 'high',
                    'locality' => $this->primary_locality
                );
            }
        }
        
        AIAB_Database::set_research_cache($cache_key, 'serpapi_paa', $topics, 24);
        
        return $topics;
    }
    
    /**
     * SerpAPI - Get Related Searches
     */
    private function get_serpapi_related_searches($query) {
        $geo = $this->get_geo_params();
        $cache_key = $query . ($this->primary_locality ? '_' . $this->primary_locality : '');
        
        $cached = AIAB_Database::get_research_cache($cache_key, 'serpapi_related');
        if ($cached) {
            return $cached;
        }
        
        $topics = array();
        
        $params = array(
            'q' => $query . $this->get_location_suffix(),
            'api_key' => $this->serpapi_key,
            'engine' => 'google',
            'gl' => isset($geo['gl']) ? $geo['gl'] : 'us',
            'hl' => isset($geo['hl']) ? $geo['hl'] : 'en'
        );
        
        $url = 'https://serpapi.com/search.json?' . http_build_query($params);
        
        $response = wp_remote_get($url, array('timeout' => 20));
        
        if (is_wp_error($response)) {
            return $topics;
        }
        
        $data = json_decode(wp_remote_retrieve_body($response), true);
        
        if (!empty($data['related_searches'])) {
            foreach ($data['related_searches'] as $search) {
                $topics[] = array(
                    'keyword' => $search['query'],
                    'source' => 'serpapi_related',
                    'base_query' => $query,
                    'search_volume' => 'medium',
                    'locality' => $this->primary_locality
                );
            }
        }
        
        AIAB_Database::set_research_cache($cache_key, 'serpapi_related', $topics, 24);
        
        return $topics;
    }
    
    /**
     * Get Autocomplete suggestions from SerpAPI
     */
    private function get_serpapi_autocomplete($query) {
        $geo = $this->get_geo_params();
        $cache_key = $query . ($this->primary_locality ? '_' . $this->primary_locality : '');
        
        $cached = AIAB_Database::get_research_cache($cache_key, 'serpapi_autocomplete');
        if ($cached) {
            return $cached;
        }
        
        $topics = array();
        
        // Query with question modifiers for better results
        $queries = array(
            $query . $this->get_location_suffix(),
            "how to $query" . $this->get_location_suffix(),
            "what is $query" . $this->get_location_suffix(),
            "why $query" . $this->get_location_suffix(),
            "best $query" . $this->get_location_suffix(),
            "$query vs",
            "$query cost" . $this->get_location_suffix(),
            "$query problems"
        );
        
        foreach ($queries as $q) {
            $params = array(
                'q' => $q,
                'api_key' => $this->serpapi_key,
                'engine' => 'google_autocomplete',
                'gl' => isset($geo['gl']) ? $geo['gl'] : 'us',
                'hl' => isset($geo['hl']) ? $geo['hl'] : 'en'
            );
            
            $url = 'https://serpapi.com/search.json?' . http_build_query($params);
            
            $response = wp_remote_get($url, array('timeout' => 15));
            
            if (is_wp_error($response)) {
                continue;
            }
            
            $data = json_decode(wp_remote_retrieve_body($response), true);
            
            if (!empty($data['suggestions'])) {
                foreach ($data['suggestions'] as $suggestion) {
                    $value = $suggestion['value'] ?? '';
                    if (!empty($value)) {
                        $topics[] = array(
                            'keyword' => $value,
                            'source' => 'autocomplete',
                            'base_query' => $query,
                            'search_volume' => 'medium',
                            'locality' => $this->primary_locality
                        );
                    }
                }
            }
            
            usleep(100000); // Rate limiting
        }
        
        AIAB_Database::set_research_cache($cache_key, 'serpapi_autocomplete', $topics, 24);
        
        return $topics;
    }
    
    /**
     * Fallback: Get Google Autocomplete suggestions (free, no API key)
     */
    private function get_google_autocomplete($query) {
        $cached = AIAB_Database::get_research_cache($query, 'autocomplete');
        if ($cached) {
            return $cached;
        }
        
        $topics = array();
        
        // Use question-focused modifiers
        $modifiers = array(
            'how to ',
            'what is ',
            'why ',
            'best ',
            'top ',
            '',
            $query . ' vs ',
            $query . ' cost ',
            $query . ' problems ',
        );
        
        $success_count = 0;
        $error_count = 0;
        
        foreach ($modifiers as $modifier) {
            $search_query = $modifier . $query;
            $url = 'https://suggestqueries.google.com/complete/search?client=firefox&q=' . urlencode($search_query);
            
            $response = wp_remote_get($url, array(
                'timeout' => 10,
                'headers' => array(
                    'User-Agent' => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36',
                    'Accept' => 'application/json, text/javascript, */*; q=0.01',
                    'Accept-Language' => 'en-US,en;q=0.9',
                )
            ));
            
            if (is_wp_error($response)) {
                $error_count++;
                AIAB_Logger::debug("Google Autocomplete error: " . $response->get_error_message());
                continue;
            }
            
            $status_code = wp_remote_retrieve_response_code($response);
            if ($status_code !== 200) {
                $error_count++;
                AIAB_Logger::debug("Google Autocomplete HTTP {$status_code} for: {$search_query}");
                continue;
            }
            
            $body = wp_remote_retrieve_body($response);
            $data = json_decode($body, true);
            
            if (isset($data[1]) && is_array($data[1])) {
                $success_count++;
                foreach ($data[1] as $suggestion) {
                    $topics[] = array(
                        'keyword' => $suggestion,
                        'source' => 'autocomplete',
                        'modifier' => $modifier,
                        'base_query' => $query,
                        'search_volume' => 'unknown'
                    );
                }
            }
            
            usleep(300000); // Rate limiting - 300ms between requests
        }
        
        // Log if we're getting blocked
        if ($error_count > 0 && $success_count === 0) {
            AIAB_Logger::warning("Google Autocomplete blocked - all {$error_count} requests failed. Your server IP may be rate-limited.");
        }
        
        AIAB_Database::set_research_cache($query, 'autocomplete', $topics, 12);
        
        return $topics;
    }
    
    /**
     * Deduplicate topics by keyword (case-insensitive)
     */
    private function deduplicate_topics($topics) {
        $seen = array();
        $unique = array();
        
        foreach ($topics as $topic) {
            $key = strtolower(trim($topic['keyword']));
            if (!isset($seen[$key])) {
                $seen[$key] = true;
                $unique[] = $topic;
            }
        }
        
        return $unique;
    }
    
    /**
     * Check if a topic is relevant to the base query/niche
     * Prevents random Wikipedia links like "How to Get Away with Murder" from being included
     */
    private function is_topic_relevant($topic, $base_query) {
        $topic_lower = strtolower($topic);
        $query_lower = strtolower($base_query);
        
        // Extract key words from base query (remove common words)
        $stop_words = array('how', 'to', 'what', 'is', 'are', 'the', 'a', 'an', 'in', 'on', 'for', 'of', 'and', 'or', 'with', 'do', 'does', 'can', 'why', 'when', 'where', 'which');
        $query_words = array_diff(explode(' ', $query_lower), $stop_words);
        $query_words = array_filter($query_words, function($w) { return strlen($w) > 2; });
        
        // If no meaningful words in query, be very restrictive
        if (empty($query_words)) {
            return false;
        }
        
        // Check if topic contains ANY of the base query keywords
        foreach ($query_words as $word) {
            if (strpos($topic_lower, $word) !== false) {
                return true;
            }
        }
        
        // Blacklist common irrelevant patterns (TV shows, movies, people names)
        $blacklist_patterns = array(
            '/\b(season|episode|series|film|movie|tv|actor|actress|singer|band|album)\b/i',
            '/\b(murder|killing|death|dead|die|kill|crime|criminal)\b/i',  // Unless your niche is crime
            '/\b(drugs|cocaine|heroin|marijuana|illegal)\b/i',             // Unless your niche is pharma
            '/\b(sex|porn|xxx|nude|naked)\b/i',                            // Adult content
            '/\b(netflix|hulu|amazon|disney|hbo|streaming)\b/i',           // Entertainment
            '/\b(trailer|review|cast|plot|ending|spoiler)\b/i',            // Media reviews
            '/\b(guy|girl|boyfriend|girlfriend|dating|romance)\b/i',       // Unless dating niche
            '/\d{4}\s*(film|tv|series|show|movie)/i',                      // Year + media type
        );
        
        // Allow crime-related terms if the niche is pest control (killing pests, etc.)
        $pest_related = array('termite', 'pest', 'insect', 'bug', 'roach', 'ant', 'rodent', 'exterminator');
        $is_pest_niche = false;
        foreach ($pest_related as $pest_word) {
            if (strpos($query_lower, $pest_word) !== false) {
                $is_pest_niche = true;
                break;
            }
        }
        
        // Apply blacklist (but skip murder/killing check for pest control niche)
        foreach ($blacklist_patterns as $i => $pattern) {
            // Skip murder/killing pattern for pest control niche
            if ($is_pest_niche && $i === 1) {
                continue;
            }
            if (preg_match($pattern, $topic)) {
                return false;
            }
        }
        
        // Topic doesn't contain query words and isn't blacklisted - reject it
        // (Better to be strict than include garbage)
        return false;
    }
    
    /**
     * Clean a title to extract the question
     */
    private function clean_title_to_question($title) {
        // Remove site names, dates, etc.
        $title = preg_replace('/\s*[\|\-–—]\s*.+$/', '', $title);
        $title = preg_replace('/\(\d{4}\)/', '', $title);
        $title = trim($title);
        return $title;
    }
    
    /**
     * Sanitize all topics - clean up formatting, remove source references
     */
    private function sanitize_all_topics($topics) {
        $sanitized = array();
        
        foreach ($topics as $topic) {
            $keyword = $topic['keyword'];
            
            // Remove Reddit subreddit references
            $keyword = preg_replace('/\s*[:\-|]\s*r\/\w+\s*$/i', '', $keyword);
            $keyword = preg_replace('/\s*\(r\/\w+\)\s*$/i', '', $keyword);
            $keyword = preg_replace('/\s*from\s+r\/\w+\s*$/i', '', $keyword);
            $keyword = preg_replace('/\s*on\s+r\/\w+\s*$/i', '', $keyword);
            $keyword = preg_replace('/\s*via\s+r\/\w+\s*$/i', '', $keyword);
            $keyword = preg_replace('/\br\/\w+\b/i', '', $keyword);
            
            // Remove other source indicators
            $keyword = preg_replace('/\s*-\s*(reddit|quora|stackoverflow|yahoo answers)$/i', '', $keyword);
            $keyword = preg_replace('/\s*\|\s*[^|]+$/i', '', $keyword); // Remove " | Site Name"
            
            // Remove brackets with source info
            $keyword = preg_replace('/\s*\[[^\]]*reddit[^\]]*\]\s*/i', '', $keyword);
            $keyword = preg_replace('/\s*\([^)]*reddit[^)]*\)\s*/i', '', $keyword);
            
            // Clean up whitespace
            $keyword = preg_replace('/\s+/', ' ', $keyword);
            $keyword = trim($keyword, ' :|-');
            $keyword = trim($keyword);
            
            // Skip if too short after cleaning
            if (strlen($keyword) < 10) {
                continue;
            }
            
            $topic['keyword'] = $keyword;
            $sanitized[] = $topic;
        }
        
        return $sanitized;
    }
    
    /**
     * Filter out topics already written
     */
    private function filter_written_topics($topics) {
        $filtered = array();
        
        foreach ($topics as $topic) {
            if (!AIAB_Database::is_topic_written($this->persona->get_id(), $topic['keyword'])) {
                $filtered[] = $topic;
            }
        }
        
        return $filtered;
    }
    
    /**
     * Score topics by SEO potential
     * Higher score = better ranking opportunity
     */
    private function score_topics($topics) {
        $scored = array();
        
        // Blacklist patterns that should NEVER be selected (TV shows, movies, etc.)
        $blacklist_patterns = array(
            '/\b(season|episode|series|film|movie|tv|actor|actress|singer|band|album)\b/i',
            '/\b(murder|killing|death|dead|die|kill|crime|criminal)\b/i',
            '/\b(drugs|cocaine|heroin|marijuana|illegal)\b/i',
            '/\b(sex|porn|xxx|nude|naked)\b/i',
            '/\b(netflix|hulu|amazon prime|disney\+|hbo|streaming)\b/i',
            '/\b(trailer|cast|plot|ending|spoiler)\b/i',
            '/\b(boyfriend|girlfriend|dating|romance|lose a guy|get a guy)\b/i',
            '/\d{4}\s*(film|tv|series|show|movie)/i',
            // Wikipedia garbage patterns
            '/\b(list of|category:|cs1|sources|webarchive|citation|wikidata)\b/i',
            '/\b(franchise|character|fictional|anime|manga|video game)\b/i',
        );
        
        foreach ($topics as $topic) {
            $score = 0;
            $keyword = strtolower($topic['keyword']);
            
            // BLACKLIST CHECK - Skip garbage topics entirely
            $is_blacklisted = false;
            foreach ($blacklist_patterns as $pattern) {
                if (preg_match($pattern, $keyword)) {
                    $is_blacklisted = true;
                    break;
                }
            }
            if ($is_blacklisted) {
                AIAB_Logger::debug("Skipping blacklisted topic: " . $topic['keyword']);
                continue; // Don't even include in scored list
            }
            
            // SOURCE SCORING (real search data = higher score)
            switch ($topic['source']) {
                case 'people_also_ask':
                    $score += 50; // HIGHEST - proven questions people ask!
                    break;
                case 'google_cse_organic':
                    $score += 45; // Official Google results
                    break;
                case 'google_cse_related':
                    $score += 40; // Google's related searches (official API)
                    break;
                case 'organic_question':
                    $score += 40; // Questions ranking in Google
                    break;
                case 'related_searches':
                    $score += 35; // Google's related searches
                    break;
                case 'google_cse_snippet':
                    $score += 30; // Questions extracted from snippets
                    break;
                case 'autocomplete':
                case 'google_autocomplete':
                    $score += 25; // Real-time search suggestions
                    break;
                case 'google_cse_suggestion':
                    $score += 25; // API suggestions
                    break;
                case 'ai_generated':
                    $score += 10; // Fallback only
                    break;
            }
            
            // SEARCH VOLUME indicator
            if (isset($topic['search_volume'])) {
                switch ($topic['search_volume']) {
                    case 'high': $score += 20; break;
                    case 'medium': $score += 10; break;
                }
            }
            
            // QUESTION FORMAT BONUS (questions rank well!)
            if (preg_match('/^(how|what|why|can|does|is|should|when|where|which)/i', $keyword)) {
                $score += 25;
            }
            
            // Ends with question mark
            if (substr($keyword, -1) === '?') {
                $score += 10;
            }
            
            // COMMERCIAL INTENT bonus (high conversion potential)
            if (preg_match('/(best|top|review|cost|price|buy|vs|compare|cheap|affordable)/i', $keyword)) {
                $score += 20;
            }
            
            // PROBLEM/SOLUTION bonus (high engagement)
            if (preg_match('/(problem|fix|solve|issue|error|won\'t|doesn\'t|not working|how to get rid)/i', $keyword)) {
                $score += 15;
            }
            
            // LENGTH SCORING (long-tail = easier to rank)
            $word_count = str_word_count($keyword);
            if ($word_count >= 4 && $word_count <= 8) {
                $score += 15; // Sweet spot for long-tail
            } elseif ($word_count >= 3) {
                $score += 10;
            }
            
            // EXPERTISE MATCH bonus
            foreach ($this->persona->get_expertise_areas() as $expertise) {
                if (stripos($keyword, $expertise) !== false) {
                    $score += 20;
                    break;
                }
            }
            
            // PASSION MATCH bonus
            foreach ($this->persona->get_passions() as $passion) {
                if (stripos($keyword, $passion) !== false) {
                    $score += 10;
                    break;
                }
            }
            
            $topic['score'] = $score;
            $scored[] = $topic;
        }
        
        // Remove duplicates
        $unique = array();
        $seen = array();
        
        foreach ($scored as $topic) {
            $key = strtolower(trim($topic['keyword']));
            // Normalize for duplicate detection
            $key = preg_replace('/[^a-z0-9\s]/', '', $key);
            $key = preg_replace('/\s+/', ' ', $key);
            
            if (!isset($seen[$key])) {
                $seen[$key] = true;
                $unique[] = $topic;
            }
        }
        
        return $unique;
    }
    
    /**
     * Generate topic using AI (FALLBACK ONLY)
     */
    private function generate_topic_from_ai() {
        AIAB_Logger::warning("Using AI topic generation - search provider recommended for better SEO results");
        
        $ai_writer = new AIAB_AI_Writer($this->persona);
        
        // Build full persona context
        $persona_context = $this->build_persona_context();
        
        $prompt = "You are about to suggest a topic as a real expert.\n\n";
        $prompt .= "=== YOUR IDENTITY ===\n";
        $prompt .= $persona_context;
        
        $prompt .= "=== YOUR TASK ===\n";
        $prompt .= "Suggest ONE specific topic that:\n";
        $prompt .= "1. People ACTUALLY search for on Google\n";
        $prompt .= "2. Matches YOUR expertise and passion\n";
        $prompt .= "3. YOU would naturally write about with authority\n";
        $prompt .= "4. Serves YOUR target audience\n\n";
        
        $prompt .= "=== FORMAT RULES (STRICT!) ===\n";
        $prompt .= "- MAXIMUM 10 words (shorter is better for SEO)\n";
        $prompt .= "- Write it as a search query or short title, NOT a full description\n";
        $prompt .= "- NO colons, NO subtitles, NO explanations\n";
        $prompt .= "- Examples of GOOD topics: 'how to start urban beekeeping', 'best electric cars 2024', 'keto diet meal prep ideas'\n";
        $prompt .= "- Examples of BAD topics: 'Complete Guide to Urban Beekeeping: Everything You Need to Know About Starting Your Own Hive'\n\n";
        $prompt .= "Respond with ONLY the topic (max 10 words), nothing else.";
        
        $topic = $ai_writer->generate_completion($prompt, 100);
        
        // Clean up
        $topic = trim($topic);
        $topic = preg_replace('/^[\"\']|[\"\']$/', '', $topic);
        
        // Remove any colons and everything after (AI sometimes adds subtitles)
        if (strpos($topic, ':') !== false) {
            $topic = trim(explode(':', $topic)[0]);
        }
        
        // Hard limit: truncate to 100 characters max (database will handle longer, but keep it reasonable)
        if (strlen($topic) > 100) {
            $original = $topic;
            // Try to cut at word boundary
            $topic = substr($topic, 0, 100);
            $last_space = strrpos($topic, ' ');
            if ($last_space > 50) {
                $topic = substr($topic, 0, $last_space);
            }
            AIAB_Logger::warning("AI generated overly long topic, truncated", array(
                'original' => $original,
                'truncated' => $topic
            ));
        }
        
        return array(
            'keyword' => $topic,
            'source' => 'ai_persona_generated',
            'score' => 40, // Higher score for persona-driven AI generation
            'search_volume' => 'unknown'
        );
    }
    
    /**
     * Generate supporting topics for a pillar
     * Uses REAL search data to find related questions
     */
    public function generate_supporting_topics($pillar_keyword, $count = 6) {
        AIAB_Logger::log("🧠 Generating $count intelligent supporting topics for pillar: $pillar_keyword");
        
        // Use AI to think strategically as the persona
        $supporting = $this->generate_strategic_supporting_topics($pillar_keyword, $count);
        
        if (count($supporting) >= $count) {
            AIAB_Logger::info("✅ AI generated all $count supporting topics successfully");
            return $supporting;
        }
        
        // If AI didn't generate enough, fall back to search-based discovery
        AIAB_Logger::warning("AI only generated " . count($supporting) . " topics, supplementing with search");
        $needed = $count - count($supporting);
        $existing_keywords = array_map(function($t) { return strtolower($t['keyword']); }, $supporting);
        
        // Get additional topics from search
        $search_topics = $this->get_search_based_topics($pillar_keyword, $needed, $existing_keywords);
        $supporting = array_merge($supporting, $search_topics);
        
        AIAB_Logger::info("Final supporting topics: " . count($supporting));
        foreach ($supporting as $i => $t) {
            AIAB_Logger::info("  " . ($i+1) . ". {$t['keyword']} (source: {$t['source']}, type: {$t['article_type']})");
        }
        
        return $supporting;
    }
    
    /**
     * 🧠 INTELLIGENT TOPIC GENERATION
     * Uses AI to think strategically as the persona and generate cohesive supporting topics
     * This is the CORE intelligence of the Eternal Auto Blogger
     */
    private function generate_strategic_supporting_topics($pillar_keyword, $count = 6) {
        // Skip AI topic generation in budget mode - use search-based only
        if (get_option('aiab_skip_research_ai', 0)) {
            AIAB_Logger::info("💰 Budget mode: Skipping AI topic generation (saves ~2000 tokens)");
            return array(); // Will fall back to search-based discovery
        }
        
        AIAB_Logger::info("🧠 AI Strategic Thinking: Generating supporting topics as persona");
        
        $ai_writer = new AIAB_AI_Writer($this->persona);
        
        // Build the complete persona context
        $persona_context = $this->build_persona_context();
        
        // Build the strategic thinking prompt
        $prompt = $this->build_strategic_prompt($pillar_keyword, $count, $persona_context);
        
        // Budget mode: use fewer tokens for research
        $max_tokens = get_option('aiab_budget_mode', 0) ? 1000 : 2000;
        
        AIAB_Logger::debug("Strategic prompt built, calling AI...", array('max_tokens' => $max_tokens));
        
        try {
            $response = $ai_writer->generate_completion($prompt, $max_tokens);
            return $this->parse_strategic_topics($response, $pillar_keyword);
        } catch (Exception $e) {
            AIAB_Logger::error("AI strategic thinking failed: " . $e->getMessage());
            return array();
        }
    }
    
    /**
     * Build complete persona context for AI
     * The WHOLE persona becomes the filter and creative driver
     */
    private function build_persona_context() {
        $context = "PERSONA IDENTITY:\n";
        $context .= "Name: " . $this->persona->get_name() . "\n\n";
        
        // Personal story & background
        $story = $this->persona->get_personal_story();
        if (!empty($story)) {
            $context .= "BACKGROUND & STORY:\n$story\n\n";
        }
        
        // Expertise
        $expertise = $this->persona->get_expertise_areas();
        if (!empty($expertise)) {
            $context .= "EXPERTISE AREAS:\n";
            foreach ($expertise as $area) {
                $context .= "- $area\n";
            }
            $context .= "\n";
        }
        
        // Passions
        $passions = $this->persona->get_passions();
        if (!empty($passions)) {
            $context .= "PASSIONS (What drives me):\n";
            foreach ($passions as $passion) {
                $context .= "- $passion\n";
            }
            $context .= "\n";
        }
        
        // Interests
        $interests = $this->persona->get_interests();
        if (!empty($interests)) {
            $context .= "INTERESTS (What I explore):\n";
            foreach ($interests as $interest) {
                $context .= "- $interest\n";
            }
            $context .= "\n";
        }
        
        // Target audience
        $audience = $this->persona->get_target_audience();
        if (!empty($audience)) {
            $context .= "MY TARGET AUDIENCE:\n$audience\n\n";
        }
        
        // Voice & tone
        $context .= "MY VOICE: " . $this->persona->get_voice_style() . "\n";
        $context .= "MY TONE: " . $this->persona->get_tone() . "\n\n";
        
        // Topics to avoid
        $avoid = $this->persona->get_avoid_topics();
        if (!empty($avoid)) {
            $context .= "TOPICS I NEVER WRITE ABOUT:\n";
            foreach ($avoid as $topic) {
                $context .= "- $topic\n";
            }
            $context .= "\n";
        }
        
        // Location/Locality - CRITICAL for geo-targeting
        $localities = $this->persona->get_localities();
        if (!empty($localities)) {
            $context .= "MY SERVICE AREA (ONLY write about these locations):\n";
            foreach ($localities as $locality) {
                $loc_data = $this->persona->get_localization_data($locality);
                if ($loc_data && is_array($loc_data)) {
                    $city = isset($loc_data['city']) ? $loc_data['city'] : '';
                    $country = isset($loc_data['country']) ? $loc_data['country'] : '';
                    if ($city || $country) {
                        $context .= "- " . trim($city . ", " . $country, ", ") . "\n";
                    }
                }
            }
            $context .= "\n⚠️ IMPORTANT: All content must be relevant to these locations ONLY. Do NOT mention other cities, countries, or regions.\n\n";
        }
        
        return $context;
    }
    
    /**
     * Build the strategic thinking prompt for supporting topic generation
     * This encodes the logic for choosing good supporting topics
     */
    private function build_strategic_prompt($pillar_keyword, $count, $persona_context) {
        $prompt = "You are about to think strategically as a real expert.\n\n";
        $prompt .= "=== YOUR IDENTITY ===\n";
        $prompt .= $persona_context;
        
        $prompt .= "=== YOUR TASK ===\n";
        $prompt .= "You have decided to write a comprehensive PILLAR article about:\n";
        $prompt .= "\"$pillar_keyword\"\n\n";
        
        $prompt .= "Now, think like the expert you are. What $count SUPPORTING ARTICLES would complete this topic?\n\n";
        
        $prompt .= "=== RULES FOR SUPPORTING TOPICS ===\n\n";
        
        $prompt .= "1. DEPTH RULE: Each supporting article must answer a question that the pillar cannot cover in full depth.\n";
        $prompt .= "   - The pillar gives a broad overview\n";
        $prompt .= "   - Supporting articles go DEEP into ONE subtopic\n";
        $prompt .= "   - Ask: 'Would adding this to the pillar make it too long or too detailed?' If yes, it's a good supporting topic.\n\n";
        
        $prompt .= "2. JOURNEY RULE: Each supporting article must help the reader progress toward the main goal.\n";
        $prompt .= "   - Explain sub-processes\n";
        $prompt .= "   - Compare solutions\n";
        $prompt .= "   - Answer specific technical questions\n";
        $prompt .= "   - Cover use cases\n";
        $prompt .= "   - Show how to troubleshoot problems\n";
        $prompt .= "   - Provide local or niche variations\n\n";
        
        $prompt .= "3. SEARCH DEMAND RULE: Each topic must be something people ACTUALLY search for.\n";
        $prompt .= "   - Think about what questions your audience asks\n";
        $prompt .= "   - Consider pain points and problems\n";
        $prompt .= "   - Include long-tail keyword variations\n\n";
        
        $prompt .= "4. ONE STEP DEEPER RULE: Each supporting article zooms into a SINGLE ANGLE.\n";
        $prompt .= "   - Example: If pillar is 'Indoor Air Quality', supporting might be:\n";
        $prompt .= "     • Mold growth signs\n";
        $prompt .= "     • VOC sources in apartments\n";
        $prompt .= "     • Air purifier types\n";
        $prompt .= "     • Humidity control methods\n\n";
        
        $prompt .= "5. LINKABLE RULE: Each must link logically back to the pillar.\n";
        $prompt .= "   - Ask: 'How does this help the reader understand the whole topic?'\n\n";
        
        $prompt .= "6. NO OVERLAP RULE: Supporting topics must NOT overlap with each other.\n";
        $prompt .= "   - Each handles ONE subtopic only\n";
        $prompt .= "   - The cluster should look like a WHEEL, not a tangled web\n\n";
        
        $prompt .= "=== THINK LIKE THIS ===\n";
        $prompt .= "\"If my pillar article is the FULL COURSE, what CHAPTERS need their own detailed lessons?\"\n\n";
        
        $prompt .= "=== YOUR OUTPUT ===\n";
        $prompt .= "Generate exactly $count supporting article topics.\n";
        $prompt .= "Format each as a search-friendly title.\n";
        $prompt .= "One topic per line.\n";
        $prompt .= "No numbers, bullets, or explanations - just the topic titles.\n\n";
        
        $prompt .= "⚠️ CRITICAL TITLE RULES:\n";
        $prompt .= "1. Each title must be UNDER 60 CHARACTERS (for SEO)\n";
        $prompt .= "2. NO asterisks (*), NO markdown formatting, NO special characters\n";
        $prompt .= "3. NO meta-language like 'supporting article', 'pillar', 'topic', 'cluster'\n";
        $prompt .= "4. Each title must be a real, searchable topic that a reader would look for\n";
        $prompt .= "5. Plain text only - these are ACTUAL ARTICLE TITLES for publication\n\n";
        
        $prompt .= "WRONG EXAMPLES (never do this):\n";
        $prompt .= "- '7 Supporting Article Topics for Kitchen Countertops' ❌\n";
        $prompt .= "- 'Essential Cluster Topics About Wood Types' ❌\n";
        $prompt .= "- 'Pillar Content Ideas for Carpentry' ❌\n\n";
        
        $prompt .= "CORRECT EXAMPLES:\n";
        $prompt .= "- 'How to Seal Wood Veneer Kitchen Countertops' ✓\n";
        $prompt .= "- 'Best Wood Types for Kitchen Countertops' ✓\n";
        $prompt .= "- 'Wood Veneer vs Solid Wood Countertops Comparison' ✓\n\n";
        
        $prompt .= "Topics should feel like they came from an expert who TRULY understands the subject and their audience's needs.\n";
        
        return $prompt;
    }
    
    /**
     * Parse the AI response into structured topic array
     */
    private function parse_strategic_topics($response, $pillar_keyword) {
        $topics = array();
        $lines = array_filter(array_map('trim', explode("\n", $response)));
        $used_types = array();
        
        // Meta-language patterns to reject (these create "articles about articles")
        $meta_patterns = array(
            '/supporting\s*(article|topic|content)/i',
            '/pillar\s*(article|topic|content)/i',
            '/cluster\s*(article|topic|content)/i',
            '/content\s*strateg/i',
            '/article\s*topics?\s*(for|about)/i',
            '/topic\s*ideas?\s*(for|about)/i',
            '/essential\s*(supporting|cluster|pillar)/i',
            '/\d+\s*(essential|best|top)\s*(supporting|cluster|article)\s*topics/i',
        );
        
        foreach ($lines as $line) {
            // Clean up the line
            $line = preg_replace('/^\d+[\.\)\-]\s*/', '', $line); // Remove numbering
            $line = preg_replace('/^[\*\-\•]\s*/', '', $line); // Remove bullets
            $line = preg_replace('/\*+/', '', $line); // Remove ALL asterisks (markdown bold/italic)
            $line = preg_replace('/_{2,}/', '', $line); // Remove markdown underscores
            $line = preg_replace('/`+/', '', $line); // Remove backticks
            $line = preg_replace('/#+\s*/', '', $line); // Remove markdown headers
            $line = trim($line, '"\'');
            $line = trim($line);
            
            // Skip empty or too short
            if (empty($line) || strlen($line) < 10) continue;
            
            // Truncate to 60 characters for SEO (RankMath guideline)
            if (strlen($line) > 60) {
                // Try to break at a word boundary
                $truncated = substr($line, 0, 60);
                $last_space = strrpos($truncated, ' ');
                if ($last_space > 40) {
                    $line = substr($truncated, 0, $last_space);
                } else {
                    $line = $truncated;
                }
                AIAB_Logger::debug("📏 Title truncated to 60 chars: " . $line);
            }
            
            // Skip if it's just a repeat of the pillar
            similar_text(strtolower($pillar_keyword), strtolower($line), $similarity);
            if ($similarity > 70) continue;
            
            // ⚠️ CRITICAL: Skip meta-topics that would create "articles about articles"
            $is_meta = false;
            foreach ($meta_patterns as $pattern) {
                if (preg_match($pattern, $line)) {
                    AIAB_Logger::warning("🚫 Rejected meta-topic: " . $line);
                    $is_meta = true;
                    break;
                }
            }
            if ($is_meta) continue;
            
            // Determine article type
            $article_type = $this->determine_article_type($line);
            
            // Ensure variety
            $type_count = array_count_values($used_types);
            if (isset($type_count[$article_type]) && $type_count[$article_type] >= 2) {
                $article_type = $this->get_varied_article_type($used_types);
            }
            $used_types[] = $article_type;
            
            $topics[] = array(
                'keyword' => $line,
                'source' => 'ai_strategic',
                'score' => 80, // High score for AI strategic topics
                'article_type' => $article_type,
                'search_volume' => 'medium'
            );
        }
        
        AIAB_Logger::info("🧠 AI Strategic Thinking generated " . count($topics) . " topics");
        
        return $topics;
    }
    
    /**
     * Fallback: Get additional topics from search when AI doesn't generate enough
     */
    private function get_search_based_topics($pillar_keyword, $needed, $existing_keywords) {
        $topics = array();
        $all_topics = array();
        
        // Use configured search provider
        switch ($this->search_provider) {
            case 'google_cse':
                if (!empty($this->google_cse_key) && !empty($this->google_cse_cx)) {
                    $all_topics = $this->research_with_google_cse($pillar_keyword);
                }
                break;
            case 'serpapi':
                if (!empty($this->serpapi_key)) {
                    $all_topics = $this->research_with_serpapi($pillar_keyword);
                }
                break;
            case 'valueserp':
                if (!empty($this->valueserp_key)) {
                    $all_topics = $this->research_with_valueserp($pillar_keyword);
                }
                break;
            default:
                $all_topics = $this->research_with_free($pillar_keyword);
        }
        
        // Filter and score
        $all_topics = $this->filter_written_topics($all_topics);
        $scored = $this->score_topics($all_topics);
        usort($scored, function($a, $b) { return $b['score'] - $a['score']; });
        
        // Filter to only niche-relevant topics
        $scored = $this->filter_by_niche_relevance($scored);
        
        // Filter out mismatched locations
        $scored = $this->filter_by_location_relevance($scored);
        
        $used_types = array();
        foreach ($scored as $topic) {
            if (count($topics) >= $needed) break;
            
            // Skip if already exists
            if (in_array(strtolower($topic['keyword']), $existing_keywords)) continue;
            
            // Skip if too similar to pillar
            similar_text(strtolower($pillar_keyword), strtolower($topic['keyword']), $similarity);
            if ($similarity > 70) continue;
            
            $article_type = $this->determine_article_type($topic['keyword']);
            $type_count = array_count_values($used_types);
            if (isset($type_count[$article_type]) && $type_count[$article_type] >= 2) {
                $article_type = $this->get_varied_article_type($used_types);
            }
            $used_types[] = $article_type;
            
            $topic['article_type'] = $article_type;
            $topics[] = $topic;
        }
        
        return $topics;
    }
    
    /**
     * Filter topics to only include those relevant to persona's niche
     * A topic must contain at least ONE keyword from expertise, passions, or interests
     */
    private function filter_by_niche_relevance($topics) {
        // Get the CORE expertise areas (these are required)
        $expertise = $this->persona->get_expertise_areas();
        $passions = $this->persona->get_passions();
        
        // STEP 1: Find the PRIMARY niche keyword (most frequently appearing word)
        // This word MUST appear in all topics
        $word_frequency = array();
        $all_text = implode(' ', array_merge($expertise, $passions));
        $all_words = preg_split('/\s+/', strtolower($all_text));
        
        foreach ($all_words as $word) {
            $word = preg_replace('/[^a-z]/', '', $word); // Remove non-letters
            if (strlen($word) >= 4 && !in_array($word, $this->get_generic_words())) {
                $word_frequency[$word] = isset($word_frequency[$word]) ? $word_frequency[$word] + 1 : 1;
            }
        }
        
        // Sort by frequency and get top word(s)
        arsort($word_frequency);
        $primary_keywords = array();
        $max_freq = 0;
        foreach ($word_frequency as $word => $freq) {
            if ($max_freq == 0) $max_freq = $freq;
            // Include words that appear at least 50% as often as the top word
            if ($freq >= $max_freq * 0.5 && count($primary_keywords) < 3) {
                $primary_keywords[] = $word;
            }
        }
        
        AIAB_Logger::info("🎯 Primary niche keyword(s) detected: " . implode(', ', $primary_keywords));
        
        // STEP 2: Build pattern for niche words (for scoring)
        $core_words = array();
        foreach ($expertise as $exp) {
            $words = preg_split('/\s+/', strtolower(trim($exp)));
            foreach ($words as $word) {
                if (strlen($word) >= 4 && !in_array($word, $this->get_generic_words())) {
                    $core_words[] = preg_quote($word, '/');
                }
            }
        }
        foreach ($passions as $passion) {
            $words = preg_split('/\s+/', strtolower(trim($passion)));
            foreach ($words as $word) {
                if (strlen($word) >= 4 && !in_array($word, $this->get_generic_words())) {
                    $core_words[] = preg_quote($word, '/');
                }
            }
        }
        
        if (empty($core_words)) {
            AIAB_Logger::warning("No core niche words found - no filtering applied");
            return $topics;
        }
        
        $core_words = array_unique($core_words);
        $pattern = '/\b(' . implode('|', $core_words) . ')\b/i';
        
        // STEP 3: Filter - REQUIRE primary keyword to appear
        $filtered = array();
        foreach ($topics as $topic) {
            $keyword_lower = strtolower($topic['keyword']);
            
            // HARD REQUIREMENT: Primary keyword must appear
            $has_primary = false;
            foreach ($primary_keywords as $pk) {
                if (strpos($keyword_lower, $pk) !== false) {
                    $has_primary = true;
                    break;
                }
            }
            
            if (!$has_primary) {
                AIAB_Logger::debug("❌ Missing primary keyword: " . $topic['keyword']);
                continue; // Skip topics without primary niche word
            }
            
            // Count how many core niche words appear in the topic
            preg_match_all($pattern, $keyword_lower, $matches);
            $match_count = count($matches[0]);
            
            // Require at least ONE core niche word (in addition to primary)
            if ($match_count >= 1) {
                // Boost score based on niche relevance
                $topic['score'] = isset($topic['score']) ? $topic['score'] + ($match_count * 10) : $match_count * 10;
                $filtered[] = $topic;
                AIAB_Logger::debug("✅ Niche match ($match_count): " . $topic['keyword']);
            } else {
                AIAB_Logger::debug("❌ Filtered out off-topic: " . $topic['keyword']);
            }
        }
        
        AIAB_Logger::info("🎯 Niche relevance filter: " . count($topics) . " → " . count($filtered) . " topics");
        
        // If filter is too aggressive (filtered out everything), use AI verification
        if (empty($filtered) && !empty($topics)) {
            AIAB_Logger::warning("All topics filtered - using AI to verify top candidates");
            return $this->ai_verify_topic_relevance(array_slice($topics, 0, 10));
        }
        
        return $filtered;
    }
    
    /**
     * Get list of generic words to exclude from niche matching
     */
    private function get_generic_words() {
        return array(
            'the', 'and', 'for', 'with', 'that', 'this', 'from', 'have', 'been',
            'what', 'when', 'where', 'which', 'while', 'about', 'after', 'before',
            'between', 'through', 'during', 'under', 'over', 'into', 'your', 'their',
            'more', 'most', 'other', 'some', 'such', 'only', 'same', 'than', 'very',
            'just', 'also', 'well', 'even', 'back', 'much', 'then', 'here', 'there',
            'when', 'make', 'like', 'time', 'year', 'good', 'best', 'know', 'take',
            'come', 'could', 'would', 'should', 'being', 'using', 'used', 'does',
            'complete', 'guide', 'ultimate', 'essential', 'effective', 'proven',
            'tips', 'ways', 'steps', 'methods', 'techniques', 'strategies',
            'services', 'solutions', 'systems', 'professional', 'experts',
            'home', 'house', 'building', 'property', 'area', 'local', 'near',
            'cost', 'price', 'free', 'cheap', 'expensive', 'affordable',
            'control', 'prevention', 'protection', 'treatment', 'management',
            'inspection', 'detection', 'removal', 'repair', 'maintenance',
            'problem', 'issue', 'damage', 'signs', 'symptoms', 'causes',
            'dubai', 'uae', 'abu', 'dhabi'
        );
    }
    
    /**
     * 🌍 LOCATION FILTER: Remove topics containing foreign/mismatched locations
     * Auto-detects the persona's target country and filters out foreign locations
     */
    private function filter_by_location_relevance($topics) {
        // Only apply if persona has localities set
        if (!$this->persona->has_localities()) {
            return $topics;
        }
        
        $localities = $this->persona->get_localities();
        $primary_locality = strtolower($localities[0] ?? '');
        
        // Extract country from locality string (usually last part after comma)
        $detected_country = $this->detect_country_from_locality($primary_locality);
        
        if (empty($detected_country)) {
            AIAB_Logger::debug("🌍 Location filter: Could not detect country from '{$primary_locality}'");
            return $topics;
        }
        
        AIAB_Logger::info("🌍 Location filter: Detected target country '{$detected_country}'");
        
        // Get all foreign location keywords to filter out
        $foreign_locations = $this->get_foreign_location_keywords($detected_country);
        
        if (empty($foreign_locations)) {
            return $topics;
        }
        
        // Build regex pattern
        $pattern = '/\b(' . implode('|', array_map(function($loc) {
            return preg_quote($loc, '/');
        }, $foreign_locations)) . ')\b/i';
        
        $filtered = array();
        $rejected_count = 0;
        
        foreach ($topics as $topic) {
            $keyword_lower = strtolower($topic['keyword']);
            
            // Check if topic contains any foreign location
            if (preg_match($pattern, $keyword_lower, $matches)) {
                AIAB_Logger::debug("❌ Location mismatch filtered: \"{$topic['keyword']}\" (contains: {$matches[0]})");
                $rejected_count++;
            } else {
                $filtered[] = $topic;
            }
        }
        
        if ($rejected_count > 0) {
            AIAB_Logger::info("🌍 Location filter removed {$rejected_count} topics with foreign locations");
        }
        
        return $filtered;
    }
    
    /**
     * Detect country from locality string
     * "Dubai, United Arab Emirates" → "uae"
     * "London, UK" → "uk"
     * "New York, USA" → "usa"
     */
    private function detect_country_from_locality($locality) {
        $locality = strtolower($locality);
        
        // Country detection patterns (check most specific first)
        $country_patterns = array(
            'uae' => array('united arab emirates', 'uae', 'emirates', 'dubai', 'abu dhabi', 'sharjah', 'ajman', 'fujairah', 'ras al khaimah', 'umm al quwain'),
            'uk' => array('united kingdom', 'england', 'scotland', 'wales', 'northern ireland', ', uk', 'britain', 'london', 'manchester', 'birmingham', 'liverpool', 'leeds', 'glasgow', 'edinburgh'),
            'usa' => array('united states', 'usa', 'america', ', us', 'new york', 'los angeles', 'chicago', 'houston', 'phoenix', 'philadelphia', 'san antonio', 'san diego', 'dallas', 'san jose'),
            'australia' => array('australia', 'sydney', 'melbourne', 'brisbane', 'perth', 'adelaide', 'gold coast', 'canberra', 'newcastle', 'wollongong'),
            'canada' => array('canada', 'toronto', 'vancouver', 'montreal', 'calgary', 'edmonton', 'ottawa', 'winnipeg', 'quebec'),
            'india' => array('india', 'mumbai', 'delhi', 'bangalore', 'hyderabad', 'chennai', 'kolkata', 'pune', 'ahmedabad', 'jaipur'),
            'singapore' => array('singapore'),
            'malaysia' => array('malaysia', 'kuala lumpur', 'penang', 'johor', 'malacca'),
            'saudi' => array('saudi arabia', 'saudi', 'riyadh', 'jeddah', 'mecca', 'medina', 'dammam', 'ksa'),
            'qatar' => array('qatar', 'doha'),
            'germany' => array('germany', 'deutschland', 'berlin', 'munich', 'frankfurt', 'hamburg', 'cologne'),
            'france' => array('france', 'paris', 'marseille', 'lyon', 'toulouse', 'nice'),
            'spain' => array('spain', 'españa', 'madrid', 'barcelona', 'valencia', 'seville'),
            'italy' => array('italy', 'italia', 'rome', 'milan', 'naples', 'turin', 'florence'),
            'netherlands' => array('netherlands', 'holland', 'amsterdam', 'rotterdam', 'the hague'),
            'ireland' => array('ireland', 'dublin', 'cork', 'galway', 'limerick'),
            'south_africa' => array('south africa', 'johannesburg', 'cape town', 'durban', 'pretoria'),
            'new_zealand' => array('new zealand', 'auckland', 'wellington', 'christchurch'),
            'philippines' => array('philippines', 'manila', 'cebu', 'davao'),
            'pakistan' => array('pakistan', 'karachi', 'lahore', 'islamabad', 'rawalpindi'),
            'egypt' => array('egypt', 'cairo', 'alexandria', 'giza'),
            'nigeria' => array('nigeria', 'lagos', 'abuja', 'kano'),
            'kenya' => array('kenya', 'nairobi', 'mombasa'),
            'japan' => array('japan', 'tokyo', 'osaka', 'kyoto', 'yokohama'),
            'china' => array('china', 'beijing', 'shanghai', 'guangzhou', 'shenzhen'),
            'hong_kong' => array('hong kong'),
            'thailand' => array('thailand', 'bangkok', 'phuket', 'chiang mai'),
            'indonesia' => array('indonesia', 'jakarta', 'bali', 'surabaya'),
            'vietnam' => array('vietnam', 'ho chi minh', 'hanoi', 'da nang'),
            'korea' => array('south korea', 'korea', 'seoul', 'busan'),
            'mexico' => array('mexico', 'mexico city', 'guadalajara', 'monterrey'),
            'brazil' => array('brazil', 'brasil', 'são paulo', 'sao paulo', 'rio de janeiro', 'brasilia'),
        );
        
        foreach ($country_patterns as $country => $patterns) {
            foreach ($patterns as $pattern) {
                if (strpos($locality, $pattern) !== false) {
                    return $country;
                }
            }
        }
        
        return null;
    }
    
    /**
     * Get location keywords to filter OUT based on target country
     */
    private function get_foreign_location_keywords($target_country) {
        // US locations (states, common patterns)
        $us_locations = array(
            // States
            'alabama', 'alaska', 'arizona', 'arkansas', 'california', 'colorado', 
            'connecticut', 'delaware', 'florida', 'georgia', 'hawaii', 'idaho', 
            'illinois', 'indiana', 'iowa', 'kansas', 'kentucky', 'louisiana', 
            'maine', 'maryland', 'massachusetts', 'michigan', 'minnesota', 
            'mississippi', 'missouri', 'montana', 'nebraska', 'nevada', 
            'new hampshire', 'new jersey', 'new mexico', 'new york', 'north carolina', 
            'north dakota', 'ohio', 'oklahoma', 'oregon', 'pennsylvania', 
            'rhode island', 'south carolina', 'south dakota', 'tennessee', 'texas', 
            'utah', 'vermont', 'virginia', 'washington state', 'west virginia', 
            'wisconsin', 'wyoming',
            // US patterns
            'county', 'township'
        );
        
        // UK locations
        $uk_locations = array(
            'england', 'scotland', 'wales', 'northern ireland',
            'london', 'manchester', 'birmingham', 'liverpool', 'leeds', 
            'sheffield', 'bristol', 'newcastle', 'nottingham', 'southampton',
            'portsmouth', 'leicester', 'coventry', 'bradford', 'cardiff',
            'belfast', 'edinburgh', 'glasgow', 'aberdeen'
        );
        
        // UAE locations
        $uae_locations = array(
            'dubai', 'abu dhabi', 'sharjah', 'ajman', 'fujairah', 
            'ras al khaimah', 'umm al quwain', 'uae', 'emirates'
        );
        
        // Australia locations
        $au_locations = array(
            'sydney', 'melbourne', 'brisbane', 'perth', 'adelaide',
            'gold coast', 'canberra', 'newcastle', 'wollongong', 'hobart',
            'new south wales', 'queensland', 'victoria', 'western australia',
            'south australia', 'tasmania', 'northern territory'
        );
        
        // Canada locations
        $ca_locations = array(
            'toronto', 'vancouver', 'montreal', 'calgary', 'edmonton',
            'ottawa', 'winnipeg', 'quebec', 'ontario', 'british columbia',
            'alberta', 'manitoba', 'saskatchewan', 'nova scotia'
        );
        
        // India locations
        $in_locations = array(
            'mumbai', 'delhi', 'bangalore', 'bengaluru', 'hyderabad', 
            'chennai', 'kolkata', 'pune', 'ahmedabad', 'jaipur',
            'maharashtra', 'karnataka', 'tamil nadu', 'kerala', 'gujarat'
        );
        
        // Saudi locations
        $sa_locations = array(
            'riyadh', 'jeddah', 'mecca', 'medina', 'dammam', 'khobar',
            'saudi', 'ksa'
        );
        
        // Build foreign locations based on target country
        $foreign = array();
        
        // Add all locations EXCEPT the target country's
        $all_regions = array(
            'usa' => $us_locations,
            'uk' => $uk_locations,
            'uae' => $uae_locations,
            'australia' => $au_locations,
            'canada' => $ca_locations,
            'india' => $in_locations,
            'saudi' => $sa_locations,
        );
        
        foreach ($all_regions as $region => $locations) {
            if ($region !== $target_country) {
                $foreign = array_merge($foreign, $locations);
            }
        }
        
        // For countries not in our detailed list, use generic filtering
        if (!isset($all_regions[$target_country])) {
            // Filter out US, UK, and other major English-speaking regions
            $foreign = array_merge($us_locations, $uk_locations, $au_locations, $ca_locations);
            AIAB_Logger::debug("🌍 Using generic foreign location filter for '{$target_country}'");
        }
        
        return array_unique($foreign);
    }
    
    /**
     * Use AI to verify if topics match persona's niche
     */
    private function ai_verify_topic_relevance($topics) {
        // Skip AI verification in budget mode
        if (get_option('aiab_budget_mode', 0) || get_option('aiab_skip_research_ai', 0)) {
            AIAB_Logger::info("💰 Budget mode: Skipping AI topic verification (saves ~500 tokens)");
            return $topics; // Return unverified topics
        }
        
        $expertise = implode(', ', $this->persona->get_expertise_areas());
        $passions = implode(', ', $this->persona->get_passions());
        
        $topic_list = array();
        foreach ($topics as $t) {
            $topic_list[] = $t['keyword'];
        }
        
        $prompt = "You are evaluating topics for a content writer with this expertise: {$expertise}\n";
        $prompt .= "Their passions include: {$passions}\n\n";
        $prompt .= "Which of these topics would this expert ACTUALLY write about? Only include topics directly related to their core expertise.\n\n";
        $prompt .= "Topics:\n" . implode("\n", $topic_list) . "\n\n";
        $prompt .= "Reply with ONLY the relevant topics, one per line. If none are relevant, reply with 'NONE'.";
        
        try {
            $ai_writer = new AIAB_AI_Writer($this->persona);
            $response = $ai_writer->generate_completion($prompt, 500);
            
            if (empty($response) || strtoupper(trim($response)) === 'NONE') {
                AIAB_Logger::info("🤖 AI verification: No relevant topics found");
                return array();
            }
            
            $approved = array_map('trim', explode("\n", $response));
            $approved = array_filter($approved);
            
            $verified = array();
            foreach ($topics as $topic) {
                foreach ($approved as $app) {
                    if (stripos($topic['keyword'], $app) !== false || stripos($app, $topic['keyword']) !== false) {
                        $verified[] = $topic;
                        AIAB_Logger::debug("✅ AI verified: " . $topic['keyword']);
                        break;
                    }
                }
            }
            
            AIAB_Logger::info("🤖 AI verification: " . count($topics) . " → " . count($verified) . " topics");
            return $verified;
            
        } catch (Exception $e) {
            AIAB_Logger::error("AI verification failed: " . $e->getMessage());
            return array();
        }
    }
    
    /**
     * Determine article type for a topic
     */
    public function determine_article_type($keyword) {
        $keyword_lower = strtolower($keyword);
        
        $patterns = array(
            'how_to' => '/^how (to|do|can|should)/i',
            'listicle' => '/(top \d+|best \d+|\d+ (best|top|ways|tips|reasons|signs|benefits))/i',
            'comparison' => '/(vs\.?|versus|compared to|difference between| or )/i',
            'question' => '/^(what is|why is|when is|where is|which is|who is|is it|are there|can you|does it|do i|should i)/i',
            'cost' => '/(cost|price|pricing|how much|fee|rate|expensive|cheap|afford)/i',
            'review' => '/(review|rating|rated|recommend)/i',
            'buyers_guide' => '/(buyer|buying|purchase|shopping|choose|choosing|select)/i',
            'local_seo' => '/(in dubai|in uae|in abu dhabi|near me|local|region|area|city|middle east)/i',
            'glossary' => '/^(what is|what are|define|definition|meaning of|what does.*mean)/i',
            'case_study' => '/(case study|example|real.?world|success story|we helped|we removed|we fixed)/i',
            'seasonal' => '/(summer|winter|spring|fall|autumn|seasonal|ramadan|holiday|weather)/i',
            'problem_solution' => '/(problem|issue|fix|solve|trouble|error|won\'t|doesn\'t|not working|why does|why is my|causes of)/i',
            'guide' => '/(guide|tutorial|complete|ultimate|comprehensive|everything|step.?by.?step)/i',
        );
        
        foreach ($patterns as $type => $pattern) {
            if (preg_match($pattern, $keyword_lower)) {
                return $type;
            }
        }
        
        return $this->get_random_article_type();
    }
    
    /**
     * Get a random article type with weighted distribution
     */
    private function get_random_article_type() {
        $weighted_types = array(
            'how_to' => 20,
            'listicle' => 18,
            'problem_solution' => 15,
            'question' => 15,
            'cost' => 12,
            'comparison' => 10,
            'buyers_guide' => 10,
            'local_seo' => 10,
            'review' => 8,
            'glossary' => 8,
            'guide' => 8,
            'case_study' => 6,
            'seasonal' => 5,
            'informational' => 5,
        );
        
        $total = array_sum($weighted_types);
        $random = mt_rand(1, $total);
        $cumulative = 0;
        
        foreach ($weighted_types as $type => $weight) {
            $cumulative += $weight;
            if ($random <= $cumulative) {
                return $type;
            }
        }
        
        return 'informational';
    }
    
    /**
     * Get varied article type to ensure diversity
     */
    public function get_varied_article_type($existing_types = array()) {
        $all_types = array(
            'how_to', 'listicle', 'problem_solution', 'question', 'cost',
            'comparison', 'buyers_guide', 'local_seo', 'review', 
            'glossary', 'guide', 'case_study', 'seasonal', 'informational'
        );
        
        $type_counts = array_count_values($existing_types);
        
        // Find unused types first
        $available = array();
        foreach ($all_types as $type) {
            $count = isset($type_counts[$type]) ? $type_counts[$type] : 0;
            if ($count === 0) {
                $available[] = $type;
            }
        }
        
        if (empty($available)) {
            $available = $all_types;
        }
        
        return $available[array_rand($available)];
    }
}
