<?php
/**
 * Thought Sphere
 * Represents a cluster of interconnected articles around a pillar topic
 */

if (!defined('ABSPATH')) {
    exit;
}

class AIAB_Thought_Sphere {
    
    private $id;
    private $persona_id;
    private $pillar_keyword;
    private $pillar_title;
    private $search_volume;
    private $difficulty_score;
    private $status;
    private $phase;
    private $total_articles;
    private $published_articles;
    private $sphere_data;
    private $research_data;
    private $error_log;
    private $started_at;
    private $completed_at;
    private $created_at;
    
    private $articles = array();
    
    // Status constants
    const STATUS_PLANNED = 'planned';
    const STATUS_RESEARCHING = 'researching';
    const STATUS_WRITING = 'writing';
    const STATUS_PUBLISHING = 'publishing';
    const STATUS_COMPLETED = 'completed';
    const STATUS_FAILED = 'failed';
    
    // Phase constants
    const PHASE_RESEARCH = 'research';
    const PHASE_PLANNING = 'planning';
    const PHASE_WRITING_PILLAR = 'writing_pillar';
    const PHASE_WRITING_CLUSTERS = 'writing_clusters';
    const PHASE_LINKING = 'linking';
    const PHASE_PUBLISHING = 'publishing';
    const PHASE_COMPLETE = 'complete';
    
    public function __construct($data = null) {
        if ($data) {
            $this->hydrate($data);
        }
    }
    
    private function hydrate($data) {
        $this->id = isset($data->id) ? intval($data->id) : null;
        $this->persona_id = isset($data->persona_id) ? intval($data->persona_id) : null;
        $this->pillar_keyword = isset($data->pillar_keyword) ? $data->pillar_keyword : '';
        $this->pillar_title = isset($data->pillar_title) ? $data->pillar_title : '';
        $this->search_volume = isset($data->search_volume) ? intval($data->search_volume) : 0;
        $this->difficulty_score = isset($data->difficulty_score) ? intval($data->difficulty_score) : 0;
        $this->status = isset($data->status) ? $data->status : self::STATUS_PLANNED;
        $this->phase = isset($data->phase) ? $data->phase : self::PHASE_RESEARCH;
        $this->total_articles = isset($data->total_articles) ? intval($data->total_articles) : 7;
        $this->published_articles = isset($data->published_articles) ? intval($data->published_articles) : 0;
        $this->sphere_data = isset($data->sphere_data) ? json_decode($data->sphere_data, true) : array();
        $this->research_data = isset($data->research_data) ? json_decode($data->research_data, true) : array();
        $this->error_log = isset($data->error_log) ? $data->error_log : '';
        $this->started_at = isset($data->started_at) ? $data->started_at : null;
        $this->completed_at = isset($data->completed_at) ? $data->completed_at : null;
        $this->created_at = isset($data->created_at) ? $data->created_at : null;
    }
    
    /**
     * Get by ID
     */
    public static function get($id) {
        global $wpdb;
        $table = AIAB_Database::get_table('thought_spheres');
        
        $data = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM $table WHERE id = %d",
            $id
        ));
        
        return $data ? new self($data) : null;
    }
    
    /**
     * Get incomplete sphere for a persona
     */
    public static function get_incomplete_for_persona($persona_id) {
        global $wpdb;
        $table = AIAB_Database::get_table('thought_spheres');
        
        $data = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM $table WHERE persona_id = %d AND status NOT IN ('completed', 'failed') ORDER BY created_at ASC LIMIT 1",
            $persona_id
        ));
        
        return $data ? new self($data) : null;
    }
    
    /**
     * Create new sphere
     */
    public static function create($persona_id, $pillar_keyword, $total_articles = 7) {
        global $wpdb;
        $table = AIAB_Database::get_table('thought_spheres');
        $now = current_time('mysql');
        
        // Sanitize and truncate pillar_keyword for SEO (max 80 chars - titles should be 50-60)
        $pillar_keyword = trim($pillar_keyword);
        $max_length = 80; // SEO-friendly length for titles
        
        if (strlen($pillar_keyword) > $max_length) {
            $original_length = strlen($pillar_keyword);
            
            // First, strip everything after a colon (AI often adds subtitles)
            if (strpos($pillar_keyword, ':') !== false) {
                $pillar_keyword = trim(explode(':', $pillar_keyword)[0]);
            }
            
            // If still too long, truncate at word boundary
            if (strlen($pillar_keyword) > $max_length) {
                $truncated = substr($pillar_keyword, 0, $max_length);
                $last_space = strrpos($truncated, ' ');
                
                if ($last_space !== false && $last_space > ($max_length * 0.6)) {
                    $pillar_keyword = substr($truncated, 0, $last_space);
                } else {
                    $pillar_keyword = $truncated;
                }
            }
            
            AIAB_Logger::warning("Pillar keyword truncated for SEO: {$original_length} → " . strlen($pillar_keyword) . " chars", array(
                'truncated_to' => $pillar_keyword
            ));
        }
        
        // Apply proper Title Case for SEO (RankMath prefers title case)
        // But preserve certain words as lowercase (articles, prepositions, conjunctions)
        $pillar_keyword = self::to_title_case($pillar_keyword);
        
        $result = $wpdb->insert($table, array(
            'persona_id' => $persona_id,
            'pillar_keyword' => $pillar_keyword,
            'total_articles' => $total_articles,
            'status' => self::STATUS_PLANNED,
            'phase' => self::PHASE_RESEARCH,
            'started_at' => $now,
            'created_at' => $now,
            'updated_at' => $now
        ));
        
        // Check for database errors
        if ($result === false) {
            AIAB_Logger::log("Database insert failed for sphere", 'error', array(
                'error' => $wpdb->last_error,
                'table' => $table,
                'pillar_keyword' => $pillar_keyword
            ));
            return null;
        }
        
        $id = $wpdb->insert_id;
        
        if (!$id) {
            AIAB_Logger::log("Sphere insert returned no ID", 'error', array(
                'table' => $table,
                'pillar_keyword' => $pillar_keyword,
                'wpdb_error' => $wpdb->last_error
            ));
            return null;
        }
        
        // Record topic in history
        AIAB_Database::record_topic($persona_id, $pillar_keyword, $id);
        
        AIAB_Logger::log("Created new thought sphere: $pillar_keyword", 'info', array(
            'sphere_id' => $id,
            'persona_id' => $persona_id
        ));
        
        return self::get($id);
    }
    
    /**
     * Save sphere
     */
    public function save() {
        global $wpdb;
        $table = AIAB_Database::get_table('thought_spheres');
        
        $data = array(
            'persona_id' => $this->persona_id,
            'pillar_keyword' => $this->pillar_keyword,
            'pillar_title' => $this->pillar_title,
            'search_volume' => $this->search_volume,
            'difficulty_score' => $this->difficulty_score,
            'status' => $this->status,
            'phase' => $this->phase,
            'total_articles' => $this->total_articles,
            'published_articles' => $this->published_articles,
            'sphere_data' => json_encode($this->sphere_data),
            'research_data' => json_encode($this->research_data),
            'error_log' => $this->error_log,
            'completed_at' => $this->completed_at,
            'updated_at' => current_time('mysql')
        );
        
        if ($this->id) {
            $wpdb->update($table, $data, array('id' => $this->id));
        }
        
        return $this->id;
    }
    
    /**
     * Update status
     */
    public function update_status($status) {
        $this->status = $status;
        
        if ($status === self::STATUS_COMPLETED) {
            $this->completed_at = current_time('mysql');
        }
        
        $this->save();
        
        AIAB_Logger::log("Sphere status updated: $status", 'info', array(
            'sphere_id' => $this->id
        ));
    }
    
    /**
     * Update phase
     */
    public function update_phase($phase) {
        $this->phase = $phase;
        $this->save();
        
        AIAB_Logger::log("Sphere phase updated: $phase", 'info', array(
            'sphere_id' => $this->id
        ));
    }
    
    /**
     * Add error to log
     */
    public function log_error($error) {
        $this->error_log .= "[" . current_time('mysql') . "] $error\n";
        $this->save();
    }
    
    /**
     * Plan the article structure
     */
    public function plan_structure(AIAB_Persona $persona, $supporting_topics) {
        // DUPLICATE PROTECTION: Check if already planned
        if (!empty($this->sphere_data['clusters']) && count($this->sphere_data['clusters']) > 0) {
            AIAB_Logger::warning("⚠️ Sphere already has planned structure - skipping duplicate planning", array(
                'sphere_id' => $this->id,
                'existing_clusters' => count($this->sphere_data['clusters'])
            ));
            return $this->sphere_data; // Return existing structure
        }
        
        $this->update_phase(self::PHASE_PLANNING);
        
        $structure = array(
            'pillar' => array(
                'keyword' => $this->pillar_keyword,
                'type' => 'pillar',
                'target_words' => get_option('aiab_pillar_word_count', 2500),
                'links_to' => array(), // Will link to all cluster articles
            ),
            'clusters' => array()
        );
        
        $research_engine = new AIAB_Research_Engine($persona);
        $used_types = array(); // Track types used in this sphere for variety
        
        foreach ($supporting_topics as $index => $topic) {
            // First try to determine type from keyword pattern
            $article_type = $research_engine->determine_article_type($topic['keyword']);
            
            // If type was randomly assigned (not pattern-matched) and already used too much,
            // get a varied type to ensure diversity
            if (in_array($article_type, $used_types) && count(array_keys($used_types, $article_type)) >= 2) {
                $article_type = $research_engine->get_varied_article_type($used_types);
            }
            
            $used_types[] = $article_type;
            
            $structure['clusters'][] = array(
                'keyword' => $topic['keyword'],
                'type' => $article_type,
                'source' => $topic['source'],
                'target_words' => get_option('aiab_cluster_word_count', 1500),
                'position' => $index + 1,
                'links_to' => array() // Will be populated during linking phase
            );
            
            // Record in history
            AIAB_Database::record_topic($this->persona_id, $topic['keyword'], $this->id);
        }
        
        $this->sphere_data = $structure;
        $this->save();
        
        // Log the variety of types used
        $type_summary = array_count_values($used_types);
        AIAB_Logger::log("Sphere structure planned", 'info', array(
            'sphere_id' => $this->id,
            'pillar' => $this->pillar_keyword,
            'cluster_count' => count($structure['clusters']),
            'article_types' => $type_summary
        ));
        
        return $structure;
    }
    
    /**
     * Get planned articles
     */
    public function get_planned_articles() {
        $articles = array();
        
        if (!empty($this->sphere_data['pillar'])) {
            $articles[] = array_merge($this->sphere_data['pillar'], array('is_pillar' => true));
        }
        
        if (!empty($this->sphere_data['clusters'])) {
            foreach ($this->sphere_data['clusters'] as $cluster) {
                $articles[] = array_merge($cluster, array('is_pillar' => false));
            }
        }
        
        return $articles;
    }
    
    /**
     * Create article records in database
     */
    public function create_article_records() {
        global $wpdb;
        $table = AIAB_Database::get_table('articles');
        
        // DUPLICATE PROTECTION: Check if articles already exist for this sphere
        $existing_count = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM $table WHERE sphere_id = %d",
            $this->id
        ));
        
        if ($existing_count > 0) {
            AIAB_Logger::warning("⚠️ Articles already exist for sphere - skipping duplicate creation", array(
                'sphere_id' => $this->id,
                'existing_articles' => $existing_count
            ));
            return; // Don't create duplicates
        }
        
        $planned = $this->get_planned_articles();
        
        foreach ($planned as $article) {
            $wpdb->insert($table, array(
                'sphere_id' => $this->id,
                'persona_id' => $this->persona_id,
                'article_type' => $article['type'],
                'is_pillar' => $article['is_pillar'] ? 1 : 0,
                'keyword' => $article['keyword'],
                'status' => 'planned'
            ));
        }
        
        AIAB_Logger::log("Created " . count($planned) . " article records", 'info', array(
            'sphere_id' => $this->id
        ));
    }
    
    /**
     * Get articles for this sphere
     */
    public function get_articles($status = null) {
        global $wpdb;
        $table = AIAB_Database::get_table('articles');
        
        $sql = $wpdb->prepare("SELECT * FROM $table WHERE sphere_id = %d", $this->id);
        
        if ($status) {
            $sql .= $wpdb->prepare(" AND status = %s", $status);
        }
        
        $sql .= " ORDER BY is_pillar DESC, id ASC";
        
        return $wpdb->get_results($sql);
    }
    
    /**
     * Get pillar article
     */
    public function get_pillar_article() {
        global $wpdb;
        $table = AIAB_Database::get_table('articles');
        
        return $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM $table WHERE sphere_id = %d AND is_pillar = 1",
            $this->id
        ));
    }
    
    /**
     * Get cluster articles
     */
    public function get_cluster_articles() {
        global $wpdb;
        $table = AIAB_Database::get_table('articles');
        
        return $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM $table WHERE sphere_id = %d AND is_pillar = 0 ORDER BY id ASC",
            $this->id
        ));
    }
    
    /**
     * Check if all articles are written (with sufficient content)
     */
    public function all_articles_written() {
        global $wpdb;
        $table = AIAB_Database::get_table('articles');
        
        // Count articles that still need writing:
        // - status is 'planned' or 'writing' or 'failed' (retryable)
        // - OR status is 'written' but word_count < 300 (failed silently - lowered from 500)
        // 
        // NOTE: We do NOT count these statuses as "pending" - they're permanently failed:
        // - auth_failed (API key invalid)
        // - budget_failed (out of credits)
        // - max_retries (failed 3+ times)
        // The sphere should continue even if some articles couldn't be written
        $pending = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM $table WHERE sphere_id = %d AND (
                status IN ('planned', 'writing', 'failed')
                OR (status = 'written' AND word_count < 300)
            )",
            $this->id
        ));
        
        // Also log what's blocking us if there are pending articles
        if ($pending > 0) {
            $breakdown = $wpdb->get_results($wpdb->prepare(
                "SELECT status, COUNT(*) as count FROM $table 
                 WHERE sphere_id = %d 
                 GROUP BY status",
                $this->id
            ));
            AIAB_Logger::debug("Articles not yet complete", array(
                'sphere_id' => $this->id,
                'pending_count' => $pending,
                'status_breakdown' => $breakdown
            ));
        }
        
        return $pending == 0;
    }
    
    /**
     * Check if all articles are published (or permanently failed)
     * Permanently failed articles don't block completion
     */
    public function all_articles_published() {
        global $wpdb;
        $table = AIAB_Database::get_table('articles');
        
        // Count articles that are NOT in a "done" state
        // Done states: published, linked, auth_failed, budget_failed, max_retries
        // These are either successfully published OR permanently failed (can't do more)
        $not_done = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM $table 
             WHERE sphere_id = %d 
             AND status NOT IN ('published', 'linked', 'auth_failed', 'budget_failed', 'max_retries')",
            $this->id
        ));
        
        return $not_done == 0;
    }
    
    /**
     * Get list of unpublished articles
     */
    public function get_unpublished_articles() {
        global $wpdb;
        $table = AIAB_Database::get_table('articles');
        
        return $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM $table WHERE sphere_id = %d AND status != 'published'",
            $this->id
        ));
    }
    
    /**
     * Increment published count
     */
    public function increment_published() {
        $this->published_articles++;
        $this->save();
    }
    
    /**
     * Generate internal link map
     */
    public function generate_link_map() {
        $articles = $this->get_articles();
        $pillar = null;
        $clusters = array();
        
        foreach ($articles as $article) {
            if ($article->is_pillar) {
                $pillar = $article;
            } else {
                $clusters[] = $article;
            }
        }
        
        $link_map = array(
            'pillar_id' => $pillar ? $pillar->id : null,
            'pillar_wp_id' => $pillar ? $pillar->wp_post_id : null,
            'cluster_links' => array()
        );
        
        // Each cluster links to: pillar + 2 adjacent clusters (circular)
        $cluster_count = count($clusters);
        
        foreach ($clusters as $index => $cluster) {
            $links = array();
            
            // Link to pillar
            if ($pillar && $pillar->wp_post_id) {
                $links[] = array(
                    'article_id' => $pillar->id,
                    'wp_post_id' => $pillar->wp_post_id,
                    'title' => $pillar->title,
                    'type' => 'pillar'
                );
            }
            
            // Link to previous cluster (circular)
            $prev_index = ($index - 1 + $cluster_count) % $cluster_count;
            if ($cluster_count > 1 && $clusters[$prev_index]->wp_post_id) {
                $links[] = array(
                    'article_id' => $clusters[$prev_index]->id,
                    'wp_post_id' => $clusters[$prev_index]->wp_post_id,
                    'title' => $clusters[$prev_index]->title,
                    'type' => 'cluster'
                );
            }
            
            // Link to next cluster (circular)
            $next_index = ($index + 1) % $cluster_count;
            if ($cluster_count > 2 && $clusters[$next_index]->wp_post_id) {
                $links[] = array(
                    'article_id' => $clusters[$next_index]->id,
                    'wp_post_id' => $clusters[$next_index]->wp_post_id,
                    'title' => $clusters[$next_index]->title,
                    'type' => 'cluster'
                );
            }
            
            $link_map['cluster_links'][$cluster->id] = $links;
        }
        
        // Pillar links to all clusters
        $link_map['pillar_links'] = array();
        foreach ($clusters as $cluster) {
            if ($cluster->wp_post_id) {
                $link_map['pillar_links'][] = array(
                    'article_id' => $cluster->id,
                    'wp_post_id' => $cluster->wp_post_id,
                    'title' => $cluster->title,
                    'type' => 'cluster'
                );
            }
        }
        
        $this->sphere_data['link_map'] = $link_map;
        $this->save();
        
        return $link_map;
    }
    
    // Getters
    public function get_id() { return $this->id; }
    public function get_persona_id() { return $this->persona_id; }
    public function get_pillar_keyword() { return $this->pillar_keyword; }
    public function get_pillar_title() { return $this->pillar_title; }
    public function get_status() { return $this->status; }
    public function get_phase() { return $this->phase; }
    public function get_total_articles() { return $this->total_articles; }
    public function get_published_articles() { return $this->published_articles; }
    public function get_sphere_data() { return $this->sphere_data; }
    public function get_research_data() { return $this->research_data; }
    
    // Setters
    public function set_pillar_title($title) { $this->pillar_title = $title; }
    public function set_research_data($data) { $this->research_data = $data; }
    
    /**
     * Convert string to proper Title Case for SEO
     * Keeps articles/prepositions/conjunctions lowercase (except at start)
     */
    private static function to_title_case($string) {
        // Words to keep lowercase (unless first word)
        $lowercase_words = array(
            'a', 'an', 'the', 'and', 'but', 'or', 'nor', 'for', 'yet', 'so',
            'at', 'by', 'in', 'of', 'on', 'to', 'up', 'as', 'is', 'if',
            'vs', 'via', 'per', 'with', 'from', 'into', 'onto', 'upon'
        );
        
        // First, lowercase everything then split into words
        $string = strtolower($string);
        $words = explode(' ', $string);
        
        foreach ($words as $i => $word) {
            // Always capitalize first word, or any word not in the lowercase list
            if ($i === 0 || !in_array($word, $lowercase_words)) {
                $words[$i] = ucfirst($word);
            }
        }
        
        return implode(' ', $words);
    }
}
